/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;

import org.apache.maven.plugin.logging.Log;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.Mock;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.DebianPackageConstants;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.impl.DebianPackageBuilderMock;
import net.sourceforge.javadpkg.io.impl.FileModeImpl;
import net.sourceforge.javadpkg.io.impl.FileOwnerImpl;
import net.sourceforge.javadpkg.plugin.AbstractDpkgPluginTest;
import net.sourceforge.javadpkg.plugin.io.FileSystemNode;
import net.sourceforge.javadpkg.plugin.io.impl.FileSystemNodeImpl;
import net.sourceforge.javadpkg.plugin.io.impl.PathImpl;
import net.sourceforge.javadpkg.replace.Replacements;


/**
 * <p>
 * Performs some tests on the {@link DataFileSystemNodeVisitor} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 20.05.2016 by Gerrit Hohl
 */
public class DataFileSystemNodeVisitorTest extends AbstractDpkgPluginTest implements DebianPackageConstants {
	
	
	/** The default encoding. */
	private static final Charset		DEFAULT_ENCODING		= Charset.forName("UTF-8");

	/** The default mode for a directory. */
	private static final FileModeImpl	DEFAULT_DIRECTORY_MODE	= new FileModeImpl(DIRECTORY_MODE);
	/** The default mode for a script file. */
	private static final FileModeImpl	DEFAULT_SCRIPT_MODE		= new FileModeImpl(SCRIPT_MODE);
	/** The default mode for a regular file. */
	private static final FileModeImpl	DEFAULT_FILE_MODE		= new FileModeImpl(FILE_MODE);
	/** The default owner. */
	private static final FileOwnerImpl	DEFAULT_OWNER			= new FileOwnerImpl(ROOT_GROUP_ID, ROOT_GROUP_NAME,
			ROOT_USER_ID, ROOT_USER_NAME);

	/** The base directory of the test resource. */
	private static final String			BASE_DIR				= "src/test/resources/net/sourceforge/javadpkg/plugin";
	/** The absolute path of the base directory of the test resource. */
	private static final String			BASE_PATH				= new File(BASE_DIR).getAbsolutePath();
	
	
	/** The log. */
	@Mock
	private Log							log;
	/** The replacements. */
	@Mock
	private Replacements				replacements;


	/**
	 * <p>
	 * Performs some tests on the {@link DataFileSystemNodeVisitor} class.
	 * </p>
	 */
	@Test
	public void test() {
		DataFileSystemNodeVisitor visitor;
		DebianPackageBuilderMock builder;
		Context context;
		FileSystemNode<FileSystemNodeInfo> root, parentNode, baseNode;


		builder = new DebianPackageBuilderMock();
		context = new ContextImpl();


		// --- Parameters ---
		try {
			visitor = new DataFileSystemNodeVisitor(null, builder, DEFAULT_ENCODING, this.replacements, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			visitor = new DataFileSystemNodeVisitor(this.log, null, DEFAULT_ENCODING, this.replacements, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			visitor = new DataFileSystemNodeVisitor(this.log, builder, null, this.replacements, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			visitor = new DataFileSystemNodeVisitor(this.log, builder, DEFAULT_ENCODING, null, context);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}
		try {
			visitor = new DataFileSystemNodeVisitor(this.log, builder, DEFAULT_ENCODING, this.replacements, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		}


		// --- Successful visiting ---
		root = this.createDefaultFileSystem();
		parentNode = root.createDirectories(PathImpl.parsePath("/usr/local"));
		baseNode = parentNode.createChildDirectory("test", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		baseNode.createChildFile("program", DEFAULT_OWNER, DEFAULT_SCRIPT_MODE,
				this.createInfo(BASE_DIR + "/program", 35, false, null));
		baseNode.createChildSymLink("other_program", PathImpl.parsePath("/usr/local/test/program"), DEFAULT_OWNER,
				DEFAULT_SCRIPT_MODE, null);
		parentNode = baseNode.createChildDirectory("lib", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode.createChildFile("library1.lib", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/lib/library1.lib", 18, false, null));
		parentNode.createChildFile("library2.lib", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/lib/library2.lib", 24, false, null));
		baseNode.createChildSymLink("lib64", PathImpl.parsePath("/usr/local/test/lib/"), DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE,
				null);
		parentNode = baseNode.createChildDirectory("ext", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode.createChildFile("ext1.dat", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/ext/ext1.dat", 8, false, null));
		parentNode = baseNode.createChildDirectory("res", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode = parentNode.createChildDirectory("images", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode.createChildFile("image1.img", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/custom_images/image1.img", 6, false, null));
		parentNode.createChildFile("image2.img", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/custom_images/image2.img", 6, false, null));
		parentNode.createChildFile("image3.img", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/custom_images/image3.img", 6, false, null));
		parentNode = baseNode.getNode(PathImpl.parsePath("res"));
		parentNode.createChildFile("res1", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/res/res1", 4, false, null));
		parentNode.createChildFile("res2", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/res/res2", 4, false, null));
		parentNode = parentNode.createChildDirectory("text", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode.createChildFile("text1.txt", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/res/text/text1.txt", 9, false, null));
		parentNode.createChildFile("text2.txt", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/res/text/text2.txt", 9, false, null));
		parentNode.createChildFile("text3.txt", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/res/text/text3.txt", 9, false, null));
		baseNode.createChildDirectory("template", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode = baseNode.createChildDirectory("conf", DEFAULT_OWNER, DEFAULT_DIRECTORY_MODE, null);
		parentNode.createChildFile("conf", DEFAULT_OWNER, DEFAULT_FILE_MODE,
				this.createInfo(BASE_DIR + "/conf/conf.default", 28, true, "UTF-8"));

		visitor = new DataFileSystemNodeVisitor(this.log, builder, DEFAULT_ENCODING, this.replacements, context);
		try {
			root.walkNodeTree(visitor);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't walk tree: " + e.getMessage());
			return;
		}
		Assert.assertEquals(166, visitor.getSize());
		Assert.assertEquals(13, visitor.getFiles());
		Assert.assertEquals(11, visitor.getDirectories());
		Assert.assertEquals(2, visitor.getSymbolicLinks());
		Assert.assertTrue(visitor.isProcess());
		Assert.assertEquals(
				"drwxr-xr-x     0 root         0 root           null /usr                                    \n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local                              \n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test                         \n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/conf                    \n"
						+ "-rw-r--r--     0 root         0 root             26 /usr/local/test/conf/conf                (Source: |"
						+ BASE_PATH + "\\conf\\conf.default|)\n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/ext                     \n"
						+ "-rw-r--r--     0 root         0 root              8 /usr/local/test/ext/ext1.dat             (Source: |"
						+ BASE_PATH + "\\ext\\ext1.dat|)\n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/lib                     \n"
						+ "-rw-r--r--     0 root         0 root             18 /usr/local/test/lib/library1.lib         (Source: |"
						+ BASE_PATH + "\\lib\\library1.lib|)\n"
						+ "-rw-r--r--     0 root         0 root             24 /usr/local/test/lib/library2.lib         (Source: |"
						+ BASE_PATH + "\\lib\\library2.lib|)\n"
						+ "lrwxr-xr-x     0 root         0 root           null /usr/local/test/lib64                    (Target: |/usr/local/test/lib|)\n"
						+ "lrwxr-xr-x     0 root         0 root           null /usr/local/test/other_program            (Target: |/usr/local/test/program|)\n"
						+ "-rwxr-xr-x     0 root         0 root             35 /usr/local/test/program                  (Source: |"
						+ BASE_PATH + "\\program|)\n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/res                     \n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/res/images              \n"
						+ "-rw-r--r--     0 root         0 root              6 /usr/local/test/res/images/image1.img    (Source: |"
						+ BASE_PATH + "\\custom_images\\image1.img|)\n"
						+ "-rw-r--r--     0 root         0 root              6 /usr/local/test/res/images/image2.img    (Source: |"
						+ BASE_PATH + "\\custom_images\\image2.img|)\n"
						+ "-rw-r--r--     0 root         0 root              6 /usr/local/test/res/images/image3.img    (Source: |"
						+ BASE_PATH + "\\custom_images\\image3.img|)\n"
						+ "-rw-r--r--     0 root         0 root              4 /usr/local/test/res/res1                 (Source: |"
						+ BASE_PATH + "\\res\\res1|)\n"
						+ "-rw-r--r--     0 root         0 root              4 /usr/local/test/res/res2                 (Source: |"
						+ BASE_PATH + "\\res\\res2|)\n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/res/text                \n"
						+ "-rw-r--r--     0 root         0 root              9 /usr/local/test/res/text/text1.txt       (Source: |"
						+ BASE_PATH + "\\res\\text\\text1.txt|)\n"
						+ "-rw-r--r--     0 root         0 root              9 /usr/local/test/res/text/text2.txt       (Source: |"
						+ BASE_PATH + "\\res\\text\\text2.txt|)\n"
						+ "-rw-r--r--     0 root         0 root              9 /usr/local/test/res/text/text3.txt       (Source: |"
						+ BASE_PATH + "\\res\\text\\text3.txt|)\n"
						+ "drwxr-xr-x     0 root         0 root           null /usr/local/test/template                ",
				builder.getDataNodesAsString());
	}


	/**
	 * <p>
	 * Creates a default file system based on the FHS.
	 * </p>
	 * <p>
	 * For more information see:
	 * <a href="https://en.wikipedia.org/wiki/Filesystem_Hierarchy_Standard">
	 * Filesystem Hierarchy Standard</a>.
	 * </p>
	 * <p>
	 * A few directories are not created as they should be used by any Debian
	 * package (like <code>/dev</code>, <code>/home</code>, etc.).
	 * </p>
	 *
	 * @return The root.
	 */
	private FileSystemNode<FileSystemNodeInfo> createDefaultFileSystem() {
		FileSystemNode<FileSystemNodeInfo> root;


		root = new FileSystemNodeImpl<>("");
		root.createDirectories(PathImpl.parsePath("/bin"));
		root.createDirectories(PathImpl.parsePath("/boot"));
		root.createDirectories(PathImpl.parsePath("/etc"));
		root.createDirectories(PathImpl.parsePath("/lib"));
		root.createDirectories(PathImpl.parsePath("/opt"));
		root.createDirectories(PathImpl.parsePath("/run"));
		root.createDirectories(PathImpl.parsePath("/sbin"));
		root.createDirectories(PathImpl.parsePath("/srv"));
		root.createDirectories(PathImpl.parsePath("/usr"));
		root.createDirectories(PathImpl.parsePath("/usr/bin"));
		root.createDirectories(PathImpl.parsePath("/usr/include"));
		root.createDirectories(PathImpl.parsePath("/usr/lib"));
		root.createDirectories(PathImpl.parsePath("/usr/local"));
		root.createDirectories(PathImpl.parsePath("/usr/sbin"));
		root.createDirectories(PathImpl.parsePath("/usr/share"));
		root.createDirectories(PathImpl.parsePath("/usr/src"));
		root.createDirectories(PathImpl.parsePath("/usr/X11R6"));
		root.createDirectories(PathImpl.parsePath("/var"));
		root.createDirectories(PathImpl.parsePath("/var/cache"));
		root.createDirectories(PathImpl.parsePath("/var/lib"));
		root.createDirectories(PathImpl.parsePath("/var/lock"));
		root.createDirectories(PathImpl.parsePath("/var/log"));
		root.createDirectories(PathImpl.parsePath("/var/mail"));
		root.createDirectories(PathImpl.parsePath("/var/opt"));
		root.createDirectories(PathImpl.parsePath("/var/run"));
		root.createDirectories(PathImpl.parsePath("/var/spool"));
		root.createDirectories(PathImpl.parsePath("/var/spool/mail"));
		return root;
	}


	/**
	 * <p>
	 * Creates the additional information for a node.
	 * </p>
	 *
	 * @param path
	 *            The path of the source file.
	 * @param length
	 *            The length of the source file.
	 * @param process
	 *            The flag if the file should be processed.
	 * @param encoding
	 *            The encoding for the processing (optional).
	 * @return The additional information.
	 */
	private FileSystemNodeInfo createInfo(String path, long length, boolean process, String encoding) {
		FileSystemNodeInfo result;


		result = new FileSystemNodeInfo(new FileInfoMock(path, length), process, encoding);
		return result;
	}


}
