/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.logging.Log;
import org.junit.Assert;
import org.junit.Test;
import org.mockito.Mock;

import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.plugin.AbstractDpkgPluginTest;
import net.sourceforge.javadpkg.plugin.cfg.DataEntry;


/**
 * <p>
 * Performs some tests on the {@link DataEntryNodeTransformer} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 19.05.2016 by Gerrit Hohl
 */
public class DataEntryNodeTransformerTest extends AbstractDpkgPluginTest {
	
	
	/** The base directory. */
	private static final String	BASEDIR	= "src/test/resources/net/sourceforge/javadpkg/plugin/";
	
	
	/** The log. */
	@Mock
	private Log					log;
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link DataEntryNodeTransformer} class.
	 * </p>
	 */
	@Test
	public void test() {
		DataEntryNodeTransformer transformer;
		List<DataEntry> entries;
		List<DataEntryNode> nodes;
		DataEntryNode node;


		transformer = new DataEntryNodeTransformer();
		
		
		// --- Parameters ---
		entries = new ArrayList<>();
		try {
			transformer.transform(null, entries);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			transformer.transform(this.log, null);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}


		// --- No entries ---
		entries = new ArrayList<>();
		try {
			nodes = transformer.transform(this.log, entries);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(nodes);
		Assert.assertEquals(0, nodes.size());


		// --- Entries ---
		entries = new ArrayList<>();
		entries.add(new DataEntry(null, "/usr/local/test/", null, null, null, null, null, null, false, false, null));
		entries.add(new DataEntry(BASEDIR + "program", "/usr/local/test/", null, null, null, null, null, Integer.valueOf(00755),
				false, false, null));
		entries.add(new DataEntry(null, "/usr/local/test/other_program", "/usr/local/test/program", null, null, null, null,
				Integer.valueOf(00755), false, false, null));
		entries.add(new DataEntry(null, "/usr/local/test/lib/", null, null, null, null, null, null, false, false, null));
		entries.add(new DataEntry(BASEDIR + "lib/*.lib", "/usr/local/test/lib/", null, null, null, null, null, null, false,
				false, null));
		entries.add(new DataEntry(null, "/usr/local/test/lib64/", "/usr/local/test/lib/", null, null, null, null, null, false,
				false, null));
		entries.add(new DataEntry(BASEDIR + "ext/*.dat", "/usr/local/test/lib/", null, null, null, null, null, null, false,
				false, null));
		entries.add(
				new DataEntry(BASEDIR + "res/", "/usr/local/test/res/", null, null, null, null, null, null, true, false, null));
		entries.add(new DataEntry(BASEDIR + "custom_images/*", "/usr/local/test/res/images/", null, null, null, null, null,
				null, false, false, null));
		entries.add(new DataEntry(BASEDIR + "template/", "/usr/local/test/template/", null, null, null, null, null, null, true,
				false, null));
		entries.add(new DataEntry(null, "/usr/local/test/conf/", null, null, null, null, null, null, false, false, null));
		entries.add(new DataEntry(BASEDIR + "conf/conf.default", "/usr/local/test/conf/conf", null, null, null, null, null,
				null, false, true, "UTF-8"));
		try {
			nodes = transformer.transform(this.log, entries);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(nodes);
		Assert.assertEquals(23, nodes.size());
		
		node = nodes.get(0);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("test", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(1);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "program").getPath(), node.getSource().getPath());
		Assert.assertEquals("program", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertEquals(Integer.valueOf(00755), node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(2);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("other_program", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertEquals("/usr/local/test/program", node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertEquals(Integer.valueOf(00755), node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(3);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("lib", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(4);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "lib/library1.lib").getPath(), node.getSource().getPath());
		Assert.assertEquals("library1.lib", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/lib", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(5);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "lib/library2.lib").getPath(), node.getSource().getPath());
		Assert.assertEquals("library2.lib", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/lib", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(6);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("lib64", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertEquals("/usr/local/test/lib/", node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(7);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "ext/ext1.dat").getPath(), node.getSource().getPath());
		Assert.assertEquals("ext1.dat", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/lib", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(8);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("res", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(9);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("images", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(10);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "res/images/image1.img").getPath(), node.getSource().getPath());
		Assert.assertEquals("image1.img", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/images", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(11);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "res/res1").getPath(), node.getSource().getPath());
		Assert.assertEquals("res1", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(12);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "res/res2").getPath(), node.getSource().getPath());
		Assert.assertEquals("res2", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(13);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("text", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(14);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "res/text/text1.txt").getPath(), node.getSource().getPath());
		Assert.assertEquals("text1.txt", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/text", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(15);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "res/text/text2.txt").getPath(), node.getSource().getPath());
		Assert.assertEquals("text2.txt", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/text", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(16);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "res/text/text3.txt").getPath(), node.getSource().getPath());
		Assert.assertEquals("text3.txt", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/text", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(17);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "custom_images/image1.img").getPath(), node.getSource().getPath());
		Assert.assertEquals("image1.img", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/images", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(18);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "custom_images/image2.img").getPath(), node.getSource().getPath());
		Assert.assertEquals("image2.img", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/images", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(19);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "custom_images/image3.img").getPath(), node.getSource().getPath());
		Assert.assertEquals("image3.img", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/res/images", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(20);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("template", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(21);
		Assert.assertNotNull(node);
		Assert.assertNull(node.getSource());
		Assert.assertEquals("conf", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertFalse(node.isProcess());
		Assert.assertNull(node.getEncoding());
		
		node = nodes.get(22);
		Assert.assertNotNull(node);
		Assert.assertNotNull(node.getSource());
		Assert.assertEquals(new File(BASEDIR + "conf/conf.default").getPath(), node.getSource().getPath());
		Assert.assertEquals("conf", node.getName());
		Assert.assertNotNull(node.getParent());
		Assert.assertEquals("/usr/local/test/conf", node.getParent().getAbsolutePath());
		Assert.assertNull(node.getSymLink());
		Assert.assertNull(node.getGroupId());
		Assert.assertNull(node.getGroupName());
		Assert.assertNull(node.getUserId());
		Assert.assertNull(node.getUserName());
		Assert.assertNull(node.getMode());
		Assert.assertTrue(node.isProcess());
		Assert.assertEquals("UTF-8", node.getEncoding());
	}


}
