/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin;

import java.util.List;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.runner.RunWith;
import org.mockito.runners.MockitoJUnitRunner;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.Architecture;
import net.sourceforge.javadpkg.control.ArchitectureParser;
import net.sourceforge.javadpkg.control.PackageName;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.impl.ArchitectureParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;


/**
 * <p>
 * The abstract superclass of all tests.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
@RunWith(MockitoJUnitRunner.class)
public abstract class AbstractDpkgPluginTest {
	
	
	/**
	 * <p>
	 * Performs some preparations before all tests are performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		// --- Do nothing. ---
	}


	/**
	 * <p>
	 * Performs some clean-up after all tests have been performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@AfterClass
	public static void tearDownAfterClass() throws Exception {
		// --- Do nothing. ---
	}


	/**
	 * <p>
	 * Performs some preparations before each test is performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@Before
	public void setUp() throws Exception {
		// --- Do nothing. ---
	}


	/**
	 * <p>
	 * Performs some clean-up after each test is performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@After
	public void tearDown() throws Exception {
		// --- Do nothing. ---
	}


	/**
	 * <p>
	 * Creates a package name.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The package name.
	 */
	protected static PackageName createPackageName(String name) {
		PackageName result;
		PackageNameParser parser;
		Context context;
		List<Warning> warnings;
		
		
		Assert.assertNotNull(name);
		parser = new PackageNameParserImpl();
		context = new ContextImpl();
		try {
			result = parser.parsePackageName(name, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create package name |" + name + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertEquals("Found some warnings while creating package name |" + name + "|: " + warnings.size(), 0,
				warnings.size());
		return result;
	}
	
	
	/**
	 * <p>
	 * Creates a package version.
	 * </p>
	 *
	 * @param version
	 *            The version.
	 * @return The package version.
	 */
	protected static PackageVersion createPackageVersion(String version) {
		PackageVersion result;
		PackageVersionParser parser;
		Context context;
		List<Warning> warnings;
		
		
		Assert.assertNotNull(version);
		parser = new PackageVersionParserImpl();
		context = new ContextImpl();
		try {
			result = parser.parsePackageVersion(version, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create package version |" + version + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertEquals("Found some warnings while creating package version |" + version + "|: " + warnings.size(), 0,
				warnings.size());
		return result;
	}
	
	
	/**
	 * <p>
	 * Creates an architecture.
	 * </p>
	 * 
	 * @param architecture
	 *            The architecture.
	 * @return The architecture.
	 */
	protected static Architecture createArchitecture(String architecture) {
		Architecture result;
		ArchitectureParser parser;
		Context context;
		List<Warning> warnings;
		
		
		Assert.assertNotNull(architecture);
		parser = new ArchitectureParserImpl();
		context = new ContextImpl();
		try {
			result = parser.parseArchitecture(architecture, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create architecture |" + architecture + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertEquals("Found some warnings while creating architecture |" + architecture + "|: " + warnings.size(), 0,
				warnings.size());
		return result;
	}


}
