/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.reflect;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * <p>
 * A {@link PropertyReflection} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 */
public class PropertyReflectionImpl implements PropertyReflection {
	
	
	/**
	 * <p>
	 * Creates a property reflection.
	 * </p>
	 */
	public PropertyReflectionImpl() {
		super();
	}


	@Override
	public Object getValue(Object obj, String path) throws PropertyReflectionException {
		Object value;
		List<String> pathElements;
		
		
		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		pathElements = new ArrayList<>(Arrays.asList(path.split("\\.")));
		try {
			value = this.getValue(obj, pathElements.remove(0), pathElements);
		} catch (PropertyReflectionException e) {
			throw new PropertyReflectionException("Couldn't return property described by the property path |" + path
					+ "| of the object |" + obj + "| (Class: " + obj.getClass().getCanonicalName() + "|): " + e.getMessage(),
					e);
		}
		return value;
	}


	/**
	 * <p>
	 * Returns the value for the specified property.
	 * </p>
	 *
	 * @param obj
	 *            The object.
	 * @param propertyName
	 *            The name of the property.
	 * @param pathElements
	 *            The remaining path elements.
	 * @return The value or <code>null</code>, if the property is not set.
	 * @throws PropertyReflectionException
	 *             If an error occurs.
	 */
	private Object getValue(Object obj, String propertyName, List<String> pathElements) throws PropertyReflectionException {
		Object value;
		Class<?> clazz;
		Field field;
		String getterName;
		Method method;
		
		
		if (propertyName.isEmpty())
			throw new PropertyReflectionException("The path contains an empty element.");
		
		clazz = obj.getClass();
		
		// --- Try to get the property value directly by accessing the field ---
		try {
			field = clazz.getField(propertyName);
			value = field.get(obj);
		} catch (NoSuchFieldException | IllegalAccessException fe) {
			// --- Try to get the property value by calling the getter method ---
			getterName = this.buildGetterName(propertyName);
			try {
				method = clazz.getMethod(getterName);

				// --- The getter method must have a return type ---
				if (Void.class.equals(method.getReturnType()))
					throw new PropertyReflectionException("The getter method |" + getterName + "| of the object |" + obj
							+ "| (Class: " + obj.getClass().getCanonicalName() + "|) has no return type.");
				
				value = method.invoke(obj);
			} catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException me) {
				throw new PropertyReflectionException("The property |" + propertyName + "| of the object |" + obj + "| (Class: "
						+ obj.getClass().getCanonicalName()
						+ "|) couldn't be accessed as field as well as using a getter method.", me);
			}
		}
		// --- Do we have to dig deeper? ---
		if ((value != null) && !pathElements.isEmpty()) {
			value = this.getValue(value, pathElements.remove(0), pathElements);
		}
		return value;
	}
	
	
	/**
	 * <p>
	 * Builds the name of the getter method for the specified property.
	 * </p>
	 *
	 * @param propertyName
	 *            The name of the property.
	 * @return The name of the getter method.
	 */
	private String buildGetterName(String propertyName) {
		StringBuilder sb;


		sb = new StringBuilder();
		sb.append("get");
		sb.append(Character.toUpperCase(propertyName.charAt(0)));
		if (propertyName.length() > 1) {
			sb.append(propertyName.substring(1));
		}
		return sb.toString();
	}
	
	
}
