/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.io.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import net.sourceforge.javadpkg.plugin.io.Path;

/**
 * <p>
 * A {@link Path} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
public class PathImpl implements Path {
	
	
	/** The elements. */
	private List<String> elements;
	
	
	/**
	 * <p>
	 * Creates a path.
	 * </p>
	 */
	public PathImpl() {
		super();
		
		this.elements = new ArrayList<>();
	}


	/**
	 * <p>
	 * Creates a path.
	 * </p>
	 *
	 * @param elements
	 *            The elements.
	 */
	private PathImpl(List<String> elements) {
		super();

		this.elements = new ArrayList<>(elements);
	}


	@Override
	public Path getParentPath() {
		if (this.elements.isEmpty())
			return null;
		
		return (new PathImpl(this.elements.subList(0, this.elements.size() - 1)));
	}
	
	
	@Override
	public Path getChildPath() {
		if (this.elements.size() <= 1)
			return null;
		
		return (new PathImpl(this.elements.subList(1, this.elements.size())));
	}
	
	
	@Override
	public Path createChild(String name) {
		Path newPath;
		List<String> newElements;
		
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		
		newElements = new ArrayList<>(this.elements.size() + 1);
		newElements.addAll(this.elements);
		newElements.add(name);
		newPath = new PathImpl(newElements);
		return newPath;
	}


	@Override
	public String getFirstElement() {
		if (this.elements.isEmpty())
			return null;
		return this.elements.get(0);
	}
	
	
	@Override
	public String getLastElement() {
		if (this.elements.isEmpty())
			return null;
		return this.elements.get(this.elements.size() - 1);
	}


	@Override
	public String getAbsolutePath() {
		StringBuilder sb;
		
		
		sb = new StringBuilder();
		sb.append("/");
		for (String element : this.elements) {
			if (sb.length() > 1) {
				sb.append("/");
			}
			sb.append(element);
		}
		return sb.toString();
	}
	
	
	/**
	 * <p>
	 * Parses the specified path.
	 * </p>
	 *
	 * @param path
	 *            The path as string.
	 * @return The path.
	 * @throws IllegalArgumentException
	 *             If the path is <code>null</code>.
	 */
	public static Path parsePath(String path) {
		PathImpl result;
		String value;
		String[] parts;


		if (path == null)
			throw new IllegalArgumentException("Argument path is null.");
		
		if (path.isEmpty() || "/".equals(path)) {
			result = new PathImpl();
		} else {
			value = path;
			if (value.startsWith("/")) {
				value = value.substring(1);
			}
			if (value.endsWith("/")) {
				value = value.substring(0, value.length() - 1);
			}
			parts = value.split("/", -1);
			result = new PathImpl(Arrays.asList(parts));
		}
		return result;
	}
	
	
	@Override
	public int hashCode() {
		int result = 17;
		
		
		result = (31 * result) + this.getAbsolutePath().hashCode();
		
		return result;
	}
	
	
	@Override
	public boolean equals(Object obj) {
		boolean equal = false;
		Path path;


		if (obj instanceof Path) {
			path = (Path) obj;
			equal = this.getAbsolutePath().equals(path.getAbsolutePath());
		}
		return equal;
	}
	
	
}
