/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.io;

import java.io.IOException;

/**
 * <p>
 * A visitor of file system nodes.
 * </p>
 *
 * @param <A>
 *            The type of the attachment.
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
public interface FileSystemNodeVisitor<A> {
	
	
	/**
	 * <p>
	 * Invoked for a directory node before children of the directory node are
	 * visited.
	 * </p>
	 * <p>
	 * If this method returns {@link FileSystemNodeVisitResult#CONTINUE}, then
	 * children of the directory node are visited. If this method returns
	 * {@link FileSystemNodeVisitResult#SKIP_SUBTREE} or
	 * {@link FileSystemNodeVisitResult#SKIP_SIBLINGS} then children of the
	 * directory node (and any descendants) will not be visited.
	 * </p>
	 *
	 * @param node
	 *            The directory node.
	 * @return The visit result.
	 * @throws IllegalArgumentException
	 *             If the directory node is <code>null</code>.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	FileSystemNodeVisitResult preVisitDirectory(FileSystemNode<A> node) throws IOException;


	/**
	 * <p>
	 * Invoked for a file node in a directory node.
	 * </p>
	 *
	 * @param node
	 *            The file node.
	 * @return The visit result.
	 * @throws IllegalArgumentException
	 *             If the file node is <code>null</code>.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	FileSystemNodeVisitResult visitFile(FileSystemNode<A> node) throws IOException;


	/**
	 * <p>
	 * Invoked for a directory node after children in the directory node, and
	 * all of their descendants, have been visited. This method is also invoked
	 * when iteration of the directory completes prematurely (by a
	 * {@link #visitFile(FileSystemNode)} method returning
	 * {@link FileSystemNodeVisitResult#SKIP_SIBLINGS} directory).
	 * </p>
	 *
	 * @param node
	 *            The directory node.
	 * @return The visit result.
	 * @throws IllegalArgumentException
	 *             If the directory node is <code>null</code>.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	FileSystemNodeVisitResult postVisitDirectory(FileSystemNode<A> node) throws IOException;
	
	
}
