/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.io;

import java.io.IOException;
import java.util.List;

import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;

/**
 * <p>
 * A node representing a file or directory in the file system.
 * </p>
 *
 * @param <A>
 *            The type of the attachment.
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
public interface FileSystemNode<A> {
	
	
	/**
	 * <p>
	 * Returns the parent.
	 * </p>
	 *
	 * @return The parent or <code>null</code>, if this node is the root node.
	 */
	FileSystemNode<A> getParent();


	/**
	 * <p>
	 * Sets the parent.
	 * </p>
	 *
	 * @param parent
	 *            The parent or <code>null</code>, if this node should be the
	 *            root node.
	 * @return The previous parent or <code>null</code>, if this node wasn't a
	 *         child node before.
	 */
	FileSystemNode<A> setParent(FileSystemNode<A> parent);
	
	
	/**
	 * <p>
	 * Returns the name of the node.
	 * </p>
	 *
	 * @return The name.
	 */
	String getName();
	
	
	/**
	 * <p>
	 * Returns the path of the node.
	 * </p>
	 *
	 * @return The path.
	 */
	Path getPath();
	
	
	/**
	 * <p>
	 * Returns the path of the target of the symbolic link.
	 * </p>
	 *
	 * @return The path or <code>null</code>, if the node doesn't represent a
	 *         symbolic link.
	 * @see #isSymLink()
	 */
	Path getSymLinkTarget();


	/**
	 * <p>
	 * Returns the owner.
	 * </p>
	 *
	 * @return The owner.
	 */
	FileOwner getOwner();
	
	
	/**
	 * <p>
	 * Returns the mode.
	 * </p>
	 *
	 * @return The mode.
	 */
	FileMode getMode();
	
	
	/**
	 * <p>
	 * Returns the flag if the node represents a directory.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the node represents a directory,
	 *         <code>false</code> otherwise.
	 */
	boolean isDirectory();
	
	
	/**
	 * <p>
	 * Returns the flag if the node represents a file.
	 * </p>
	 * <p>
	 * Symbolic links are handled as files.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the node represents a file,
	 *         <code>false</code> otherwise.
	 * @see #isSymLink()
	 */
	boolean isFile();
	
	
	/**
	 * <p>
	 * Returns the flag if the node represents a symbolic link.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the node represents a symbolic
	 *         link, <code>false</code> otherwise.
	 * @see #getSymLinkTarget()
	 */
	boolean isSymLink();
	
	
	/**
	 * <p>
	 * Returns the attachment.
	 * </p>
	 *
	 * @return The attachment file or <code>null</code>, if no attachment is
	 *         set.
	 */
	A getAttachment();
	
	
	/**
	 * <p>
	 * Sets the attachment.
	 * </p>
	 *
	 * @param attachment
	 *            The attachment.
	 */
	void setAttachment(A attachment);
	
	
	/**
	 * <p>
	 * Creates a child node within this node representing a directory.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param owner
	 *            The owner.
	 * @param mode
	 *            The mode.
	 * @param attachment
	 *            The attachment (optional).
	 * @return The child node.
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represents a directory or if a child
	 *             node with the specified name already exists.
	 */
	FileSystemNode<A> createChildDirectory(String name, FileOwner owner, FileMode mode, A attachment);
	
	
	/**
	 * <p>
	 * Creates a child node within this node representing a file.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param owner
	 *            The owner.
	 * @param mode
	 *            The mode.
	 * @param attachment
	 *            The attachment (optional).
	 * @return The child node.
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represents a directory or if a child
	 *             node with the specified name already exists.
	 */
	FileSystemNode<A> createChildFile(String name, FileOwner owner, FileMode mode, A attachment);


	/**
	 * <p>
	 * Creates a child node within this node representing a file.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param target
	 *            The path of the target.
	 * @param owner
	 *            The owner.
	 * @param mode
	 *            The mode.
	 * @param attachment
	 *            The attachment (optional).
	 * @return The child node.
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represents a directory or if a child
	 *             node with the specified name already exists.
	 */
	FileSystemNode<A> createChildSymLink(String name, Path target, FileOwner owner, FileMode mode, A attachment);


	/**
	 * <p>
	 * Adds a child node to this node.
	 * </p>
	 *
	 * @param child
	 *            The child node.
	 * @return The child node which was replaced by the specified child node or
	 *         <code>null</code>, if no child node with the same name exists.
	 * @throws IllegalArgumentException
	 *             If the child node is <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represent a directory.
	 */
	FileSystemNode<A> addChild(FileSystemNode<A> child);


	/**
	 * <p>
	 * Returns the child node with the specified name.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The child node or <code>null</code>, if no child node with the
	 *         specified name exists.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represent a directory.
	 */
	FileSystemNode<A> getChild(String name);
	
	
	/**
	 * <p>
	 * Removes the child node with the specified name.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The removed child node or <code>null</code>, if no child node
	 *         with the specified name exists.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represent a directory.
	 */
	FileSystemNode<A> removeChild(String name);
	
	
	/**
	 * <p>
	 * Removes the specified child node.
	 * </p>
	 *
	 * @param child
	 *            The child node.
	 * @return <code>true</code>, if the child node was successfully removed,
	 *         <code>false</code>, if the specified node is not a child node of
	 *         this node.
	 * @throws IllegalArgumentException
	 *             If the child node is <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represent a directory.
	 */
	boolean removeChild(FileSystemNode<A> child);


	/**
	 * <p>
	 * Moves all child nodes of this node to the specified node.
	 * </p>
	 *
	 * @param node
	 *            The node.
	 * @return All child nodes which were replaced in the specified node by the
	 *         child nodes of this node.
	 * @throws IllegalArgumentException
	 *             If the node is <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node or the specified node do not represent a
	 *             directory.
	 */
	List<FileSystemNode<A>> moveChildrenTo(FileSystemNode<A> node);
	
	
	/**
	 * <p>
	 * Creates a node for the directory specified by the path and all
	 * directories on the path there.
	 * </p>
	 * <p>
	 * All created directories are marked as created by a dependency.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The node representing the directory.
	 * @throws IllegalArgumentException
	 *             If the path is <code>null</code>.
	 * @throws IllegalStateException
	 *             If this node does not represent a directory or any of the
	 *             already existing nodes along the path.
	 */
	FileSystemNode<A> createDirectories(Path path);
	
	
	/**
	 * <p>
	 * Returns the node for the specified path.
	 * </p>
	 *
	 * @param path
	 *            The path.
	 * @return The node or <code>null</code>, if the node or any nodes on path
	 *         to it don't exist.
	 * @throws IllegalArgumentException
	 *             If the path is <code>null</code>.
	 * @throws IllegalStateException
	 *             If any node along the path - except the node of the last part
	 *             - does not represent a directory.
	 */
	FileSystemNode<A> getNode(Path path);
	
	
	/**
	 * <p>
	 * Returns the flag if this node was created by a dependency.
	 * </p>
	 * <p>
	 * This is the case if a sub-folder was added, but the parent weren't added
	 * before and created automatically.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if this node was created by a
	 *         dependency, <code>false</code> otherwise.
	 * @see #containsOnlyCreatedByDependency()
	 */
	boolean isCreatedByDependency();


	/**
	 * <p>
	 * Returns the flag if the child nodes (recursively) are all created by
	 * dependency.
	 * </p>
	 * <p>
	 * This method doesn't check the flag of this node.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if all child nodes were created by a
	 *         dependency, <code>false</code> otherwise.
	 * @see #isCreatedByDependency()
	 */
	boolean containsOnlyCreatedByDependency();
	
	
	/**
	 * <p>
	 * Walks the node tree starting at this node.
	 * </p>
	 *
	 * @param visitor
	 *            The visitor to invoke for each node.
	 * @return The visit result of this node.
	 * @throws IllegalArgumentException
	 *             If the visitor is <code>null</code>.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	FileSystemNodeVisitResult walkNodeTree(FileSystemNodeVisitor<A> visitor) throws IOException;
	
	
}
