/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataSwap;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.impl.DataFileSource;
import net.sourceforge.javadpkg.io.impl.DataTempFileSwap;
import net.sourceforge.javadpkg.replace.ReplacementException;
import net.sourceforge.javadpkg.replace.Replacements;
import net.sourceforge.javadpkg.replace.Replacer;
import net.sourceforge.javadpkg.replace.ReplacerImpl;


/**
 * <p>
 * A file-based source which replaces the content of the source by the
 * replacements.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 19.05.2016 by Gerrit Hohl
 */
public class DataReplacementSource implements DataSource {
	
	
	/** The file. */
	private File			file;
	/** The encoding of the file. */
	private Charset			encoding;
	/** The replacements. */
	private Replacements	replacements;
	/** The context. */
	private Context			context;
	
	/** The name. */
	private String			name;
	/** The swap. */
	private DataSwap		swap;
	/** The flag if the source was closed. */
	private boolean			closed;
	/** The replacer. */
	private Replacer		replacer;
	
	
	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 * <p>
	 * The context will be used when the content is processed. Potential
	 * warnings will be available afterwards.
	 * </p>
	 *
	 * @param file
	 *            The file.
	 * @param encoding
	 *            The encoding of the file.
	 * @param replacements
	 *            The replacements.
	 * @param context
	 *            The context.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataReplacementSource(File file, Charset encoding, Replacements replacements, Context context) {
		super();

		if (file == null)
			throw new IllegalArgumentException("Argument file is null.");
		if (encoding == null)
			throw new IllegalArgumentException("Argument encoding is null.");
		if (replacements == null)
			throw new IllegalArgumentException("Argument replacements is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		this.file = file;
		this.encoding = encoding;
		this.replacements = replacements;
		this.context = context;
		
		this.name = file.getAbsolutePath();
		this.swap = null;
		this.closed = false;
		this.replacer = new ReplacerImpl();
	}
	
	
	@Override
	public String getName() {
		return this.name;
	}
	
	
	@Override
	public long getLength() {
		long length = -1;
		
		
		try {
			if (this.swap == null) {
				this.replace();
			}
			length = this.swap.getSource().getLength();
		} catch (IOException e) {
			// --- Do nothing. ---
		}
		return length;
	}
	
	
	@Override
	public boolean isResettable() {
		if (this.swap != null) {
			try {
				return this.swap.getSource().isResettable();
			} catch (IOException e) {
				// --- Do nothing. ---
			}
		}
		return false;
	}
	
	
	@Override
	public void reset() throws IOException {
		if (this.swap != null) {
			this.swap.getSource().reset();
		} else
			throw new IOException("Source |" + this.name + "| doesn't support a reset.");
	}
	
	
	@Override
	public InputStream getInputStream() throws IOException {
		if (this.closed)
			throw new IOException("The source |" + this.name + "| was already closed.");
		
		if (this.swap == null) {
			this.replace();
		}
		return this.swap.getSource().getInputStream();
	}
	
	
	@Override
	public void close() throws IOException {
		try {
			if (this.swap != null) {
				this.swap.close();
			}
		} finally {
			this.swap = null;
			this.closed = true;
		}
	}
	
	
	/**
	 * <p>
	 * Replaces the content of the file.
	 * </p>
	 *
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void replace() throws IOException {
		String line;


		// --- Create the swap ---
		this.swap = new DataTempFileSwap(this.file.getName() + "-swap");

		// --- Read the original file ---
		try {
			try (DataSource source = new DataFileSource(this.file)) {
				try (BufferedReader reader = new BufferedReader(
						new InputStreamReader(source.getInputStream(), this.encoding))) {
					
					try (DataTarget target = this.swap.getTarget()) {
						try (BufferedWriter writer = new BufferedWriter(
								new OutputStreamWriter(target.getOutputStream(), this.encoding))) {
							
							while ((line = reader.readLine()) != null) {
								line = this.replacer.replace(line, this.replacements, this.context);
								
								writer.write(line);
								writer.write("\n");
							}
						}
					}
				}
			}
		} catch (IOException | ReplacementException e) {
			// --- Close swap ---
			try {
				this.swap.close();
			} catch (IOException e2) {
				// --- Do nothing. ---
			} finally {
				this.swap = null;
			}
			
			throw new IOException("Couldn't replace content of file |" + this.file.getAbsolutePath() + "|: " + e.getMessage(),
					e);
		}
	}
	
	
}
