/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.IOException;
import java.nio.charset.Charset;

import org.apache.maven.plugin.logging.Log;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.DebianPackageBuilder;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.FileOwner;
import net.sourceforge.javadpkg.io.impl.DataFileSource;
import net.sourceforge.javadpkg.plugin.io.FileSystemNode;
import net.sourceforge.javadpkg.plugin.io.FileSystemNodeVisitResult;
import net.sourceforge.javadpkg.plugin.io.FileSystemNodeVisitor;
import net.sourceforge.javadpkg.replace.Replacements;


/**
 * <p>
 * A visitor for the node tree based on the data configuration.
 * </p>
 * <p>
 * The visitor removes branches which only contain nodes which have been created
 * as a dependency. It also counts the size of all files.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.05.2016 by Gerrit Hohl
 */
public class DataFileSystemNodeVisitor implements FileSystemNodeVisitor<FileSystemNodeInfo> {
	
	
	/** The log. */
	private Log						log;
	/** The builder. */
	private DebianPackageBuilder	builder;
	/** The default encoding. */
	private Charset					defaultEncoding;
	/** The replacements. */
	private Replacements			replacements;
	/** The context. */
	private Context					context;
	
	/** The size. */
	private long					size;
	/** The number of visited files. */
	private long					files;
	/** The number of visited directories. */
	private long					directories;
	/** The number of visited symbolic links. */
	private long					symbolicLinks;
	/** The flag if at least one file has to be processed. */
	private boolean					process;


	/**
	 * <p>
	 * Creates a visitor.
	 * </p>
	 *
	 * @param log
	 *            The logging.
	 * @param builder
	 *            The builder.
	 * @param defaultEncoding
	 *            The default encoding.
	 * @param replacements
	 *            The replacements.
	 * @param context
	 *            The context.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataFileSystemNodeVisitor(Log log, DebianPackageBuilder builder, Charset defaultEncoding, Replacements replacements,
			Context context) {
		super();
		
		if (log == null)
			throw new IllegalArgumentException("Argument log is null.");
		if (builder == null)
			throw new IllegalArgumentException("Argument builder is null.");
		if (defaultEncoding == null)
			throw new IllegalArgumentException("Argument defaultEncoding is null.");
		if (replacements == null)
			throw new IllegalArgumentException("Argument replacements is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		this.log = log;
		this.builder = builder;
		this.defaultEncoding = defaultEncoding;
		this.replacements = replacements;
		this.context = context;
		
		this.size = 0;
		this.files = 0;
		this.directories = 0;
		this.symbolicLinks = 0;
		this.process = false;
	}
	
	
	/**
	 * <p>
	 * Returns the size of all visited files.
	 * </p>
	 *
	 * @return The size in bytes.
	 */
	public long getSize() {
		return this.size;
	}
	
	
	/**
	 * <p>
	 * Returns the number of visited files.
	 * </p>
	 *
	 * @return The number.
	 */
	public long getFiles() {
		return this.files;
	}
	
	
	/**
	 * <p>
	 * Returns the number of visited directories.
	 * </p>
	 *
	 * @return The number.
	 */
	public long getDirectories() {
		return this.directories;
	}


	/**
	 * <p>
	 * Returns the number of visited symbolic links.
	 * </p>
	 *
	 * @return The number.
	 */
	public long getSymbolicLinks() {
		return this.symbolicLinks;
	}
	
	
	/**
	 * <p>
	 * Returns the flag if at least one file has to be processed.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if at least one file has to be
	 *         processed, <code>false</code> otherwise.
	 */
	public boolean isProcess() {
		return this.process;
	}
	
	
	@Override
	public FileSystemNodeVisitResult preVisitDirectory(FileSystemNode<FileSystemNodeInfo> node) throws IOException {
		FileSystemNodeVisitResult result;
		FileSystemNode<FileSystemNodeInfo> parent;


		if (node == null)
			throw new IllegalArgumentException("Argument node is null.");
		
		if (node.isCreatedByDependency() && node.containsOnlyCreatedByDependency()) {
			result = FileSystemNodeVisitResult.SKIP_SUBTREE;
			parent = node.getParent();
			if (parent != null) {
				parent.removeChild(node);
			}
		} else {
			this.log(node);
			this.directories++;
			// --- Skip the root node. It will be automatically part of the builder ---
			if (node.getParent() != null) {
				this.builder.addDataDirectory(node.getPath().getAbsolutePath(), node.getOwner(), node.getMode());
			}
			result = FileSystemNodeVisitResult.CONTINUE;
		}
		
		return result;
	}


	@SuppressWarnings("resource")
	@Override
	public FileSystemNodeVisitResult visitFile(FileSystemNode<FileSystemNodeInfo> node) throws IOException {
		FileSystemNodeInfo info;
		FileInfo fileInfo;
		Charset encoding;
		DataSource source;
		
		
		if (node == null)
			throw new IllegalArgumentException("Argument node is null.");
		
		this.log(node);
		
		// --- A symbolic link? ---
		if (node.isSymLink()) {
			this.builder.addDataSymLink(node.getPath().getAbsolutePath(), node.getSymLinkTarget().getAbsolutePath(),
					node.getOwner(), node.getMode());
			this.symbolicLinks++;
		}
		// --- Otherwise it is a regular file ---
		else {
			info = node.getAttachment();
			if (info == null)
				throw new IOException("Node |" + node.getPath().getAbsolutePath() + "| doesn't have a file as attachment.");
			fileInfo = info.getSource();
			if (!fileInfo.exists())
				throw new IOException("The file |" + fileInfo.getAbsolutePath() + "| of node |"
						+ node.getPath().getAbsolutePath() + "| doesn't exist.");
			if (!fileInfo.isFile())
				throw new IOException("The file |" + fileInfo.getAbsolutePath() + "| of node |"
						+ node.getPath().getAbsolutePath() + "| is not a regular file.");
			
			this.size += fileInfo.getLength();
			this.files++;
			
			// --- Should we process the content of the file? ---
			if (info.isProcess()) {
				// --- Which encoding should we use? ---
				if (info.getEncoding() == null) {
					encoding = this.defaultEncoding;
				} else {
					try {
						encoding = Charset.forName(info.getEncoding());
					} catch (IllegalArgumentException e) {
						throw new IOException("Encoding |" + info.getEncoding() + "| for source file |"
								+ fileInfo.getAbsolutePath() + "| is not supported by this JVM: " + e.getMessage());
					}
				}
				
				// --- Create a source which replaces the content when it is accessed ---
				source = new DataReplacementSource(fileInfo.getFile(), encoding, this.replacements, this.context);
				this.process = true;
			} else {
				source = new DataFileSource(fileInfo.getFile());
			}
			this.builder.addDataFile(source, node.getPath().getAbsolutePath(), node.getOwner(), node.getMode());
		}
		return FileSystemNodeVisitResult.CONTINUE;
	}


	@Override
	public FileSystemNodeVisitResult postVisitDirectory(FileSystemNode<FileSystemNodeInfo> node) throws IOException {
		if (node == null)
			throw new IllegalArgumentException("Argument node is null.");
		
		return FileSystemNodeVisitResult.CONTINUE;
	}


	/**
	 * <p>
	 * Logs the specified node.
	 * </p>
	 *
	 * @param node
	 *            The node.
	 */
	private void log(FileSystemNode<FileSystemNodeInfo> node) {
		StringBuilder sb;
		FileOwner owner;
		Long groupId, userId, size;
		FileSystemNodeInfo info;
		String source, target;
		
		
		if (this.log.isInfoEnabled()) {
			owner = node.getOwner();
			
			groupId = Long.valueOf(owner.getGroupId());
			userId = Long.valueOf(owner.getUserId());
			size = Long.valueOf(0);
			source = null;
			target = null;
			
			sb = new StringBuilder();
			// --- Directory or regular file? ---
			if (node.isDirectory()) {
				sb.append('d');
			} else {
				info = node.getAttachment();
				if (info != null) {
					size = Long.valueOf(info.getSource().getLength());
					source = info.getSource().getAbsolutePath();
				}
				if (node.isSymLink()) {
					target = node.getSymLinkTarget().getAbsolutePath();
					sb.append('l');
				} else {
					sb.append('-');
				}
			}
			// --- Rights of owner, group and other ---
			sb.append(node.getMode().getText());
			
			// --- Format the rest of the information ---
			sb.append(String.format(" %5d %-8s %5d %-8s %10d %-40s", groupId, owner.getGroupName(), userId, owner.getUserName(),
					size, node.getPath().getAbsolutePath()));
			if (source != null) {
				sb.append(String.format(" (Source: |%s|)", source));
			} else if (target != null) {
				sb.append(String.format(" (Target: |%s|)", target));
			}
			this.log.info(sb.toString());
		}
	}
	
	
}
