/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.File;

import net.sourceforge.javadpkg.plugin.cfg.DataEntry;
import net.sourceforge.javadpkg.plugin.io.Path;

/**
 * <p>
 * A node of a {@link DataEntry}.
 * </p>
 * <p>
 * This class is part of the pre-processing / the parsing of the configuration.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 19.05.2016 by Gerrit Hohl
 */
public class DataEntryNode {
	
	
	/** The source file. */
	private File	source;
	/** The name of the file in the target. */
	private String	name;
	/** The path of the parent in the target. */
	private Path	parent;
	/** The target file or directory of the symbolic link the target. */
	private String	symLink;
	/** The group ID. */
	private Long	groupId;
	/** The group name. */
	private String	groupName;
	/** The user ID. */
	private Long	userId;
	/** The user name. */
	private String	userName;
	/** The mode. */
	private Integer	mode;
	/** The flag if the file should be processed. */
	private boolean	process;
	/** The encoding for the processing. */
	private String	encoding;
	
	
	/**
	 * <p>
	 * Creates a node for a directory.
	 * </p>
	 *
	 * @param name
	 *            The name of the directory in the target.
	 * @param parent
	 *            The path of the parent in the target.
	 * @param groupId
	 *            The group ID (optional).
	 * @param groupName
	 *            The group name (optional).
	 * @param userId
	 *            The user ID (optional).
	 * @param userName
	 *            The user name (optional).
	 * @param mode
	 *            The mode (optional).
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 */
	public DataEntryNode(String name, Path parent, Long groupId, String groupName, Long userId, String userName, Integer mode) {
		
		super();

		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (parent == null)
			throw new IllegalArgumentException("Argument parent is null.");
		
		this.source = null;
		this.name = name;
		this.parent = parent;
		this.symLink = null;
		this.groupId = groupId;
		this.groupName = groupName;
		this.userId = userId;
		this.userName = userName;
		this.mode = mode;
		this.process = false;
		this.encoding = null;
	}


	/**
	 * <p>
	 * Creates a node for a regular file.
	 * </p>
	 *
	 * @param source
	 *            The source file.
	 * @param name
	 *            The name of the directory in the target.
	 * @param parent
	 *            The path of the parent in the target.
	 * @param groupId
	 *            The group ID (optional).
	 * @param groupName
	 *            The group name (optional).
	 * @param userId
	 *            The user ID (optional).
	 * @param userName
	 *            The user name (optional).
	 * @param mode
	 *            The mode (optional).
	 * @param process
	 *            The flag if the file should be processed.
	 * @param encoding
	 *            The encoding for the processing (optional).
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 */
	public DataEntryNode(File source, String name, Path parent, Long groupId, String groupName, Long userId, String userName,
			Integer mode, boolean process, String encoding) {
		
		super();

		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (parent == null)
			throw new IllegalArgumentException("Argument parent is null.");
		
		this.source = source;
		this.name = name;
		this.parent = parent;
		this.symLink = null;
		this.groupId = groupId;
		this.groupName = groupName;
		this.userId = userId;
		this.userName = userName;
		this.mode = mode;
		this.process = process;
		this.encoding = encoding;
	}


	/**
	 * <p>
	 * Creates a node for a symbolic link.
	 * </p>
	 *
	 * @param name
	 *            The name of the symbolic link in the target.
	 * @param parent
	 *            The path of the parent in the target.
	 * @param symLink
	 *            The target file or directory of the symbolic link the target.
	 * @param groupId
	 *            The group ID (optional).
	 * @param groupName
	 *            The group name (optional).
	 * @param userId
	 *            The user ID (optional).
	 * @param userName
	 *            The user name (optional).
	 * @param mode
	 *            The mode (optional).
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 */
	public DataEntryNode(String name, Path parent, String symLink, Long groupId, String groupName, Long userId, String userName,
			Integer mode) {
		
		super();

		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (parent == null)
			throw new IllegalArgumentException("Argument parent is null.");
		if (symLink == null)
			throw new IllegalArgumentException("Argument symLink is null.");
		
		this.source = null;
		this.name = name;
		this.parent = parent;
		this.symLink = symLink;
		this.groupId = groupId;
		this.groupName = groupName;
		this.userId = userId;
		this.userName = userName;
		this.mode = mode;
		this.process = false;
		this.encoding = null;
	}


	/**
	 * <p>
	 * Returns the source file.
	 * </p>
	 *
	 * @return The source file or <code>null</code>, if this node doesn't
	 *         represent a regular file.
	 */
	public File getSource() {
		return this.source;
	}


	/**
	 * <p>
	 * Returns the name of the file in the target.
	 * </p>
	 *
	 * @return The name.
	 */
	public String getName() {
		return this.name;
	}


	/**
	 * <p>
	 * Returns the path of the parent in the target.
	 * </p>
	 *
	 * @return The path.
	 */
	public Path getParent() {
		return this.parent;
	}
	
	
	/**
	 * <p>
	 * Returns the The target file or directory of the symbolic link the target.
	 * </p>
	 *
	 * @return The target or <code>null</code>, if this node doesn't represent a
	 *         symbolic link.
	 */
	public String getSymLink() {
		return this.symLink;
	}


	/**
	 * <p>
	 * Returns the group ID.
	 * </p>
	 *
	 * @return The group ID or <code>null</code>, if the group ID is not set.
	 */
	public Long getGroupId() {
		return this.groupId;
	}


	/**
	 * <p>
	 * returns the group name.
	 * </p>
	 *
	 * @return The group name or <code>null</code>, if the group name is not
	 *         set.
	 */
	public String getGroupName() {
		return this.groupName;
	}


	/**
	 * <p>
	 * Returns the user ID.
	 * </p>
	 *
	 * @return The user ID or <code>null</code>, if the user ID is not set.
	 */
	public Long getUserId() {
		return this.userId;
	}


	/**
	 * <p>
	 * returns the user name.
	 * </p>
	 *
	 * @return The user name or <code>null</code>, if the user name is not set.
	 */
	public String getUserName() {
		return this.userName;
	}


	/**
	 * <p>
	 * Returns the mode.
	 * </p>
	 *
	 * @return The mode or <code>null</code>, if the mode is not set.
	 */
	public Integer getMode() {
		return this.mode;
	}


	/**
	 * <p>
	 * Returns the flag if the file should be processed.
	 * </p>
	 * <p>
	 * This flag has only an effect if this node represents a regular file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the file should be processed,
	 *         <code>false</code> otherwise.
	 * @see #getSource()
	 * @see #getEncoding()
	 */
	public boolean isProcess() {
		return this.process;
	}


	/**
	 * <p>
	 * Returns the encoding for the processing.
	 * </p>
	 * <p>
	 * This flag has only an effect if this node should be processed.
	 * </p>
	 *
	 * @return The encoding or <code>null</code>, if the default encoding should
	 *         be used.
	 * @see #isProcess()
	 */
	public String getEncoding() {
		return this.encoding;
	}


}
