/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.File;
import java.io.IOException;

import org.apache.maven.plugin.logging.Log;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.CopyrightLicense;
import net.sourceforge.javadpkg.CopyrightParser;
import net.sourceforge.javadpkg.FilesCopyright;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.impl.CopyrightImpl;
import net.sourceforge.javadpkg.impl.CopyrightLicenseImpl;
import net.sourceforge.javadpkg.impl.CopyrightParserImpl;
import net.sourceforge.javadpkg.impl.FilesCopyrightImpl;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.impl.DataFileSource;
import net.sourceforge.javadpkg.plugin.CopyrightConfigurationParser;
import net.sourceforge.javadpkg.plugin.cfg.CopyrightConfiguration;
import net.sourceforge.javadpkg.plugin.cfg.CopyrightFilesConfiguration;
import net.sourceforge.javadpkg.plugin.cfg.CopyrightLicenseConfiguration;


/**
 * <p>
 * A {@link CopyrightConfigurationParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 09.05.2016 by Gerrit Hohl
 */
public class CopyrightConfigurationParserImpl implements CopyrightConfigurationParser {
	
	
	/** The parser for the copyright. */
	private CopyrightParser copyrightParser;
	
	
	// TODO Check configuration(s).
	
	
	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public CopyrightConfigurationParserImpl() {
		super();
		
		this.copyrightParser = new CopyrightParserImpl();
	}


	@Override
	public Copyright parseCopyrightConfiguration(Log log, CopyrightConfiguration config, Context context)
			throws IOException, ParseException {
		
		Copyright copyright;
		File file;


		if (config == null)
			throw new IllegalArgumentException("Argument config is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		file = config.getFile();
		if (file != null) {
			copyright = this.parseFile(log, file, context);
		} else {
			copyright = this.parseConfiguration(log, config, context);
		}

		return copyright;
	}


	/**
	 * <p>
	 * Parses the copyright from the specified file.
	 * </p>
	 *
	 * @param log
	 *            The log.
	 * @param file
	 *            The file.
	 * @param context
	 *            The context.
	 * @return The copyright.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private Copyright parseFile(Log log, File file, Context context) throws IOException, ParseException {
		Copyright copyright;


		if (log.isInfoEnabled()) {
			log.info("Read copyright from file |" + file.getAbsolutePath() + "|.");
		}
		try {
			try (DataSource source = new DataFileSource(file)) {
				copyright = this.copyrightParser.parseCopyright(source, context);
			}
		} catch (IOException e) {
			throw new IOException("Couldn't parse the copyright file |" + file.getAbsolutePath() + "|: " + e.getMessage(), e);
		} catch (ParseException e) {
			throw new IOException("Couldn't parse the copyright file |" + file.getAbsolutePath() + "|: " + e.getMessage(), e);
		}
		return copyright;
	}
	
	
	/**
	 * <p>
	 * Parses the copyright from the specified configuration.
	 * </p>
	 *
	 * @param log
	 *            The log.
	 * @param config
	 *            The configuration.
	 * @param context
	 *            The context.
	 * @return The copyright.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private Copyright parseConfiguration(Log log, CopyrightConfiguration config, Context context) throws ParseException {
		CopyrightImpl copyright;
		FilesCopyright filesCopyright;
		CopyrightLicense license;


		if (log.isInfoEnabled()) {
			log.info("Read copyright from configuration.");
		}
		copyright = new CopyrightImpl();
		copyright.setUpstreamName(config.getUpstreamName());
		copyright.setUpstreamContact(config.getUpstreamContact());
		copyright.setSource(config.getSource());
		copyright.setDisclaimer(config.getDisclaimer());
		copyright.setComment(config.getComment());
		copyright.setLicense(this.parseLicenseConfiguration(config.getLicense(), context));
		copyright.setCopyright(config.getCopyright());
		for (CopyrightFilesConfiguration cfg : config.getFiles()) {
			filesCopyright = this.parseFilesConfiguration(cfg, context);
			copyright.addFilesCopyright(filesCopyright);
		}
		for (CopyrightLicenseConfiguration cfg : config.getLicenses()) {
			license = this.parseLicenseConfiguration(cfg, context);
			copyright.addLicense(license);
		}
		return copyright;
	}


	/**
	 * <p>
	 * Parses the license from the specified configuration.
	 * </p>
	 *
	 * @param config
	 *            The configuration.
	 * @param context
	 *            The context.
	 * @return The copyright.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private CopyrightLicense parseLicenseConfiguration(CopyrightLicenseConfiguration config, Context context)
			throws ParseException {
		
		CopyrightLicenseImpl license;


		if (config == null)
			return null;
		
		license = new CopyrightLicenseImpl();
		license.setName(config.getName());
		license.setText(config.getText());

		return license;
	}


	/**
	 * <p>
	 * Parses the copyright for certain files from the specified configuration.
	 * </p>
	 *
	 * @param config
	 *            The configuration.
	 * @param context
	 *            The context.
	 * @return The copyright.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private FilesCopyright parseFilesConfiguration(CopyrightFilesConfiguration config, Context context) throws ParseException {
		FilesCopyrightImpl files;
		
		
		if (config == null)
			return null;
		
		files = new FilesCopyrightImpl();
		files.setFiles(config.getFiles());
		files.setCopyright(config.getCopyright());
		files.setLicense(this.parseLicenseConfiguration(config.getLicense(), context));
		files.setComment(config.getComment());
		
		return files;
	}


}
