/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.logging.Log;

import net.sourceforge.javadpkg.ChangeLog;
import net.sourceforge.javadpkg.ChangeLogConstants;
import net.sourceforge.javadpkg.ChangeLogParser;
import net.sourceforge.javadpkg.ChangeLogUrgencyParser;
import net.sourceforge.javadpkg.ChangeLogVersionEntry;
import net.sourceforge.javadpkg.ChangeLogVersionEntryDetail;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackageMaintainerParser;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.impl.PackageMaintainerParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.impl.ChangeLogImpl;
import net.sourceforge.javadpkg.impl.ChangeLogParserImpl;
import net.sourceforge.javadpkg.impl.ChangeLogUrgencyParserImpl;
import net.sourceforge.javadpkg.impl.ChangeLogVersionEntryDetailImpl;
import net.sourceforge.javadpkg.impl.ChangeLogVersionEntryImpl;
import net.sourceforge.javadpkg.io.impl.DataFileSource;
import net.sourceforge.javadpkg.plugin.ChangeLogConfigurationParser;
import net.sourceforge.javadpkg.plugin.cfg.ChangeLogConfiguration;
import net.sourceforge.javadpkg.plugin.cfg.ChangeLogVersionEntryConfiguration;


/**
 * <p>
 * A {@link ChangeLogConfigurationParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 09.05.2016 by Gerrit Hohl
 */
public class ChangeLogConfigurationParserImpl implements ChangeLogConfigurationParser, ChangeLogConstants {
	
	
	/** The parser for the change log. */
	private ChangeLogParser			changeLogParser;
	/** The parser for the package name. */
	private PackageNameParser		packageNameParser;
	/** The parser for the package version. */
	private PackageVersionParser	packageVersionParser;
	/** The parser for the urgency. */
	private ChangeLogUrgencyParser	changeLogUrgencyParser;
	/** The parser for the package maintainer. */
	private PackageMaintainerParser	packageMaintainerParser;
	
	
	// TODO Check configuration(s).


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public ChangeLogConfigurationParserImpl() {
		super();
		
		this.changeLogParser = new ChangeLogParserImpl();
		this.packageNameParser = new PackageNameParserImpl();
		this.packageVersionParser = new PackageVersionParserImpl();
		this.changeLogUrgencyParser = new ChangeLogUrgencyParserImpl();
		this.packageMaintainerParser = new PackageMaintainerParserImpl();
	}
	
	
	@Override
	public ChangeLog parseChangeLogConfiguration(Log log, ChangeLogConfiguration config, Context context)
			throws IOException, ParseException {
		
		ChangeLog changeLog;
		File file;


		if (config == null)
			throw new IllegalArgumentException("Argument config is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		file = config.getFile();
		if (file != null) {
			changeLog = this.parseFile(log, file, context);
		} else {
			changeLog = this.parseConfiguration(log, config, context);
		}

		return changeLog;
	}


	/**
	 * <p>
	 * Parses the change log from the specified file.
	 * </p>
	 *
	 * @param log
	 *            The log.
	 * @param file
	 *            The file.
	 * @param context
	 *            The context.
	 * @return The change log.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private ChangeLog parseFile(Log log, File file, Context context) throws IOException, ParseException {
		ChangeLog changeLog;
		
		
		if (log.isInfoEnabled()) {
			log.info("Read change log from file |" + file.getAbsolutePath() + "|.");
		}
		try {
			try (DataFileSource source = new DataFileSource(file)) {
				changeLog = this.changeLogParser.parseChangeLog(source, context);
			}
		} catch (IOException e) {
			throw new IOException("Couldn't parse the change log file |" + file.getAbsolutePath() + "|: " + e.getMessage(), e);
		} catch (ParseException e) {
			throw new IOException("Couldn't parse the change log file |" + file.getAbsolutePath() + "|: " + e.getMessage(), e);
		}
		return changeLog;
	}
	
	
	/**
	 * <p>
	 * Parses the change log from the specified configuration.
	 * </p>
	 *
	 * @param log
	 *            The log.
	 * @param config
	 *            The configuration.
	 * @param context
	 *            The context.
	 * @return The change log.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private ChangeLog parseConfiguration(Log log, ChangeLogConfiguration config, Context context) throws ParseException {
		ChangeLogImpl changeLog;
		ChangeLogVersionEntry entry;
		
		
		if (log.isInfoEnabled()) {
			log.info("Read change log from configuration.");
		}
		changeLog = new ChangeLogImpl();
		for (ChangeLogVersionEntryConfiguration cfg : config.getEntries()) {
			entry = this.parseVersionEntryConfiguration(cfg, context);
			changeLog.addEntry(entry);
		}
		return changeLog;
	}


	/**
	 * <p>
	 * Parses the version entry of a change log from the specified
	 * configuration.
	 * </p>
	 *
	 * @param config
	 *            The configuration.
	 * @param context
	 *            The context.
	 * @return The version entry.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private ChangeLogVersionEntry parseVersionEntryConfiguration(ChangeLogVersionEntryConfiguration config, Context context)
			throws ParseException {
		
		ChangeLogVersionEntryImpl entry;


		entry = new ChangeLogVersionEntryImpl();
		entry.setPackageName(this.packageNameParser.parsePackageName(config.getName(), context));
		entry.setVersion(this.packageVersionParser.parsePackageVersion(config.getVersion(), context));
		entry.setDistributions(config.getDistributions());
		entry.setUrgency(this.changeLogUrgencyParser.parseChangeLogUrgency(config.getUrgency(), context));
		entry.setDetails(this.parseDetails(config.getDetails(), context));
		entry.setMaintainer(this.packageMaintainerParser.parsePackageMaintainer(config.getMaintainer(), context));
		try {
			entry.setDate(TIMESTAMP_FORMAT.parse(config.getDate()));
		} catch (java.text.ParseException e) {
			throw new ParseException("Couldn't parse timestamp |" + config.getDate() + "| of version entry: " + e.getMessage(),
					e);
		}
		return entry;
	}


	/**
	 * <p>
	 * Parses the details of a version entry of a change log from the specified
	 * configuration.
	 * </p>
	 *
	 * @param configs
	 *            The configurations.
	 * @param context
	 *            The context.
	 * @return The details.
	 * @throws ParseException
	 *             If an error occurs during the parsing.
	 */
	private List<ChangeLogVersionEntryDetail> parseDetails(List<String> configs, Context context) throws ParseException {
		List<ChangeLogVersionEntryDetail> details;
		ChangeLogVersionEntryDetail detail;
		
		
		details = new ArrayList<>(configs.size());
		for (String config : configs) {
			detail = new ChangeLogVersionEntryDetailImpl(config);
			details.add(detail);
		}
		return details;
	}
	
	
}
