/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.Architecture;
import net.sourceforge.javadpkg.control.ArchitectureParser;
import net.sourceforge.javadpkg.control.BinaryControl;
import net.sourceforge.javadpkg.control.Description;
import net.sourceforge.javadpkg.control.DescriptionParser;
import net.sourceforge.javadpkg.control.Homepage;
import net.sourceforge.javadpkg.control.HomepageParser;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependencyParser;
import net.sourceforge.javadpkg.control.PackageMaintainer;
import net.sourceforge.javadpkg.control.PackageMaintainerParser;
import net.sourceforge.javadpkg.control.PackageName;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackagePriority;
import net.sourceforge.javadpkg.control.PackagePriorityParser;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.Section;
import net.sourceforge.javadpkg.control.SectionParser;
import net.sourceforge.javadpkg.control.Size;
import net.sourceforge.javadpkg.control.impl.ArchitectureParserImpl;
import net.sourceforge.javadpkg.control.impl.BinaryControlImpl;
import net.sourceforge.javadpkg.control.impl.DescriptionParserImpl;
import net.sourceforge.javadpkg.control.impl.HomepageParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageDependencyParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageMaintainerParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackagePriorityParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionRelationOperatorParserImpl;
import net.sourceforge.javadpkg.control.impl.SectionParserImpl;
import net.sourceforge.javadpkg.field.FieldType;
import net.sourceforge.javadpkg.plugin.BinaryControlBuilder;
import net.sourceforge.javadpkg.plugin.ControlRecommendedFieldNotSetWarning;
import net.sourceforge.javadpkg.plugin.cfg.ControlConfiguration;
import net.sourceforge.javadpkg.plugin.cfg.DependenciesConfiguration;


/**
 * <p>
 * A {@link BinaryControlBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 03.05.2016 by Gerrit Hohl
 */
public class BinaryControlBuilderImpl implements BinaryControlBuilder {
	
	
	/** The parser for the package names. */
	private PackageNameParser		packageNameParser;
	/** The parser for the package version. */
	private PackageVersionParser	packageVersionParser;
	/** The parser for the architecture. */
	private ArchitectureParser		architectureParser;
	/** The parser for the &quot;Section&quot; field. */
	private SectionParser			sectionParser;
	/** The parser for the package priority. */
	private PackagePriorityParser	packagePriorityParser;
	/** The parser for the package dependency. */
	private PackageDependencyParser	packageDependencyParser;
	/** The parser for the maintainer. */
	private PackageMaintainerParser	packageMaintainerParser;
	/** The parser for the home-page. */
	private HomepageParser			homepageParser;
	/** The parser for the description. */
	private DescriptionParser		descriptionParser;
	
	
	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 */
	public BinaryControlBuilderImpl() {
		super();

		this.packageNameParser = new PackageNameParserImpl();
		this.packageVersionParser = new PackageVersionParserImpl();
		this.architectureParser = new ArchitectureParserImpl();
		this.sectionParser = new SectionParserImpl();
		this.packagePriorityParser = new PackagePriorityParserImpl();
		this.packageDependencyParser = new PackageDependencyParserImpl(this.packageNameParser,
				new PackageVersionRelationOperatorParserImpl(), this.packageVersionParser);
		this.packageMaintainerParser = new PackageMaintainerParserImpl();
		this.homepageParser = new HomepageParserImpl();
		this.descriptionParser = new DescriptionParserImpl();
	}


	@Override
	public BinaryControl buildBinaryControl(Log log, ControlConfiguration config, Size installedSize, Context context)
			throws MojoExecutionException, MojoFailureException {
		
		BinaryControlImpl control;
		PackageName packageName;
		PackageVersion version;
		Architecture architecture;
		Section section;
		PackagePriority priority;
		Size size;
		List<PackageDependency> dependencies;
		PackageMaintainer maintainer;
		Homepage homepage;
		Description description;


		if (log == null)
			throw new IllegalArgumentException("Argument log is null.");
		if (config == null)
			throw new IllegalArgumentException("Argument config is null.");
		if (installedSize == null)
			throw new IllegalArgumentException("Argument installedSize is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		control = new BinaryControlImpl();

		if (log.isDebugEnabled()) {
			log.debug("Name           : " + config.getName());
			log.debug("Version        : " + config.getVersion());
			log.debug("Architecture   : " + config.getArchitecture());
			log.debug("Section        : " + config.getSection());
			log.debug("Priority       : " + config.getPriority());
			log.debug("Installed Size : " + config.getInstalledSize());
			this.log(log, "Depends", config.getDepends());
			this.log(log, "Recommends", config.getRecommends());
			this.log(log, "Suggest", config.getSuggests());
			this.log(log, "Enhances", config.getEnhances());
			this.log(log, "Pre-Depends", config.getPreDepends());
			this.log(log, "Breaks", config.getBreaks());
			this.log(log, "Conflicts", config.getConflicts());
			this.log(log, "Provides", config.getProvides());
			this.log(log, "Replaces", config.getReplaces());
			this.log(log, "Built-Using", config.getBuiltUsing());
			log.debug("Maintainer     : " + config.getMaintainer());
			log.debug("Homepage       : " + config.getHomepage());
			log.debug("Description    : " + config.getDescription());
		}

		// --- Package name (Mandatory) ---
		if (this.isNotNull("name", config.getName(), FieldType.MANDATORY, context)) {
			try {
				packageName = this.packageNameParser.parsePackageName(config.getName(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |name|: " + e.getMessage(), e);
			}
			control.setPackage(packageName);
		}

		// --- Version (Mandatory) ---
		if (this.isNotNull("version", config.getVersion(), FieldType.MANDATORY, context)) {
			try {
				version = this.packageVersionParser.parsePackageVersion(config.getVersion(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |version|: " + e.getMessage(), e);
			}
			control.setVersion(version);
		}

		// --- Architecture (Mandatory) ---
		if (this.isNotNull("architecture", config.getArchitecture(), FieldType.MANDATORY, context)) {
			try {
				architecture = this.architectureParser.parseArchitecture(config.getArchitecture(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |architecture|: " + e.getMessage(), e);
			}
			control.setArchitecture(architecture);
		}

		// --- Section (Recommended) ---
		if (this.isNotNull("section", config.getSection(), FieldType.RECOMMENDED, context)) {
			try {
				section = this.sectionParser.parseSection(config.getSection(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |section|: " + e.getMessage(), e);
			}
			control.setSection(section);
		}

		// --- Priority (Recommended) ---
		if (this.isNotNull("priority", config.getPriority(), FieldType.RECOMMENDED, context)) {
			try {
				priority = this.packagePriorityParser.parsePackagePriority(config.getPriority(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |priority|: " + e.getMessage(), e);
			}
			control.setPriority(priority);
		}

		// --- Installed-Size ---
		size = this.parseInstalledSize(config.getInstalledSize(), installedSize);
		control.setInstalledSize(size);

		// --- Depends ---
		if (this.isNotNull("depends", config.getDepends(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("depends", config.getDepends(), context);
			control.setDepends(dependencies);
		}
		
		// --- Recommends ---
		if (this.isNotNull("recommends", config.getRecommends(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("recommends", config.getRecommends(), context);
			control.setRecommends(dependencies);
		}

		// --- Suggests ---
		if (this.isNotNull("suggests", config.getSuggests(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("suggests", config.getSuggests(), context);
			control.setSuggests(dependencies);
		}

		// --- Enhances ---
		if (this.isNotNull("enhances", config.getEnhances(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("enhances", config.getEnhances(), context);
			control.setEnhances(dependencies);
		}

		// --- Pre-Depends ---
		if (this.isNotNull("preDepends", config.getPreDepends(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("preDepends", config.getPreDepends(), context);
			control.setPreDepends(dependencies);
		}

		// --- Breaks ---
		if (this.isNotNull("breaks", config.getBreaks(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("breaks", config.getBreaks(), context);
			control.setBreaks(dependencies);
		}

		// --- Conflicts ---
		if (this.isNotNull("conflicts", config.getConflicts(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("conflicts", config.getConflicts(), context);
			control.setConflicts(dependencies);
		}

		// --- Provides ---
		if (this.isNotNull("provides", config.getProvides(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("provides", config.getProvides(), context);
			control.setProvides(dependencies);
		}

		// --- Replaces ---
		if (this.isNotNull("replaces", config.getReplaces(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("replaces", config.getReplaces(), context);
			control.setReplaces(dependencies);
		}

		// --- Built-Using ---
		if (this.isNotNull("builtUsing", config.getBuiltUsing(), FieldType.OPTIONAL, context)) {
			dependencies = this.parsePackageDependencies("builtUsing", config.getBuiltUsing(), context);
			control.setBuiltUsing(dependencies);
		}

		// --- Maintainer (Mandatory) ---
		if (this.isNotNull("maintainer", config.getMaintainer(), FieldType.MANDATORY, context)) {
			try {
				maintainer = this.packageMaintainerParser.parsePackageMaintainer(config.getMaintainer(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |maintainer|: " + e.getMessage(), e);
			}
			control.setMaintainer(maintainer);
		}

		// --- Home-page ---
		if (this.isNotNull("homepage", config.getHomepage(), FieldType.OPTIONAL, context)) {
			try {
				homepage = this.homepageParser.parseHomepage(config.getHomepage(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |homepage|: " + e.getMessage(), e);
			}
			control.setHomepage(homepage);
		}

		// --- Description (Mandatory) ---
		if (this.isNotNull("description", config.getDescription(), FieldType.MANDATORY, context)) {
			try {
				description = this.descriptionParser.parseDescription(config.getDescription(), context);
			} catch (ParseException e) {
				throw new MojoFailureException("Couldn't parse field of type |description|: " + e.getMessage(), e);
			}
			control.setDescription(description);
		}

		return control;
	}
	
	
	/**
	 * <p>
	 * Logs the dependency configuration on level &quot;debug&quot;.
	 * </p>
	 *
	 * @param log
	 *            The log.
	 * @param name
	 *            The name of the dependency configuration type.
	 * @param depCfg
	 *            The dependency configuration.
	 */
	private void log(Log log, String name, DependenciesConfiguration depCfg) {
		List<String> dependencies = null;
		
		
		if (log.isDebugEnabled()) {
			if (depCfg != null) {
				dependencies = depCfg.getDependencies();
			}
			if ((dependencies == null) || dependencies.isEmpty()) {
				log.debug(String.format("%-14s : No dependencies configured.", name));
			} else {
				log.debug(String.format("%-14s :", name));
				for (String value : dependencies) {
					log.debug("   " + value);
				}
			}
		}
	}
	
	
	/**
	 * <p>
	 * Returns the flag if the property is not <code>null</code>.
	 * </p>
	 *
	 * @param propertyName
	 *            The name of the property.
	 * @param propertyValue
	 *            The value of the property.
	 * @param type
	 *            The type of the property.
	 * @param context
	 *            The context.
	 * @return The flag; <code>true</code>, if the property is not
	 *         <code>null</code>, <code>false</code> otherwise.
	 * @throws MojoFailureException
	 *             If the property is a mandatory property and the value is
	 *             <code>null</code>.
	 */
	private boolean isNotNull(String propertyName, Object propertyValue, FieldType type, Context context)
			throws MojoFailureException {
		
		if (propertyValue == null) {
			switch (type) {
				case MANDATORY:
					throw new MojoFailureException("The mandatory field |" + propertyName + "| is not set.");

				case RECOMMENDED:
					context.addWarning(new ControlRecommendedFieldNotSetWarning(propertyName));
					break;
				
				case OPTIONAL:
					break;
				
				default:
			}
			return false;
		}
		return true;
	}


	/**
	 * <p>
	 * Parses the installed size (if necessary).
	 * </p>
	 *
	 * @param config
	 *            The configuration value.
	 * @param installedSize
	 *            The calculated installed size.
	 * @return The installed size.
	 * @throws MojoExecutionException
	 *             If the configuration value exists and an error occurs while
	 *             parsing it.
	 */
	private Size parseInstalledSize(String config, Size installedSize) throws MojoExecutionException {
		Size result;
		boolean overhead;
		String value;
		long valueLong;
		
		
		if (config == null) {
			result = installedSize;
		} else {
			overhead = config.startsWith("+");
			if (overhead) {
				value = config.substring(1);
			} else {
				value = config;
			}
			try {
				valueLong = Long.parseLong(value);
			} catch (NumberFormatException e) {
				throw new MojoExecutionException(
						"Couldn't parse number |" + value + "| of installed-size |" + config + "|: " + e.getMessage(), e);
			}
			if (overhead) {
				result = new Size(valueLong + installedSize.getBytes());
			} else {
				result = new Size(valueLong);
			}
		}
		return result;
	}
	
	
	/**
	 * <p>
	 * Parses the dependencies of the specified property.
	 * </p>
	 *
	 * @param propertyName
	 *            The name of the property.
	 * @param config
	 *            The dependencies configuration.
	 * @param context
	 *            The context.
	 * @return The dependencies.
	 * @throws MojoExecutionException
	 *             If a dependencies couldn't be parsed.
	 */
	private List<PackageDependency> parsePackageDependencies(String propertyName, DependenciesConfiguration config,
			Context context) throws MojoExecutionException {
		
		List<PackageDependency> dependencies;
		PackageDependency dependency;
		
		
		dependencies = new ArrayList<>();
		for (String cfg : config.getDependencies()) {
			try {
				dependency = this.packageDependencyParser.parsePackageDependency(cfg, context);
			} catch (ParseException e) {
				throw new MojoExecutionException(
						"Couldn't parse dependency |" + cfg + "| of type |" + propertyName + "|: " + e.getMessage(), e);
			}
			dependencies.add(dependency);
		}
		return dependencies;
	}
	
	
}
