/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.cfg;

import org.apache.maven.plugins.annotations.Parameter;

/**
 * <p>
 * An entry in the {@link DataConfiguration}.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 03.05.2016 by Gerrit Hohl
 */
public class DataEntry {
	
	
	/** The source path. */
	@Parameter(name = "sourcePath")
	private String	sourcePath;
	/** The target path. */
	@Parameter(name = "targetPath", required = true)
	private String	targetPath;
	/** The symbolic link path. */
	@Parameter(name = "symLinkPath")
	private String	symLinkPath;
	/** The group ID. */
	@Parameter(name = "groupId")
	private Long	groupId;
	/** The group name. */
	@Parameter(name = "groupName")
	private String	groupName;
	/** The user ID. */
	@Parameter(name = "userId")
	private Long	userId;
	/** The user name. */
	@Parameter(name = "userName")
	private String	userName;
	/** The mode. */
	@Parameter(name = "mode")
	private Integer	mode;
	/** The flag if the directory should be included recursively. */
	@Parameter(name = "recursive", defaultValue = "false")
	private boolean	recursive;
	/** The flag if the file should be processed. */
	@Parameter(name = "process", defaultValue = "false")
	private boolean	process;
	/** The encoding for the processing. */
	@Parameter(name = "encoding")
	private String	encoding;


	/**
	 * <p>
	 * Creates an entry.
	 * </p>
	 */
	public DataEntry() {
		super();
		
		this.sourcePath = null;
		this.targetPath = null;
		this.symLinkPath = null;
		this.groupId = null;
		this.groupName = null;
		this.userId = null;
		this.userName = null;
		this.mode = null;
		this.recursive = false;
		this.process = false;
		this.encoding = null;
	}


	/**
	 * <p>
	 * Creates an entry
	 * </p>
	 *
	 * @param sourcePath
	 *            The source path.
	 * @param targetPath
	 *            The target path.
	 * @param symLinkPath
	 *            The symbolic link path.
	 * @param groupId
	 *            The group ID.
	 * @param groupName
	 *            The group name.
	 * @param userId
	 *            The user ID.
	 * @param userName
	 *            The user name.
	 * @param mode
	 *            The mode.
	 * @param recursive
	 *            The flag if the directory should be included recursively.
	 * @param process
	 *            The flag if the file should be processed.
	 * @param encoding
	 *            The encoding for the processing.
	 */
	public DataEntry(String sourcePath, String targetPath, String symLinkPath, Long groupId, String groupName, Long userId,
			String userName, Integer mode, boolean recursive, boolean process, String encoding) {
		
		super();

		this.sourcePath = sourcePath;
		this.targetPath = targetPath;
		this.symLinkPath = symLinkPath;
		this.groupId = groupId;
		this.groupName = groupName;
		this.userId = userId;
		this.userName = userName;
		this.mode = mode;
		this.recursive = recursive;
		this.process = process;
		this.encoding = encoding;
	}
	
	
	/**
	 * <p>
	 * Returns the source path.
	 * </p>
	 *
	 * @return The source path.
	 */
	public String getSourcePath() {
		return this.sourcePath;
	}
	
	
	/**
	 * <p>
	 * Sets the source path.
	 * </p>
	 *
	 * @param sourcePath
	 *            The source path.
	 */
	public void setSourcePath(String sourcePath) {
		this.sourcePath = sourcePath;
	}
	
	
	/**
	 * <p>
	 * Returns the target path.
	 * </p>
	 *
	 * @return The target path.
	 */
	public String getTargetPath() {
		return this.targetPath;
	}
	
	
	/**
	 * <p>
	 * Sets the target path.
	 * </p>
	 *
	 * @param targetPath
	 *            The target path.
	 */
	public void setTargetPath(String targetPath) {
		this.targetPath = targetPath;
	}
	
	
	/**
	 * <p>
	 * Returns the symbolic link path.
	 * </p>
	 *
	 * @return The symbolic link path.
	 */
	public String getSymLinkPath() {
		return this.symLinkPath;
	}
	
	
	/**
	 * <p>
	 * Sets the symbolic link path.
	 * </p>
	 *
	 * @param symLinkPath
	 *            The symbolic link path.
	 */
	public void setSymLinkPath(String symLinkPath) {
		this.symLinkPath = symLinkPath;
	}
	
	
	/**
	 * <p>
	 * Returns the group ID.
	 * </p>
	 *
	 * @return The group ID.
	 */
	public Long getGroupId() {
		return this.groupId;
	}
	
	
	/**
	 * <p>
	 * Sets the group ID.
	 * </p>
	 *
	 * @param groupId
	 *            The group ID.
	 */
	public void setGroupId(Long groupId) {
		this.groupId = groupId;
	}
	
	
	/**
	 * <p>
	 * Returns the group name.
	 * </p>
	 *
	 * @return The group name.
	 */
	public String getGroupName() {
		return this.groupName;
	}
	
	
	/**
	 * <p>
	 * Sets the group name.
	 * </p>
	 *
	 * @param groupName
	 *            The group name.
	 */
	public void setGroupName(String groupName) {
		this.groupName = groupName;
	}
	
	
	/**
	 * <p>
	 * Returns the user ID.
	 * </p>
	 *
	 * @return The user ID.
	 */
	public Long getUserId() {
		return this.userId;
	}
	
	
	/**
	 * <p>
	 * Sets the user ID.
	 * </p>
	 *
	 * @param userId
	 *            The user ID.
	 */
	public void setUserId(Long userId) {
		this.userId = userId;
	}
	
	
	/**
	 * <p>
	 * Returns the user name.
	 * </p>
	 *
	 * @return The user name.
	 */
	public String getUserName() {
		return this.userName;
	}
	
	
	/**
	 * <p>
	 * Sets the user name.
	 * </p>
	 *
	 * @param userName
	 *            The user name.
	 */
	public void setUserName(String userName) {
		this.userName = userName;
	}
	
	
	/**
	 * <p>
	 * Returns the mode.
	 * </p>
	 *
	 * @return The mode.
	 */
	public Integer getMode() {
		return this.mode;
	}
	
	
	/**
	 * <p>
	 * Sets the mode.
	 * </p>
	 *
	 * @param mode
	 *            The mode.
	 */
	public void setMode(Integer mode) {
		this.mode = mode;
	}
	
	
	/**
	 * <p>
	 * Returns the flag if the directory should be included recursively.
	 * </p>
	 * <p>
	 * This flag has only an effect if the source path is pointing on a
	 * directory.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the directory should be included
	 *         recursively, <code>false</code> otherwise.
	 */
	public boolean isRecursive() {
		return this.recursive;
	}
	
	
	/**
	 * <p>
	 * Sets the flag if the directory should be included recursively.
	 * </p>
	 * <p>
	 * This flag has only an effect if the source path is pointing on a
	 * directory.
	 * </p>
	 *
	 * @param recursive
	 *            The flag: <code>true</code>, if the directory should be
	 *            included recursively, <code>false</code> otherwise.
	 */
	public void setRecursive(boolean recursive) {
		this.recursive = recursive;
	}
	
	
	/**
	 * <p>
	 * Returns the flag if the file should be processed.
	 * </p>
	 * <p>
	 * This flag has only an effect if the source path is pointing on a regular
	 * file. Another possibility if the source path is pointing on a directory,
	 * but the recursive option is activated (see the {@link #isRecursive()}
	 * method). Then all found regular files are processed.
	 * </p>
	 * <p>
	 * All variables in that file will be replaced. Therefore it has to be a
	 * text file.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the file should be processed,
	 *         <code>false</code> otherwise.
	 * @see #getEncoding()
	 */
	public boolean isProcess() {
		return this.process;
	}
	
	
	/**
	 * <p>
	 * Sets the flag if the file should be processed.
	 * </p>
	 * <p>
	 * This flag has only an effect if the source path is pointing on a regular
	 * file. Another possibility if the source path is pointing on a directory,
	 * but the recursive option is activated (see the {@link #isRecursive()}
	 * method). Then all found regular files are processed.
	 * </p>
	 * <p>
	 * All variables in that file will be replaced. Therefore it has to be a
	 * text file.
	 * </p>
	 *
	 * @param process
	 *            The flag: <code>true</code>, if the file should be processed,
	 *            <code>false</code> otherwise.
	 * @see #setEncoding(String)
	 */
	public void setProcess(boolean process) {
		this.process = process;
	}
	
	
	/**
	 * <p>
	 * Returns the encoding of the file which should be processed.
	 * </p>
	 *
	 * @return The encoding or <code>null</code>, if the default encoding should
	 *         be used.
	 * @see #isProcess()
	 */
	public String getEncoding() {
		return this.encoding;
	}
	
	
	/**
	 * <p>
	 * Sets the encoding of the file which should be processed.
	 * </p>
	 *
	 * @param encoding
	 *            The encoding or <code>null</code>, if the default encoding
	 *            should be used.
	 * @see #setProcess(boolean)
	 */
	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}
	
	
}
