/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin.cfg;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.maven.plugins.annotations.Parameter;

/**
 * <p>
 * The copyright configuration.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 09.05.2016 by Gerrit Hohl
 */
public class CopyrightConfiguration {
	
	
	/** The file. */
	@Parameter(name = "file")
	private File								file;
	/** The name upstream uses for the software. */
	@Parameter(name = "upstreamName")
	private String								upstreamName;
	/** The upstream contact. */
	@Parameter(name = "upstreamContact")
	private String								upstreamContact;
	/** The source. */
	@Parameter(name = "source")
	private String								source;
	/** The disclaimer. */
	@Parameter(name = "disclaimer")
	private String								disclaimer;
	/** The comment. */
	@Parameter(name = "comment")
	private String								comment;
	/** The license configuration. */
	@Parameter(name = "license")
	private CopyrightLicenseConfiguration		license;
	/** The copyright. */
	@Parameter(name = "copyright")
	private String								copyright;
	/** The files copyright configurations. */
	@Parameter(name = "files")
	private List<CopyrightFilesConfiguration>	files;
	/** The configurations of the licenses. */
	@Parameter(name = "licenses")
	private List<CopyrightLicenseConfiguration>	licenses;
	
	
	/**
	 * <p>
	 * Creates the configuration.
	 * </p>
	 */
	public CopyrightConfiguration() {
		super();
		
		this.file = null;
		this.upstreamName = null;
		this.upstreamContact = null;
		this.source = null;
		this.disclaimer = null;
		this.comment = null;
		this.license = null;
		this.copyright = null;
		this.files = new ArrayList<>();
		this.licenses = new ArrayList<>();
	}


	/**
	 * <p>
	 * Returns the file.
	 * </p>
	 *
	 * @return The file or <code>null</code>, if no file is set.
	 */
	public File getFile() {
		return this.file;
	}


	/**
	 * <p>
	 * Sets the file.
	 * </p>
	 *
	 * @param file
	 *            The file.
	 */
	public void setFile(File file) {
		this.file = file;
	}
	
	
	/**
	 * <p>
	 * Returns the name upstream uses for the software.
	 * </p>
	 *
	 * @return The name or <code>null</code>, if no name is set.
	 */
	public String getUpstreamName() {
		return this.upstreamName;
	}
	
	
	/**
	 * <p>
	 * Sets the name upstream uses for the software.
	 * </p>
	 *
	 * @param upstreamName
	 *            The name.
	 */
	public void setUpstreamName(String upstreamName) {
		this.upstreamName = upstreamName;
	}
	
	
	/**
	 * <p>
	 * Returns the upstream contact.
	 * </p>
	 *
	 * @return The upstream contact or <code>null</code>, if no upstream contact
	 *         is set.
	 */
	public String getUpstreamContact() {
		return this.upstreamContact;
	}
	
	
	/**
	 * <p>
	 * Sets the upstream contact.
	 * </p>
	 *
	 * @param upstreamContact
	 *            The upstream contact.
	 */
	public void setUpstreamContact(String upstreamContact) {
		this.upstreamContact = upstreamContact;
	}
	
	
	/**
	 * <p>
	 * Returns the source.
	 * </p>
	 *
	 * @return The source.
	 */
	public String getSource() {
		return this.source;
	}
	
	
	/**
	 * <p>
	 * Sets the source.
	 * </p>
	 *
	 * @param source
	 *            The source.
	 */
	public void setSource(String source) {
		this.source = source;
	}
	
	
	/**
	 * <p>
	 * Returns the disclaimer.
	 * </p>
	 *
	 * @return The disclaimer.
	 */
	public String getDisclaimer() {
		return this.disclaimer;
	}
	
	
	/**
	 * <p>
	 * Sets the disclaimer.
	 * </p>
	 *
	 * @param disclaimer
	 *            The disclaimer.
	 */
	public void setDisclaimer(String disclaimer) {
		this.disclaimer = disclaimer;
	}
	
	
	/**
	 * <p>
	 * Returns the comment.
	 * </p>
	 *
	 * @return The comment or <code>null</code>, if no comment is set.
	 */
	public String getComment() {
		return this.comment;
	}
	
	
	/**
	 * <p>
	 * Sets the comment.
	 * </p>
	 *
	 * @param comment
	 *            The comment.
	 */
	public void setComment(String comment) {
		this.comment = comment;
	}
	
	
	/**
	 * <p>
	 * Returns the license configuration.
	 * </p>
	 *
	 * @return The license configuration or <code>null</code>, if no license
	 *         configuration is set.
	 */
	public CopyrightLicenseConfiguration getLicense() {
		return this.license;
	}
	
	
	/**
	 * <p>
	 * Sets the license configuration.
	 * </p>
	 *
	 * @param license
	 *            The license configuration.
	 */
	public void setLicense(CopyrightLicenseConfiguration license) {
		this.license = license;
	}


	/**
	 * <p>
	 * Returns the copyright.
	 * </p>
	 * 
	 * @return The copyright or <code>null</code>, if no copyright is set.
	 */
	public String getCopyright() {
		return this.copyright;
	}


	/**
	 * <p>
	 * Sets the copyright.
	 * </p>
	 * 
	 * @param copyright
	 *            The copyright.
	 */
	public void setCopyright(String copyright) {
		this.copyright = copyright;
	}


	/**
	 * <p>
	 * Returns the files copyright configurations.
	 * </p>
	 *
	 * @return The files copyright configurations.
	 */
	public List<CopyrightFilesConfiguration> getFiles() {
		return (new ArrayList<>(this.files));
	}
	
	
	/**
	 * <p>
	 * Sets the files copyright configurations.
	 * </p>
	 *
	 * @param files
	 *            The files copyright configurations.
	 */
	public void setFiles(List<CopyrightFilesConfiguration> files) {
		if (files == null) {
			this.files = new ArrayList<>();
		} else {
			this.files = new ArrayList<>(files);
		}
	}
	
	
	/**
	 * <p>
	 * Returns the configurations of the licenses.
	 * </p>
	 *
	 * @return The configurations.
	 */
	public List<CopyrightLicenseConfiguration> getLicenses() {
		return (new ArrayList<>(this.licenses));
	}
	
	
	/**
	 * <p>
	 * Sets the configurations of the licenses.
	 * </p>
	 *
	 * @param licenses
	 *            The configurations.
	 */
	public void setLicenses(List<CopyrightLicenseConfiguration> licenses) {
		if (licenses == null) {
			this.licenses = new ArrayList<>();
		} else {
			this.licenses = new ArrayList<>(licenses);
		}
	}
	
	
}
