/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.plugin;

import java.util.Map;

import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Settings;

import net.sourceforge.javadpkg.plugin.reflect.PropertyReflection;
import net.sourceforge.javadpkg.plugin.reflect.PropertyReflectionException;
import net.sourceforge.javadpkg.plugin.reflect.PropertyReflectionImpl;
import net.sourceforge.javadpkg.replace.ReplacementException;
import net.sourceforge.javadpkg.replace.Replacements;


/**
 * <p>
 * The replacements which are based on the plug-in properties, the
 * {@link MavenProject}, the environment variables (see {@link System#getenv()})
 * and the system properties (see {@link System#getProperties()}).
 * </p>
 * <p>
 * If the value of a variable is requested by calling the
 * {@link #getValue(String)} method the lookup of the variable is performed the
 * following way:
 * <ul>
 * <li>Does the variable exist in the plug-in properties? If yes, return the
 * value.</li>
 * <li>Does the variable exist in the {@link MavenProject} properties (see
 * {@link MavenProject#getProperties()})? If yes, return the value.</li>
 * <li>Does the variable - removing the leading &quot;project&quot; /
 * &quot;project.&quot; - exist in the {@link MavenProject} as property? If yes,
 * return the value.</li>
 * <li>Does the variable - removing the leading &quot;env&quot; /
 * &quot;env.&quot; - exist as a environment variable? If yes, return the value.
 * </li>
 * <li>Does the variable exist as a system property? If yes, return the value.
 * </li>
 * <li>Otherwise return <code>null</code>.
 * </p>
 * </ul>
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 */
public class ReplacementsMaven implements Replacements {
	
	
	/** The &quot;project&quot; variable. */
	private static final String	PROJECT_VAR			= "project";
	/** The prefix for all &quot;project&quot; variables. */
	private static final String	PROJECT_VAR_PREFIX	= PROJECT_VAR + ".";
	/** The &quot;settings&quot; variable. */
	private static final String	SETTINGS_VAR		= "settings";
	/** The prefix for all &quot;settings&quot; variables. */
	private static final String	SETTINGS_VAR_PREFIX	= SETTINGS_VAR + ".";
	/** The &quot;env&quot; variable. */
	private static final String	ENV_VAR				= "env";
	/** The prefix for all &quot;env&quot; variables. */
	private static final String	ENV_VAR_PREFIX		= ENV_VAR + ".";
	
	
	/** The properties of the plug-in configuration. */
	private Map<Object, Object>	pluginProperties;
	/** The Maven project. */
	private MavenProject		project;
	/** The settings of the local Maven installation. */
	private Settings			settings;
	/** The environment variables. */
	private Map<String, String>	environment;
	/** The system properties. */
	private Map<Object, Object>	systemProperties;
	
	/** The property reflection. */
	private PropertyReflection	propertyReflection;
	
	
	/**
	 * <p>
	 * Creates replacements.
	 * </p>
	 *
	 * @param pluginProperties
	 *            The properties of the plug-in configuration.
	 * @param project
	 *            The Maven project.
	 * @param settings
	 *            The settings of the local Maven installation.
	 * @param environment
	 *            The environment variables.
	 * @param systemProperties
	 *            The system properties.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public ReplacementsMaven(Map<Object, Object> pluginProperties, MavenProject project, Settings settings,
			Map<String, String> environment, Map<Object, Object> systemProperties) {
		
		super();

		if (pluginProperties == null)
			throw new IllegalArgumentException("Argument pluginProperties is null.");
		if (project == null)
			throw new IllegalArgumentException("Argument project is null.");
		if (settings == null)
			throw new IllegalArgumentException("Argument settings is null.");
		if (environment == null)
			throw new IllegalArgumentException("Argument environment is null.");
		if (systemProperties == null)
			throw new IllegalArgumentException("Argument systemProperties is null.");
		
		this.pluginProperties = pluginProperties;
		this.project = project;
		this.settings = settings;
		this.environment = environment;
		this.systemProperties = systemProperties;
		
		this.propertyReflection = new PropertyReflectionImpl();
	}
	
	
	@Override
	public String getValue(String name) throws ReplacementException {
		Object value;


		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		
		// ROADMAP Maybe implement a cache.

		value = this.getMapValue(this.pluginProperties, null, name);
		if (value == null) {
			value = this.getMapValue(this.project.getProperties(), null, name);
		}
		if (value == null) {
			if (PROJECT_VAR.equals(name) || name.startsWith(PROJECT_VAR_PREFIX)) {
				value = this.getProjectValue(name);
			} else if (SETTINGS_VAR.equals(name) || name.startsWith(SETTINGS_VAR_PREFIX)) {
				value = this.getSettingsValue(name);
			} else if (ENV_VAR.equals(name) || name.startsWith(ENV_VAR_PREFIX)) {
				value = this.getEnvironmentValue(name);
			} else {
				value = this.getMapValue(this.systemProperties, null, name);
			}
		}

		if (value instanceof String)
			return ((String) value);
		else if (value != null)
			return value.toString();
		else
			return null;
	}


	/**
	 * <p>
	 * Returns the value for the specified project property.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The value or <code>null</code>, if the property isn't set.
	 * @throws ReplacementException
	 *             If an error occurs while determine the value.
	 */
	private Object getProjectValue(String name) throws ReplacementException {
		Object value;
		String path;


		if (PROJECT_VAR.equals(name))
			return this.project;
		
		path = name.substring(PROJECT_VAR_PREFIX.length());
		try {
			value = this.propertyReflection.getValue(this.project, path);
		} catch (PropertyReflectionException e) {
			throw new ReplacementException("Couldn't determine value for variable |" + name + "|: " + e.getMessage(), e);
		}
		return value;
	}


	/**
	 * <p>
	 * Returns the value for the specified settings property.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The value or <code>null</code>, if the property isn't set.
	 * @throws ReplacementException
	 *             If an error occurs while determine the value.
	 */
	private Object getSettingsValue(String name) throws ReplacementException {
		Object value;
		String path;


		if (SETTINGS_VAR.equals(name))
			return this.settings;
		
		path = name.substring(SETTINGS_VAR_PREFIX.length());
		try {
			value = this.propertyReflection.getValue(this.settings, path);
		} catch (PropertyReflectionException e) {
			throw new ReplacementException("Couldn't determine value for variable |" + name + "|: " + e.getMessage(), e);
		}
		return value;
	}


	/**
	 * <p>
	 * Returns the value for the specified environment property.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The value or <code>null</code>, if no such property exists.
	 */
	private Object getEnvironmentValue(String name) {
		Object value;
		
		
		if (ENV_VAR.equals(name))
			return this.environment;
		
		value = this.getMapValue(this.environment, ENV_VAR_PREFIX, name);
		return value;
	}
	
	
	/**
	 * <p>
	 * Returns the value for the specified key.
	 * </p>
	 *
	 * @param map
	 *            The map.
	 * @param prefix
	 *            The prefix within the name which must be removed before
	 *            accessing the map (optional).
	 * @param name
	 *            The name of the key.
	 * @return The value or <code>null</code>, if no key with the specified name
	 *         exists.
	 */
	private <K, V> V getMapValue(Map<K, V> map, String prefix, String name) {
		Object key;
		V value;
		
		
		if (map.isEmpty())
			return null;
		
		// --- Get the key ---
		if ((prefix == null) || !name.startsWith(prefix)) {
			key = name;
		} else {
			key = name.substring(prefix.length());
		}
		
		// --- Get the value ---
		value = map.get(key);
		return value;
	}


}
