/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.tasks;

import java.util.ArrayList;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;
import net.sourceforge.basher.Task;
import net.sourceforge.basher.TaskFailedException;
import net.sourceforge.basher.TaskNotRunException;
import org.apache.commons.logging.Log;
import org.easymock.MockControl;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestAbstractTask extends TestCase
{
    private MockControl _logControl;
    private Log _log;

    public void testInitalization()
    {

        initializeControls();

        replay();

        final ATestAbstractTask aTestAbstractTask = new ATestAbstractTask();
        final ATestTask aTestTask = new ATestTask();


        final ArrayList<Task> followers = new ArrayList<Task>();
        followers.add(aTestTask);
        aTestAbstractTask.setFollowers(followers);
        aTestAbstractTask.setInertia(1.1F);
        aTestAbstractTask.setLog(_log);
        aTestAbstractTask.setMaxInvocations(10);
        aTestAbstractTask.setRunFrom(100);
        aTestAbstractTask.setStopAfter(200);
        aTestAbstractTask.setWeight(55);


        assertNotNull("no followers", aTestAbstractTask.getFollowers());
        assertEquals("bad follower size", 1, aTestAbstractTask.getFollowers().size());
        assertEquals("bad content of follower", aTestAbstractTask.getFollowers().get(0), aTestTask);


        assertEquals("Bad inertia", 0.0F, 1.1F, aTestAbstractTask.getInertia());
        assertEquals("Bad max invocations", 10, aTestAbstractTask.getMaxInvocations());
        assertEquals("Bad run from", 100, aTestAbstractTask.getRunFrom());
        assertEquals("Bad stop after", 200, aTestAbstractTask.getStopAfter());
        assertEquals("Bad weight", 55, aTestAbstractTask.getWeight());


        assertEquals("bad successes", 0, aTestAbstractTask.getSuccesses());
        assertEquals("bad failures", 0, aTestAbstractTask.getFailures());
        assertEquals("bad not run", 0, aTestAbstractTask.getNotRun());
        assertEquals("bad invocations", 0, aTestAbstractTask.getInvocations());
        assertEquals("bad max time", -1, aTestAbstractTask.getMaxTime());

        assertEquals("bad name", "net.sourceforge.basher.tasks.ATestAbstractTask", aTestAbstractTask.getName());


        verify();

    }

    public void testInertia()
    {


        initializeControls();

        replay();

        ATestAbstractTask aTestAbstractTask = new ATestAbstractTask();


        final ArrayList followers = new ArrayList();
        followers.add(new Integer(1));
        aTestAbstractTask.setFollowers(followers);
        aTestAbstractTask.setInertia(2.0F);
        aTestAbstractTask.setLog(_log);
        aTestAbstractTask.setMaxInvocations(10);
        aTestAbstractTask.setRunFrom(100);
        aTestAbstractTask.setStopAfter(200);
        aTestAbstractTask.setWeight(10);

        try
        {
            aTestAbstractTask.executeTask();
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }


        assertEquals("bad new weight", 20, aTestAbstractTask.getWeight());

        verify();


    }

    public void testInvoke()
    {
        initializeControls();
        replay();

        ATestAbstractTask aTestAbstractTask = new ATestAbstractTask();

        final ArrayList followers = new ArrayList();
        followers.add(new Integer(1));
        aTestAbstractTask.setFollowers(followers);
        aTestAbstractTask.setInertia(2.0F);
        aTestAbstractTask.setLog(_log);
        aTestAbstractTask.setMaxInvocations(10);
        aTestAbstractTask.setRunFrom(100);
        aTestAbstractTask.setStopAfter(200);
        aTestAbstractTask.setWeight(10);

        try
        {
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 1, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 1, aTestAbstractTask.getInvocations());
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 2, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 2, aTestAbstractTask.getInvocations());
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 3, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 3, aTestAbstractTask.getInvocations());
        }
        catch (AssertionFailedError e)
        {
            throw e;
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }


        assertEquals("bad new weight", 80, aTestAbstractTask.getWeight());

        verify();


    }

    public void testReset()
    {
        initializeControls();
        replay();

        final ATestAbstractTask aTestAbstractTask = new ATestAbstractTask();
        final ATestTask aTestTask = new ATestTask();

        final ArrayList<Task> followers = new ArrayList<Task>();
        followers.add(aTestTask);
        aTestAbstractTask.setFollowers(followers);
        aTestAbstractTask.setInertia(2.0F);
        aTestAbstractTask.setLog(_log);
        aTestAbstractTask.setMaxInvocations(10);
        aTestAbstractTask.setRunFrom(100);
        aTestAbstractTask.setStopAfter(200);
        aTestAbstractTask.setWeight(10);

        try
        {
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 1, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 1, aTestAbstractTask.getInvocations());
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 2, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 2, aTestAbstractTask.getInvocations());
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 3, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 3, aTestAbstractTask.getInvocations());
        }
        catch (AssertionFailedError e)
        {
            throw e;
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }

        assertEquals("bad new weight", 80, aTestAbstractTask.getWeight());

        aTestAbstractTask.reset();


        assertNotNull("no followers", aTestAbstractTask.getFollowers());
        assertEquals("bad follower size", 1, aTestAbstractTask.getFollowers().size());
        assertEquals("bad content of follower", aTestAbstractTask.getFollowers().get(0), aTestTask);


        assertEquals("Bad inertia", 0.0F, 1.1F, aTestAbstractTask.getInertia());
        assertEquals("Bad max invocations", 10, aTestAbstractTask.getMaxInvocations());
        assertEquals("Bad run from", 100, aTestAbstractTask.getRunFrom());
        assertEquals("Bad stop after", 200, aTestAbstractTask.getStopAfter());
        assertEquals("Bad weight", 10, aTestAbstractTask.getWeight());


        assertEquals("bad successes", 0, aTestAbstractTask.getSuccesses());
        assertEquals("bad failures", 0, aTestAbstractTask.getFailures());
        assertEquals("bad not run", 0, aTestAbstractTask.getNotRun());
        assertEquals("bad invocations", 0, aTestAbstractTask.getInvocations());
        assertEquals("bad max time", -1, aTestAbstractTask.getMaxTime());


        assertEquals("bad name", "net.sourceforge.basher.tasks.ATestAbstractTask", aTestAbstractTask.getName());


        verify();


    }


    public void testInvokeNotRun()
    {
        initializeControls();
        replay();

        ATestAbstractTask aTestAbstractTask = new ATestAbstractTask();

        final ArrayList followers = new ArrayList();
        followers.add(new Integer(1));
        aTestAbstractTask.setFollowers(followers);
        aTestAbstractTask.setInertia(2.0F);
        aTestAbstractTask.setLog(_log);
        aTestAbstractTask.setMaxInvocations(10);
        aTestAbstractTask.setRunFrom(100);
        aTestAbstractTask.setStopAfter(200);
        aTestAbstractTask.setWeight(10);

        try
        {
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 1, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 1, aTestAbstractTask.getInvocations());
            aTestAbstractTask.setDontRun(true);
            try
            {
                aTestAbstractTask.executeTask();
                fail("Task run");
            }
            catch (TaskNotRunException e)
            {
                // should happen
            }

            assertEquals("bad new not run", 1, aTestAbstractTask.getNotRun());
            assertEquals("bad new total", 1, aTestAbstractTask.getInvocations());

            try
            {
                aTestAbstractTask.executeTask();
                fail("Task run");
            }
            catch (TaskNotRunException e)
            {
                // should happen
            }
            assertEquals("bad new not run", 2, aTestAbstractTask.getNotRun());
            assertEquals("bad new total", 1, aTestAbstractTask.getInvocations());

        }
        catch (AssertionFailedError e)
        {
            throw e;
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }


        assertEquals("bad new weight", 20, aTestAbstractTask.getWeight());

        verify();


    }


    public void testInvokeFailed()
    {
        initializeControls();
        replay();

        ATestAbstractTask aTestAbstractTask = new ATestAbstractTask();

        final ArrayList followers = new ArrayList();
        followers.add(new Integer(1));
        aTestAbstractTask.setFollowers(followers);
        aTestAbstractTask.setInertia(2.0F);
        aTestAbstractTask.setLog(_log);
        aTestAbstractTask.setMaxInvocations(10);
        aTestAbstractTask.setRunFrom(100);
        aTestAbstractTask.setStopAfter(200);
        aTestAbstractTask.setWeight(10);

        try
        {
            aTestAbstractTask.executeTask();
            assertEquals("bad new success", 1, aTestAbstractTask.getSuccesses());
            assertEquals("bad new total", 1, aTestAbstractTask.getInvocations());
            aTestAbstractTask.setFail(true);
            try
            {
                aTestAbstractTask.executeTask();
                fail("Task run");
            }
            catch (TaskFailedException e)
            {
                // should happen

                assertNull("unexpectedcause", e.getCause());

            }

            assertEquals("bad new failures", 1, aTestAbstractTask.getFailures());
            assertEquals("bad new total", 2, aTestAbstractTask.getInvocations());

            aTestAbstractTask.setFail(false);
            aTestAbstractTask.setFailWithException(true);

            try
            {
                aTestAbstractTask.executeTask();
                fail("Task run");
            }
            catch (TaskFailedException e)
            {
                // should happen
                assertTrue("bad cause", e.getCause() instanceof NullPointerException);
            }
            assertEquals("bad new failures", 2, aTestAbstractTask.getFailures());
            assertEquals("bad new total", 3, aTestAbstractTask.getInvocations());

        }
        catch (AssertionFailedError e)
        {
            throw e;
        }
        catch (Throwable throwable)
        {
            fail(throwable.getMessage());
        }


        assertEquals("bad new weight", 20, aTestAbstractTask.getWeight());

        verify();


    }

    private void replay()
    {
        _logControl.replay();
    }

    private void initializeControls()
    {
        _logControl = MockControl.createControl(Log.class);
        _log = (Log) _logControl.getMock();
    }

    private void verify()
    {
        _logControl.verify();
    }


}
