/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.internal.impl;

import junit.framework.TestCase;
import net.sourceforge.basher.BasherContext;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.Task;
import net.sourceforge.basher.TaskManager;
import net.sourceforge.basher.events.PhaseTransitionEvent;
import net.sourceforge.basher.internal.Randomizer;
import net.sourceforge.basher.internal.TaskInvoker;
import net.sourceforge.basher.tasks.ATestTask;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.easymock.MockControl;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestTaskRunnerImpl extends TestCase
{
    private MockControl _randomizerControl;
    private MockControl _taskManagerControl;
    private MockControl _taskInvokerControl;
    private MockControl _logControl;
    private Log _log;
    private Randomizer _randomizer;
    private TaskManager _taskManager;
    private TaskInvoker _taskInvoker;
    public BasherContext _basherContext;
    public TaskRunnerImpl _taskRunnerImpl;


    private void initializeControls()
    {
        _randomizerControl = MockControl.createControl(Randomizer.class);
        _taskManagerControl = MockControl.createControl(TaskManager.class);
        _taskInvokerControl = MockControl.createControl(TaskInvoker.class);
        _logControl = MockControl.createControl(Log.class);

        _log = (Log) _logControl.getMock();
        _taskInvoker = (TaskInvoker) _taskInvokerControl.getMock();
        _randomizer = (Randomizer) _randomizerControl.getMock();
        _taskManager = (TaskManager) _taskManagerControl.getMock();
    }

    private void verify()
    {
        _taskManagerControl.verify();
        _randomizerControl.verify();
        _taskInvokerControl.verify();
        _logControl.verify();
    }

    private void replay()
    {
        _taskManagerControl.replay();
        _randomizerControl.replay();
        _taskInvokerControl.replay();
        _logControl.replay();
    }


    public void testInvoke()
    {
        setupBasherContext(_taskRunnerImpl,500,0);

        final ATestTask aTestTask = new ATestTask();

        _taskManager.getNextRandomTask();
        _taskManagerControl.setReturnValue(aTestTask);

        _taskInvoker.invokeTask(aTestTask);


        _randomizer.getRandomInt(500);
        _randomizerControl.setReturnValue(1000);


        replay();

        Thread t = new Thread()
        {
            public void run()
            {
                try
                {
                    Thread.sleep(500);
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
                _taskRunnerImpl.stopInvoking();
            }
        };

        t.start();
        _taskRunnerImpl.run();

        verify();

    }

    private void setupBasherContext(final TaskRunnerImpl taskRunner, int maxDelay, int minTime)
    {
        final BasherContext basherContext = new BasherContext();
        basherContext.setTaskMaxDelay(maxDelay);
        basherContext.setTaskMinDelay(minTime);
        taskRunner.basherEvent(new PhaseTransitionEvent(basherContext, Phase.START,Phase.RUN));
    }

    public void testInvokeNullTask()
    {
        _taskManager.getNextRandomTask();
        _taskManagerControl.setReturnValue(null);

        setupBasherContext(_taskRunnerImpl, 500, 0);

        _randomizer.getRandomInt(500);
        _randomizerControl.setReturnValue(1000);


        replay();


        Thread t = new Thread()
        {
            public void run()
            {
                try
                {
                    Thread.sleep(500);
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
                _taskRunnerImpl.stopInvoking();
            }
        };

        t.start();
        _taskRunnerImpl.run();
        verify();

    }

    public void testStopRunning()
    {
        setupBasherContext(_taskRunnerImpl, 500, 0);

        _taskRunnerImpl.setTaskInvoker(new TaskInvoker()
        {
            public void invokeTask(Task task)
            {
                try
                {
                    Thread.sleep(200);
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
            }

            public void invokeTask(Task task, boolean invokeFollowers)
            {
            }
        });


        final ATestTask aTestTask = new ATestTask(100);

        _taskManager.getNextRandomTask();
        _taskManagerControl.setReturnValue(aTestTask);

        replay();

        Thread t = new Thread()
        {
            public void run()
            {
                try
                {
                    Thread.sleep(150);
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
                _taskRunnerImpl.stopInvoking();
            }
        };

        t.start();
        _taskRunnerImpl.run();
        verify();

    }


    public void testInvokeWithError()
    {
        setupBasherContext(_taskRunnerImpl, 500, 0);

        final ATestTask aTestTask = new ATestTask();
        _taskManager.getNextRandomTask();
        _taskManagerControl.setReturnValue(aTestTask);
        _taskManager.getNextRandomTask();
        _taskManagerControl.setReturnValue(null);

        _taskInvoker.invokeTask(aTestTask);
        final NullPointerException nullPointerException = new NullPointerException("hehe");
        _taskInvokerControl.setThrowable(nullPointerException);


        _randomizer.getRandomInt(500);
        _randomizerControl.setReturnValue(1000);

        _taskRunnerImpl.setLog(_log);
        _log.error("hehe", nullPointerException);

        replay();

        Thread t = new Thread()
        {
            public void run()
            {
                try
                {
                    Thread.sleep(500);
                }
                catch (InterruptedException e)
                {
                    e.printStackTrace();
                }
                _taskRunnerImpl.stopInvoking();
            }
        };

        t.start();
        _taskRunnerImpl.run();
        verify();

    }

    protected void setUp() throws Exception
    {
        super.setUp();
        _basherContext = new BasherContext();
        _taskRunnerImpl = new TaskRunnerImpl();

        initializeControls();

        _taskRunnerImpl.setLog(LogFactory.getLog(TaskRunnerImpl.class));
        _taskRunnerImpl.setRandomizer(_randomizer);
        _taskRunnerImpl.setTaskManager(_taskManager);
        _taskRunnerImpl.setTaskInvoker(_taskInvoker);

    }
}
