/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.impl;

import java.util.Timer;
import java.util.TimerTask;

import net.sourceforge.basher.BasherContext;
import net.sourceforge.basher.BasherTestCase;
import net.sourceforge.basher.ContextManager;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.events.EventManager;
import net.sourceforge.basher.events.PhaseTransitionEvent;
import net.sourceforge.basher.events.ThreadAddedEvent;
import net.sourceforge.basher.events.ThreadRemovedEvent;
import net.sourceforge.basher.internal.TaskRunner;
import org.apache.commons.logging.Log;
import static org.easymock.EasyMock.*;
import org.easymock.EasyMock;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TestSchedulerImpl extends BasherTestCase
{
    private Log _log;
    private ContextManager _contextManager;
    private EventManager _eventManager;

    private SchedulerImpl _schedulerImpl;
    public BasherContext _basherContext;

    private TaskRunner _taskRunner;

    private Timer _timer;

    public void testOneThread()
    {
        initializeSchedulerImpl(1, 10);
        callGetActiveBasherContext(1);

        replayAll();

        _schedulerImpl.start();

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

    }

    private void initializeLoggingForStart(final int count)
    {
        _log.info("Starting scheduler with context: default");

        for (int i = 0; i < count; i++)
        {
            _log.debug("Adding new thread");
            _log.info("Thread added.  " + (i+1) + " thread(s) running");
        }
        _log.info("Scheduler started.  " + count + " thread(s) running");

        _log.debug("Scheduling phase transition SETUP -> RUN at 1000");
        _log.debug("Scheduling phase transition RUN -> COOLDOWN at 1001");
        _log.debug("Scheduling phase transition COOLDOWN -> END at 1002");
    }

    private void initializeSchedulerImpl(final int initialNumberThreads, final int maxThreads)
    {
        _log = createMock(Log.class);

        initializeLoggingForStart(initialNumberThreads);

        _timer = org.easymock.classextension.EasyMock.createMock(Timer.class);

        _schedulerImpl = new SchedulerImpl();

        _schedulerImpl.setLog(_log);

        _contextManager = createMock(ContextManager.class);
        _basherContext = new BasherContext();
        _basherContext.setName("default");
        _basherContext.setInitialNumberThreads(initialNumberThreads);
        _basherContext.setMaxNumberThreads(maxThreads);

        _contextManager.setActiveBasherContext(_basherContext);

        expect(_contextManager.getBasherContext("default")).andReturn(_basherContext);

        _eventManager = createMock(EventManager.class);
        makeThreadSafe(_eventManager, true);

        expect(_timer.purge()).andReturn(0);
        _log.debug("Purged 0 old timer tasks");

        _eventManager.publish(new PhaseTransitionEvent(_basherContext,null,Phase.START));
//        _timer.schedule(new EventEmitterTimerTask(_eventManager, new PhaseTransitionEvent(_basherContext, Phase.SETUP, Phase.RUN)), 0);
//        _timer.schedule(new EventEmitterTimerTask(_eventManager, new PhaseTransitionEvent(_basherContext, Phase.RUN, Phase.COOLDOWN)), 1001);
//        _timer.schedule(new EventEmitterTimerTask(_eventManager, new PhaseTransitionEvent(_basherContext, Phase.COOLDOWN, Phase.END)), 1002);
        _timer.schedule((TimerTask) anyObject(), anyLong());
        EasyMock.expectLastCall().times(5);



        _taskRunner = createMock(TaskRunner.class);
        makeThreadSafe(_taskRunner, true);

        _schedulerImpl.setEventManager(_eventManager);
        _schedulerImpl.setContextManager(_contextManager);
        _schedulerImpl.setTimer(_timer);
        _schedulerImpl.setTaskRunner(_taskRunner);

        _eventManager.publish(new PhaseTransitionEvent(_basherContext, Phase.START, Phase.SETUP));

        for (int i = 0; i < initialNumberThreads; i++)
        {
            _eventManager.publish(new ThreadAddedEvent("TaskRunner-" + i, _basherContext));
            _taskRunner.run();
        }

    }

    public void testAddOneThread()
    {
        initializeSchedulerImpl(1, 10);
        callGetActiveBasherContext(2);

        initializeLoggingForStart(1);

        _log.debug("Adding new thread");
        _log.info("Thread added.  2 thread(s) running");


        _taskRunner.run();
        _eventManager.publish(new ThreadAddedEvent("TaskRunner-1", _basherContext));

        replayAll();

        _schedulerImpl.start();

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

        _schedulerImpl.addThread();
        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 2, numThreads);

    }

    public void testAddOneThreadAndStop()
    {
        initializeSchedulerImpl(1, 10);
        callGetActiveBasherContext(2);
        initializeLoggingForStart(1);

        _log.debug("Adding new thread");
        _log.info("Thread added.  2 thread(s) running");

        initLoggingForStop(2);

        _taskRunner.run();
        _eventManager.publish(new ThreadAddedEvent("TaskRunner-1", _basherContext));
        _eventManager.publish(new PhaseTransitionEvent(_basherContext, Phase.RUN, Phase.END));

        _timer.cancel();

        replayAll();

        _schedulerImpl.start();

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

        _schedulerImpl.addThread();
        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 2, numThreads);

        sleep(200);
        _schedulerImpl.stop();


    }

    private void initLoggingForStop(final int count)
    {
        _log.info("Stopping scheduler");
        for (int i = 0; i < count; i++)
        {
            _log.debug("Stopping task runner name");
        }
        _log.info("Scheduler stopped");
    }


    public void testStop()
    {
        initializeSchedulerImpl(0, 10);
        initializeLoggingForStart(1);
        initLoggingForStop(0);

        _timer.cancel();

        expect(_contextManager.getActiveBasherContext()).andReturn(_basherContext);

        _eventManager.publish(new PhaseTransitionEvent(_basherContext, Phase.RUN, Phase.END));

        replayAll();

        _schedulerImpl.start();

        sleep(100);

        final int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 0, numThreads);

        _schedulerImpl.stop();
        assertFalse("Running when not supposed to", _schedulerImpl.isRunning());


    }

    public void testStart()
    {

        initializeSchedulerImpl(0, 10);

        _log.info("Starting scheduler with context: default");

        _log.info("Scheduler started.  0 thread(s) running");

        replayAll();

        _schedulerImpl.start();

        final int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 0, numThreads);

        assertTrue("Not running", _schedulerImpl.isRunning());

        try
        {
            _schedulerImpl.start();
            fail("Could start again");
        }
        catch (IllegalStateException e)
        {
            assertEquals("Already started", e.getMessage());
        }
        verifyAllMockControls();

    }

    public void testStartNullParams()
    {
        initializeSchedulerImpl(0, 0);
        replayAll();

        try
        {
            _schedulerImpl.start((String) null);
            fail("Could start with null context name");
        }
        catch (NullPointerException e)
        {
            // should happen
        }
        try
        {
            _schedulerImpl.start((BasherContext) null);
            fail("Could start with null context");
        }
        catch (NullPointerException e)
        {
            // should happen
        }
    }

    public void testStartWithUnknownContextName()
    {
        initializeSchedulerImpl(0, 0);
        try
        {
            _schedulerImpl.start("notknown");
            fail("Could start with unknown context name");
        }
        catch (IllegalArgumentException e)
        {
            // should happen
        }
    }

    public void testSchedulingAtStart()
    {
        initializeSchedulerImpl(0, 0);

        replayAll();

        _schedulerImpl.start();


        verifyAll();
    }

    public void testAddOneThreadWhenMaxReached()
    {
        initializeSchedulerImpl(1, 1);
        callGetActiveBasherContext(2);

        _log.warn("Maximum thread limit (1) reached, not adding more threads");

        replayAll();

        _schedulerImpl.start();

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

        _schedulerImpl.addThread();
        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

        sleep(100);

        verifyAll();
    }

    public void testAddManyThreads()
    {

        initializeSchedulerImpl(1, 10);
        callGetActiveBasherContext(2);

        _log.debug("Adding 2 thread(s)");

        _log.debug("Adding new thread");
        _log.info("Thread added.  2 thread(s) running");

        _log.debug("Adding new thread");
        _log.info("Thread added.  3 thread(s) running");

        _log.debug("2 thread(s) added");

        _taskRunner.run();
        _taskRunner.run();

        _eventManager.publish(new ThreadAddedEvent("TaskRunner-1", _basherContext));
        _eventManager.publish(new ThreadAddedEvent("TaskRunner-2", _basherContext));

        replayAll();

        _schedulerImpl.start();

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

        _schedulerImpl.addThreads(2);

        sleep(200);

        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 3, numThreads);

        verifyAll();


    }

    private void callGetActiveBasherContext(final int numInvocations)
    {
        for (int i = 0; i < numInvocations; i++)
        {
            expect(_contextManager.getActiveBasherContext()).andReturn(_basherContext);
        }
    }

    private void verifyAll()
    {
        verify(_contextManager, _log, _eventManager, _taskRunner);
        org.easymock.classextension.EasyMock.verify(_timer);
    }

    public void testRemoveThreads()
    {
        initializeSchedulerImpl(1, 10);

        _log.debug("Removing thread from active list");

        _log.debug("Thread TaskRunner-0 removed from active list");

        _log.debug("Signalling stop thread for thread named: TaskRunner-0");

        _log.debug("Thread stop signalled");

        _eventManager.publish(new ThreadRemovedEvent("TaskRunner-0"));

        replayAll();

        _schedulerImpl.start();

        sleep(100);

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 1, numThreads);

        _schedulerImpl.removeThread();
        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 0, numThreads);

        verifyAll();

    }

    public void testRemoveThreadsWhenNoneToRemove()
    {

        initializeSchedulerImpl(0, 10);

        _log.warn("No threads to remove");

        replayAll();

        _schedulerImpl.start();

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 0, numThreads);

        _schedulerImpl.removeThread();
        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 0, numThreads);


        verifyAll();
    }

    public void testInitialization()
    {
        _schedulerImpl = new SchedulerImpl();
        try
        {
            _schedulerImpl.start();
            fail("Passed log check");
        }
        catch (IllegalStateException e)
        {
            assertEquals("no log", e.getMessage());
        }

        _schedulerImpl = new SchedulerImpl();
        _log = createMock(Log.class);
        _schedulerImpl.setLog(_log);
        try
        {
            _schedulerImpl.start();
            fail("Passed context manager check");
        }
        catch (IllegalStateException e)
        {
            assertEquals("no context manager", e.getMessage());
        }
        _contextManager = createMock(ContextManager.class);
        _schedulerImpl.setContextManager(_contextManager);
        try
        {
            _schedulerImpl.start();
            fail("Passed event manager check");
        }
        catch (IllegalStateException e)
        {
            assertEquals("no event manager", e.getMessage());
        }

        _eventManager = createMock(EventManager.class);
        _schedulerImpl.setEventManager(_eventManager);
        try
        {
            _schedulerImpl.start();
            fail("Passed task runner check");
        }
        catch (IllegalStateException e)
        {
            assertEquals("no task runner", e.getMessage());
        }

        _schedulerImpl = new SchedulerImpl();
        final String message = "Not started";
        try
        {
            _schedulerImpl.addThread();
            fail("Passed add thread");
        }
        catch (IllegalStateException e)
        {
            assertEquals(message, e.getMessage());
        }
        try
        {
            _schedulerImpl.addThreads(5);
            fail("Passed add n threads");
        }
        catch (IllegalStateException e)
        {
            assertEquals(message, e.getMessage());
        }
        try
        {
            _schedulerImpl.getNumberOfActiveThreads();
        }
        catch (IllegalStateException e)
        {
            assertEquals(message, e.getMessage());
        }
        try
        {
            _schedulerImpl.removeAllThreads();
        }
        catch (IllegalStateException e)
        {
            assertEquals(message, e.getMessage());
        }
        try
        {
            _schedulerImpl.removeThread();
        }
        catch (IllegalStateException e)
        {
            assertEquals(message, e.getMessage());
        }

        try
        {
            _schedulerImpl.stop();
        }
        catch (IllegalStateException e)
        {
            assertEquals("Already stopped", e.getMessage());
        }
        // verifyAll();
    }

    public void testRemoveAllThreads()
    {

        initializeSchedulerImpl(2, 10);

        _log.debug("Removing 2 thread(s)");

        _log.debug("Removing thread from active list");
        _log.debug("Removing thread from active list");

        _log.debug("Thread TaskRunner-0 removed from active list");
        _log.debug("Thread TaskRunner-1 removed from active list");

        _log.debug("Signalling stop thread for thread named: TaskRunner-0");
        _log.debug("Signalling stop thread for thread named: TaskRunner-1");

        _log.debug("Thread stop signalled");
        _log.debug("Thread stop signalled");

        _log.debug("2 thread(s) removed");

        _eventManager.publish(new ThreadRemovedEvent("TaskRunner-0"));
        _eventManager.publish(new ThreadRemovedEvent("TaskRunner-1"));

        replayAll();

        _schedulerImpl.start();

        sleep(100);

        int numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 2, numThreads);

        _schedulerImpl.removeAllThreads();
        numThreads = _schedulerImpl.getNumberOfActiveThreads();
        assertEquals("Invalid number of threads", 0, numThreads);

        verifyAll();

    }

    private void replayAll()
    {
        replay(_contextManager, _log, _eventManager, _taskRunner);
        org.easymock.classextension.EasyMock.replay(_timer);
    }

}
