/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.tasks;

import java.util.Collections;
import java.util.List;
import java.util.Arrays;

import net.sourceforge.basher.Task;
import net.sourceforge.basher.TaskFailedException;
import net.sourceforge.basher.TaskNotRunException;
import net.sourceforge.basher.Phase;
import org.apache.commons.logging.Log;

/**
 * A base implementation of the <code>Task</code> interface.  This class provides a skeleton implementastion
 * for all the methods defined by the <code>Task</code> interface.  Sub-classes should only be concerned with with
 * implementing the abstract method {@link net.sourceforge.basher.tasks.AbstractTask#doExecuteTask()}.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public abstract class AbstractTask implements Task
{
    /**
     * Instance logger
     */
    protected Log _log;

    /**
     * instance failure counter
     */
    private volatile int _failures = 0;
    /**
     * instance success counter
     */
    private volatile int _successes = 0;
    /**
     * instance not run counter
     */
    private volatile int _notRun = 0;

    /**
     * Instance weight as a number between 0 and 100 - 0 indicating the task would not run, 100 that it always runs.
     * Defaults to 100 (always runs).
     */
    private int _weight = 100;

    /**
     * instance mirror of weight as the "real" float.  This is to avoid stompin on the same number.
     */
    private float _floatWeight = _weight;

    /**
     * Instance backup of the weight - kept for reset
     */
    private int _originalWeight = _weight;

    /**
     * instance inertia
     */
    private float _inertia = 1.0F;

    /**
     * maximum invocation - if non-zero, this task should execute at most _maxInvocations times
     */
    private int _maxInvocations = 0;

    /**
     * Instance list containing tasks that may follow this <code>Task</code>
     */
    private List<Task> _followers = Collections.emptyList();

    /**
     * Defines the time (internal Basher time) at which point the task will start running.  Defaults to 0 (start initially).
     */
    private long _runFrom = 0;

    /**
     * Defines the time (internal Basher time) at which point the task will stop running.  Defaults to
     * <code>Long.MAX_VALUE</code> (run forever).
     */
    private long _stopAfter = Long.MAX_VALUE;

    private int _maxTime = -1;

    /**
     * Main method that should be called by a component to execute this <code>AbstractTask</code>.  This will ensure that
     * counters of failures/successes are kept up to date.
     *
     * @throws Throwable Re-thrown by this <code>AbstractTask</code> from invoking the {@link AbstractTask#doExecuteTask()} method.
     */
    public final void executeTask() throws Throwable
    {
        try
        {
            doExecuteTask();
            _successes++;
            if (_inertia != 1.0F)
            {
                reCalculateWeight();
            }
        }
        catch (TaskNotRunException e)
        {
            _notRun++;
            throw e;
        }
        catch (Throwable throwable)
        {
            _failures++;
            throw throwable;
        }
    }

    /**
     * Method required to be implemented by sub-classes of this <code>AbstractTask</code>.  This will be invoked by
     * the <code>AbstractTask</code> and it's success or failure to complete will be recorded.<br/>
     *
     * @throws Throwable Added to allow sub-classes to simply fall through.  This exception is caught fully (including any
     *                   runtime/throwable/error exceptions) internally by the <code>AbstractTask</code>.
     */
    protected abstract void doExecuteTask() throws Throwable;

    /**
     * Resets the counters of this <code>Task</code> instance.
     */
    public final void reset()
    {
        _failures = 0;
        _notRun = 0;
        _successes = 0;

        _weight = _originalWeight;
        _floatWeight = _weight;

    }

    /**
     * Retrieves the weight of this <code>AbstractTask</code>.  This could be used by certain schedulers to allow for
     * more fine grained control of task scheduling
     *
     * @return A value between 1 and 100 (higher signals a wish to run more frequently)
     */
    public final int getWeight()
    {
        return _weight;
    }

    /**
     * Retrieves the inertia of this <code>AbstractTask</code>.
     * See {@link net.sourceforge.basher.tasks.AbstractTask#setInertia(float)}}.
     *
     * @return The rate of change for this instance.
     */
    public final float getInertia()
    {
        return _inertia;
    }

    /**
     * Sets the inertia of this <code>AbstractTask</code>.  This is used to control the change of
     * the weight assocaited with this task, calculated for each invocation of the task (if it is
     * not 1.0).
     *
     * @param inertia The inertia to give this instance.  This should be a value between 0.1 and 2.0.
     */
    public final void setInertia(final float inertia)
    {
        _inertia = inertia;
    }

    /**
     * Sets the weight for this task.  This <i>could</i> be used by a scheduler to control the invocations of this
     * <code>Task</code>.
     *
     * @param weight The weight of this task.  Should be a value between 1 and 100.
     */
    public final void setWeight(final int weight)
    {
        _weight = weight;
        // Mirror the weight in the float calculations too ...
        _floatWeight = _weight;

        // And back these settings up ...
        _originalWeight = weight;
        _floatWeight = weight;

    }

    /**
     * Sets the maximum number of invocations that this task should do.  See
     * {@link net.sourceforge.basher.Task#getMaxInvocations()} for more information.
     *
     * @param maxInvocations The number of invocations to do.  A value of 0 (zero) or less indicates that the task
     *                       should always be invoked.
     */
    public final void setMaxInvocations(final int maxInvocations)
    {
        _maxInvocations = maxInvocations;
    }

    /**
     * Retrieves the maximum number of invocations that this task should do.  The number returned <i>could</i> be used
     * by a scheduler to control the invocation of this <code>Task</code>.
     *
     * @return The maximum number of invocations
     */
    public final int getMaxInvocations()
    {
        return _maxInvocations;
    }

    /**
     * Retrieves the number of times this task was invoked (successes+failures).
     *
     * @return The number of invocations.
     */
    public final int getInvocations()
    {
        return _successes + _failures;
    }

    /**
     * Retrieves the number of failures of this task.
     *
     * @return The nunber of times this task failed
     */
    public final int getFailures()
    {
        return _failures;
    }

    /**
     * Retrieves the number of successes of this task.
     *
     * @return The number of times this task succeeded.
     */
    public final int getSuccesses()
    {
        return _successes;
    }

    /**
     * Retrieves the number of not run of this task.
     *
     * @return The number of times this task didn't run.
     */
    public final int getNotRun()
    {
        return _notRun;
    }

    /**
     * Convenience method for sub-classes to signal a not run.
     *
     * @param reason The reason for not running
     * @param cause  The (optional) cause of not run
     * @throws TaskNotRunException Will <b>ALWAYS</b> throw a <code>TaskNotRunException</code> to indicate the task didnt run.
     */
    protected final void notRun(final String reason, final Throwable cause)
    {
        throw new TaskNotRunException(reason);
    }


    /**
     * Convenience method for sub-classes to signal a failure.
     *
     * @param reason The reason for the failure
     * @param cause  The (optional) cause of the failure
     * @throws TaskFailedException Will <b>ALWAYS</b> throw a <code>TaskFailedException</code> to indicate a failure.
     */
    protected final void failed(final String reason, final Throwable cause)
    {
        throw new TaskFailedException(reason, cause);
    }

    /**
     * Convenience method for sub-classes to signal a failure.
     *
     * @param reason The reason for the failure
     * @throws TaskFailedException Will <b>ALWAYS</b> throw a <code>TaskFailedException</code> to indicate a failure.
     */
    protected final void failed(final String reason)
    {
        failed(reason, null);
    }

    /**
     * Recalculates the weight of this <code>Task</code>.  This will take the inertia and multiply it with the current
     * weight and cast the result to an integer.
     */
    private void reCalculateWeight()
    {
        // This ensure we dont stand stompin' on the same number
        // if the inertia is small ...
        _floatWeight = _floatWeight * _inertia;
        _weight = (int) _floatWeight;
    }

    /**
     * Retrieves the followers (tasks that could optionally be invoked after invoking this task).
     *
     * @return A list of <code>Task</code> instances that could follow this task.
     */
    public final List<Task> getFollowers()
    {
        return _followers;
    }

    /**
     * Sets the list of followers (tasks that could optionally be invoked after invokng this task).
     *
     * @param followers A list of <code>Task</code> instances that could follow this task.
     */
    public final void setFollowers(final List<Task> followers)
    {
        _followers = Collections.unmodifiableList(followers);
    }

    /**
     * Retrieves the defualt time from which this task should run.
     *
     * @return The time from which this task should run.  Always returns 0 (always run) unless overriden by sub-class.
     */
    public final long getRunFrom()
    {
        return _runFrom;
    }

    /**
     * Retrieves the defualt time from which this task should stop running.
     *
     * @return The time from which this task should stop running.  Always returns {@link java.lang.Long#MAX_VALUE} (run forever) unless overriden by sub-class.
     */
    public final long getStopAfter()
    {
        return _stopAfter;
    }

    public final void setRunFrom(final long runFrom)
    {
        _runFrom = runFrom;
    }

    public final void setStopAfter(final long stopAfter)
    {
        _stopAfter = stopAfter;
    }

    /** Defines the <code>Phase</code>s in which this <code>Task</code> is run.  This, by default, returns only
     * <code>Phase.RUN</code>.
     *
     * @return List of phases this <code>Task</code> should run in - only RUN phase is included by default
     */
    public List<Phase> applicablePhases()
    {
        return Arrays.asList(Phase.RUN);
    }

    /** Returns the default max time for <code>Task</code>s.
     *
      * @return Always returns -1, signifying no maximum time
     */
    public int getMaxTime()
    {
        return _maxTime;
    }

    public void setMaxTime(final int maxTime)
    {
        _maxTime = maxTime;
    }

    /**
     * Sets the <code>Log</code> instance for this <code>Task</code> instance.
     *
     * @param log The log instance to use for logging.
     */
    public final void setLog(final Log log)
    {
        _log = log;
    }

    /**
     * A default implementation of the {@link Task#getName()} method, simply returning the class name of this
     * <code>Task</code> instance.  Sub-classes may return their own name should so be desired.
     *
     * @return The class name of this task instance.
     */
    public  String getName()
    {
        return getClass().getName();
    }

    /**
     * Returns a string representation of this <code>Task</code> instance.
     *
     * @return A string representation of this <code>Task</code>.
     *         <p/>
     *         {@inheritDoc}
     */
    public final String toString()
    {
        return "AbstractTask{" +
                "_failures=" + _failures +
                ", _successes=" + _successes +
                ", _notRun=" + _notRun +
                ", _weight=" + _weight +
                ", _floatWeight=" + _floatWeight +
                ", _inertia=" + _inertia +
                ", _maxInvocations=" + _maxInvocations +
                ", _followers=" + _followers +
                "}";
    }
}
