/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.internal.impl;

import java.io.File;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.Reader;
import java.util.*;

import bsh.EvalError;
import bsh.Interpreter;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.Scheduler;
import net.sourceforge.basher.TaskManager;
import net.sourceforge.basher.events.BasherEvent;
import net.sourceforge.basher.events.BasherEventListener;
import net.sourceforge.basher.events.PhaseTransitionEvent;
import net.sourceforge.basher.internal.BeanShellInitializer;
import org.apache.commons.logging.Log;

/**
 * @author Johan Lindquist
 */
public class BeanShellInitializerImpl implements BeanShellInitializer, BasherEventListener
{
    private TaskManager _taskManager;
    private Log _logger;
    private Scheduler _scheduler;

    private FilenameFilter _filenameFilter;
    private Comparator<File> _filenameComparator;

    public void setFilenameFilter(final FilenameFilter filenameFilter)
    {
        _filenameFilter = filenameFilter;
    }

    public void setFilenameComparator(final Comparator<File> filenameComparator)
    {
        _filenameComparator = filenameComparator;
    }

    public void setLog(final Log logger)
    {
        _logger = logger;
    }

    public void setScheduler(final Scheduler scheduler)
    {
        _scheduler = scheduler;
    }

    public void setTaskManager(final TaskManager taskManager)
    {
        _taskManager = taskManager;
    }

    private void executeScripts(final File scriptDirectory) throws Exception
    {
        // Validate some of the params
        if (!scriptDirectory.exists())
        {
            throw new Exception("Script directory '" + scriptDirectory.getAbsolutePath() + "' does not exist");
        }
        if (!scriptDirectory.canRead())
        {
            throw new Exception("Read access denied to script directory '" + scriptDirectory.getAbsolutePath() + "'");
        }

        _logger.info("Using script directory " + scriptDirectory.getAbsolutePath());

        // Loop over all scripts in the script directory, sorted by name
        final File[] scriptFiles = scriptDirectory.listFiles(_filenameFilter);

        final List<File> scriptFileList = new ArrayList<File>(Arrays.asList(scriptFiles));
        Collections.sort(scriptFileList, _filenameComparator);

        _logger.debug("Executing " + scriptFileList.size() + " script(s)");

        for (Iterator iterator = scriptFileList.iterator(); iterator.hasNext();)
        {
            final File scriptFile = (File) iterator.next();
            _logger.debug("Executing " + scriptFile.getName());
            final Interpreter interpreter = new Interpreter();
            interpreter.set("taskManager", _taskManager);
            interpreter.set("scheduler", _scheduler);
            final Reader reader = new FileReader(scriptFile);
            try
            {
                interpreter.eval(reader);
                _logger.debug(scriptFile.getName() + " executed successfully");
            }
            catch (EvalError evalError)
            {
                _logger.error(evalError.getMessage(), evalError);
            }
        }
    }

    public void basherEvent(final BasherEvent basherEvent)
    {
        if (basherEvent instanceof PhaseTransitionEvent)
        {
            final PhaseTransitionEvent phaseTransitionEvent = (PhaseTransitionEvent) basherEvent;
            if (phaseTransitionEvent.getNewPhase() == Phase.START)
            {
                try
                {
                    final String beanshellScriptDirectory = phaseTransitionEvent.getBasherContext().getBeanShellScriptDirectory();
                    if (_logger.isDebugEnabled())
                    {
                        _logger.debug("Executing scripts in: " + beanshellScriptDirectory);
                    }
                    executeScripts(new File(beanshellScriptDirectory));
                    if (_logger.isDebugEnabled())
                    {
                        _logger.debug("Scripts execution completed");
                    }
                }
                catch (Exception e)
                {
                    _logger.error(e.getMessage(), e);
                }
            }
        }
    }
}
