/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.basher.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;

import net.sourceforge.basher.BasherContext;
import net.sourceforge.basher.ContextManager;
import static net.sourceforge.basher.Phase.*;
import net.sourceforge.basher.Scheduler;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.events.*;
import net.sourceforge.basher.internal.TaskRunner;
import org.apache.commons.logging.Log;

/**
 * Implementation of the <code>Scheduler</code> interface.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public class SchedulerImpl implements Scheduler, BasherEventListener
{
    private Log _logger;
    private ContextManager _contextManager;
    private EventManager _eventManager;
    private TaskRunner _taskRunner;

    private List<String> _threadNames;
    private ThreadGroup _threadGroup;

    private int _threadCounter = 0;
    private Timer _timer = new Timer();
    private boolean _running = false;

    /**
     * {@inheritDoc}
     */
    public synchronized void addThread()
    {
        checkInitialized();
        addNewThread(_contextManager.getActiveBasherContext());
    }

    /**
     * {@inheritDoc}
     */
    public void addThreads(final int numToAdd)
    {
        checkInitialized();
        _logger.debug("Adding " + numToAdd + " thread(s)");
        for (int i = 0; i < numToAdd; i++)
        {
            addThread();
        }
        _logger.debug(numToAdd + " thread(s) added");
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void removeThread()
    {
        checkInitialized();

        if (_threadNames.size() > 0)
        {
            _logger.debug("Removing thread from active list");
            String threadName = _threadNames.remove(0);
            _logger.debug("Thread " + threadName + " removed from active list");
            _logger.debug("Signalling stop thread for thread named: " + threadName);
            _eventManager.publish(new ThreadRemovedEvent(threadName));
            _logger.debug("Thread stop signalled");
        }
        else
        {
            _logger.warn("No threads to remove");
        }
    }

    /**
     * {@inheritDoc}
     */
    public void removeAllThreads()
    {
        checkInitialized();

        int threadsToRemove = _threadNames.size();
        _logger.debug("Removing " + threadsToRemove + " thread(s)");
        for (int i = 0; i < threadsToRemove; i++)
        {
            removeThread();
        }
        _logger.debug(threadsToRemove + " thread(s) removed");
    }

    /**
     * {@inheritDoc}
     */
    public void stop()
    {
        stopInternal(false);

    }

    private void stopInternal(boolean calledFromEvent)
    {
        // Check that we can stop
        checkStopPrecondition();

        _logger.info("Stopping scheduler");

        if (!calledFromEvent)
        {
            // Signal stop to all interested parties
            _eventManager.publish(new PhaseTransitionEvent(_contextManager.getActiveBasherContext(), RUN, END));
        }

        // Clear the thread group and the list of active runners
        _threadGroup = null;
        _threadNames.clear();

        _timer.cancel();

        // This is required to allow the timer to restart if called to start again
        _timer = new Timer();

        //
        _running = false;

        _logger.info("Scheduler stopped");
    }

    /**
     * {@inheritDoc}
     */
    public void start()
    {

        // Delegate the call
        start(ContextManager.DEFAULT_BASHER_CONTEXT_NAME);

    }


    public void start(final String contextName)
    {
        // Validate the parameters
        if (contextName == null)
        {
            throw new NullPointerException("contextName");
        }

        // Check if we have all dependencies required for start
        checkStartPrecondition();

        // Look up the default context name
        final BasherContext basherContext = lookupBasherContext(contextName);

        // Check that we actually have a context
        if (basherContext == null)
        {
            throw new IllegalArgumentException("The context specified by '" + contextName + "' could not be found");
        }

        // Delegate the call
        start(basherContext);

    }

    public void start(final BasherContext basherContext)
    {
        // Validate the parameters
        if (basherContext == null)
        {
            throw new NullPointerException("basherContext");
        }

        // Check if we have all dependencies required for start
        checkStartPrecondition();

        _logger.info("Starting scheduler with context: " + basherContext.getName());

        // Initial signalling that we are starting up
        _eventManager.publish(new PhaseTransitionEvent(basherContext, null,Phase.START));

        _threadGroup = new ThreadGroup("Basher Threads");

        _threadNames = new ArrayList<String>();

        // Start the threads
        for (int i = 0; i < basherContext.getInitialNumberThreads(); i++)
        {
            addNewThread(basherContext);
        }

        // Clear the timer
        final int numPurged = _timer.purge();
        _logger.debug("Purged " + numPurged + " old timer tasks");

        _contextManager.setActiveBasherContext(basherContext);

        _running = true;

        // Signal interested parties that systems are go
        _eventManager.publish(new PhaseTransitionEvent(basherContext, START, SETUP));

        // Schedule end of setup and start of run
        long nextEventTime = (basherContext.getSetupDuration() + 1) * 1000;

        _logger.debug("Scheduling phase transition SETUP -> RUN at " + nextEventTime);
        _timer.schedule(new EventEmitterTimerTask(_eventManager, new PhaseTransitionEvent(basherContext, SETUP, RUN)), nextEventTime);


        // Schedule cooldown
        nextEventTime = nextEventTime + 1 + (basherContext.getRunDuration() * 1000);
        _logger.debug("Scheduling phase transition RUN -> COOLDOWN at " + nextEventTime);
        _timer.schedule(new EventEmitterTimerTask(_eventManager, new PhaseTransitionEvent(basherContext, RUN, COOLDOWN)), nextEventTime);

        // Schedule end of run
        nextEventTime = nextEventTime + 1 + (basherContext.getCooldownDuration() * 1000);
        _logger.debug("Scheduling phase transition COOLDOWN -> END at " + nextEventTime);
        _timer.schedule(new EventEmitterTimerTask(_eventManager, new PhaseTransitionEvent(basherContext, COOLDOWN, END)), nextEventTime);

        // Schedule collection start
        _timer.schedule(new EventEmitterTimerTask(_eventManager, new CollectionStartedEvent()), (basherContext.getSetupDuration() + 1 + basherContext.getStartCollectionFrom()) * 1000);
        // And stop
        _timer.schedule(new EventEmitterTimerTask(_eventManager, new CollectionStoppedEvent()), ((basherContext.getSetupDuration() + 1 + basherContext.getStopCollectionAfter()) * 1000));

        _logger.info("Scheduler started.  " + basherContext.getInitialNumberThreads() + " thread(s) running");
    }

    private BasherContext lookupBasherContext(final String contextName)
    {
        return _contextManager.getBasherContext(contextName);
    }

    /**
     * {@inheritDoc}
     */
    public int getNumberOfActiveThreads()
    {
        checkInitialized();
        return _threadNames.size();
    }

    /**
     * {@inheritDoc}
     */
    public boolean isRunning()
    {
        return _running;
    }

    /**
     * Adds a new thread to the currently running thread group.  Will only add a new thread if we have not yet reached
     * the maximum number of threads allowed.
     *
     * @param basherContext The context currently used.  This is to determine the various boundaries of the threads.
     */
    private void addNewThread(final BasherContext basherContext)
    {
        // Have we reached the maximum number of threads?
        if (_threadNames.size() < basherContext.getMaxNumberThreads())
        {
            // No, so add a new one
            _logger.debug("Adding new thread");

            // And the new thread to run and start it
            final String threadName = "TaskRunner-" + _threadCounter++;
            final Thread thread = new Thread(_threadGroup, _taskRunner, threadName);
            thread.setDaemon(true);
            thread.start();

            // Add the task runner to our internally managed list
            _threadNames.add(threadName);
            _logger.info("Thread added.  " + _threadNames.size() + " thread(s) running");

            // And signal the start
            _eventManager.publish(new ThreadAddedEvent(threadName, basherContext));
        }
        else
        {
            _logger.warn("Maximum thread limit (" + _contextManager.getActiveBasherContext().getMaxNumberThreads() + ") reached, not adding more threads");
        }
    }

    /**
     * Checks if this <code>SchedulerImpl</code> instance has been started or not.
     *
     * @throws IllegalStateException If the instance is NOT running
     */
    private void checkInitialized()
    {
        if (!_running)
        {
            throw new IllegalStateException("Not started");
        }
    }

    /**
     * Convenience method to check that all pre-conditions for a start are met
     *
     * @throws IllegalStateException If any of the pre-conditions fails
     */
    private void checkStartPrecondition()
    {
        if (_running)
        {
            throw new IllegalStateException("Already started");
        }

        if (_logger == null)
        {
            throw new IllegalStateException("no log");
        }

        if (_contextManager == null)
        {
            throw new IllegalStateException("no context manager");
        }

        if (_eventManager == null)
        {
            throw new IllegalStateException("no event manager");
        }

        if (_taskRunner == null)
        {
            throw new IllegalStateException("no task runner");
        }
    }

    /**
     * Convenience method to check that all pre-conditions for a stop are met
     *
     * @throws IllegalStateException If any of the pre-conditions fails
     */
    private void checkStopPrecondition()
    {
        if (!_running)
        {
            throw new IllegalStateException("Already stopped");
        }
    }


    /** Setters for various fields */

    /**
     * Sets the logging instance to use
     *
     * @param logger The logging instance to use
     */
    public void setLog(final Log logger)
    {
        _logger = logger;
    }

    /**
     * Sets the context manager instance to use
     *
     * @param contextManager The context manager instance to use
     */
    public void setContextManager(final ContextManager contextManager)
    {
        _contextManager = contextManager;
    }

    /**
     * Sets the event manager to use
     *
     * @param eventManager The event manager instance to use
     */
    public void setEventManager(final EventManager eventManager)
    {
        _eventManager = eventManager;
    }

    /**
     * Sets the task runner to use
     *
     * @param taskRunner The task runner instance to use
     */
    public void setTaskRunner(final TaskRunner taskRunner)
    {
        _taskRunner = taskRunner;
    }

    public void setTimer(final Timer timer)
    {
        _timer = timer;
    }

    public void basherEvent(final BasherEvent basherEvent)
    {
        // Phase transition events

        // Ticks should be cleared after run stage
        if (basherEvent instanceof PhaseTransitionEvent)
        {
            final PhaseTransitionEvent phaseTransitionEvent = (PhaseTransitionEvent) basherEvent;
            final BasherContext basherContext = phaseTransitionEvent.getBasherContext();
            switch (phaseTransitionEvent.getNewPhase())
            {
                case RUN:
                    // Schedule the ticker - emitting ticks within the system
                    _timer.scheduleAtFixedRate(new TickTimerTask(_eventManager), basherContext.getMarkAverageInterval() * 1000, basherContext.getMarkAverageInterval() * 1000);
                    break;
                case COOLDOWN:
                    // Unschedule the ticker
                    break;
                case END:
                    stopInternal(true);

            }
        }


    }
}
