package net.sourceforge.basher.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import net.sourceforge.basher.*;
import net.sourceforge.basher.events.*;
import net.sourceforge.basher.internal.TimeSource;
import org.apache.commons.logging.Log;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public abstract class AbstractCollector implements Collector, BasherEventListener
{
    private long _successes;
    private long _failures;
    private long _notRun;

    private long _min = 0;
    private long _max;
    private long _total;
    private long _timeElapsed = System.currentTimeMillis();

    private List<Average> _averages = new ArrayList<Average>();
    private boolean _collecting;

    protected TimeSource _timeSource;
    protected Log _log;

    private int _runningThreads;


    public void startCollecting()
    {
        _collecting = true;
    }

    public void stopCollecting()
    {
        _collecting = false;
    }

    protected abstract void initializeCollector(final BasherContext basherContext) throws Exception;

    public void success(final Task task, final long elapsedTime)
    {
        _successes++;
        updateStats(elapsedTime);
    }

    public void notRun(final Task task, final long elapsedTime)
    {
        _notRun++;
        // We omit stats for not run tasks
    }

    public void fail(final Task task, final long elapsedTime, final Throwable throwable)
    {
        _failures++;
        updateStats(elapsedTime);
    }

    private void updateStats(final long elapsedTime)
    {
        _total += elapsedTime;
        if (elapsedTime < _min)
        {
            _min = elapsedTime;
        }
        if (elapsedTime > _max)
        {
            _max = elapsedTime;
        }
    }

    public long getFailures()
    {
        return _failures;
    }

    public long getSuccesses()
    {
        return _successes;
    }

    public long getNotRun()
    {
        return _notRun;
    }

    public long getTotal()
    {
        return _failures + _successes;
    }

    public List<Average> getAverages()
    {
        return Collections.unmodifiableList(_averages);
    }

    public Average markAverage()
    {
        // FIXME: This should lock the average
        final Average average = new Average(_min, _max, _total, (System.currentTimeMillis() - _timeElapsed), _successes, _failures,_notRun, _runningThreads);
        if (isCollecting())
        {
            _averages.add(average);
        }
        _min = Long.MAX_VALUE;
        _max = 0;
        _total = 0;
        _successes = 0;
        _failures = 0;
        _notRun = 0;
        _timeElapsed = System.currentTimeMillis();

        return average;
    }

    public void setTimeSource(final TimeSource timeSource)
    {
        _timeSource = timeSource;
    }

    public void setLog(final Log log)
    {
        _log = log;
    }

    public boolean isCollecting()
    {
        return _collecting;
    }

    public void basherEvent(final BasherEvent basherEvent)
    {
        if (basherEvent instanceof PhaseTransitionEvent)
        {
            final PhaseTransitionEvent phaseTransitionEvent = (PhaseTransitionEvent) basherEvent;
            final Phase newPhase = phaseTransitionEvent.getNewPhase();
            switch (newPhase)
            {
                case RUN:
                    try
                    {
                        final BasherContext basherContext = phaseTransitionEvent.getBasherContext();
                        initializeCollector(basherContext);
                        _runningThreads = basherContext.getInitialNumberThreads();

                    }
                    catch (Exception e)
                    {
                        _log.error("Unable to initialize collector for run: " + e.getMessage(), e);
                    }

                    break;
                case COOLDOWN:
                case END:
                    stopCollecting();
                    break;
                default:
            }

        }
        else if (basherEvent instanceof CollectionStartedEvent)
        {
            startCollecting();
        }
        else if (basherEvent instanceof CollectionStoppedEvent)
        {
            stopCollecting();
        }
        else if (basherEvent instanceof ThreadAddedEvent)
        {
            _runningThreads++;
        }
        else if (basherEvent instanceof ThreadRemovedEvent)
        {
            _runningThreads--;
        }
        else if (basherEvent instanceof TickEvent)
        {
            Average average = markAverage();
        }
    }
}
