/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher;

import java.util.List;

/**
 * Defines operations for interacting with the task manager.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public interface TaskManager
{
    /**
     * Adds a <code>Task</code> into a running task manager.  This will include the specified task
     * in the list of available tasks.
     *
     * @param task The task to add to the task manager.
     */
    public void addTask(Task task);

    /** Adds a task class to the task manager.  The class does not necessarily have to implement <code>Task<code> to the
     * task manager but <strong>must</strong> provide an empty constructor (but not necessarily public).
     *
     * @param taskClassName The task class name to add.  This class must <strong>either<strong> be annotated <strong>or</strong>
     * expose a public method named <code>executeTask</code> which will be used for task execution.  If not annotated,
     * the class <strong>may</strong> also expose any of the methods defined in <code>Task</code> to control the settings
     * used during invocation.
     * @param followers Followers which will be added to the task instance.
     * @since 1.3
     * @return The task created from adding the specified task instance
     */
    public Task addTaskClass(Class taskClassName, Task... followers);

    /** Adds a task instance.  The instance does not necessarily have to implement <code>Task<code> to the
     * task manager.
     *
     * @param taskInstance The task instance to add.  The instance must <strong>either<strong> be annotated <strong>or</strong>
     * expose a public method named <code>executeTask</code> which will be used for task execution
     * @param followers Followers which will be added to the task instance.
     * @since 1.3
     * @return The task created from adding the specified task instance
     */
    public Task addTaskInstance(Object taskInstance, Task... followers);

    /**
     * Retrieves the next, randomly choosen task available.
     *
     * @return A task instance or null of no tasks are available
     */
    public Task getNextRandomTask();

    /**
     * Retrieves all active tasks with the task manager.
     *
     * @return A list of <code>Task</code> instances.
     */
    public List<Task> getTasks();

    /**
     * Retrieves all tasks registered with the task manager.
     *
     * @return A list of <code>Task</code> instances.
     * @since 1.3
     */
    public List<Task> getRegisteredTasks();

    /**
     * Retrieves the task identified by the specified name.  This will search both active and removed task lists.
     *
     * @param name The name of the task to retrieve.
     * @return The specified task or null if it can not be found
     */
    public Task getTaskByName(String name);

    /**
     * Retrives the count of the currently registered tasks.
     *
     * @return The number of tasks currently registered.
     */
    public int getNumberOfTasks();

    /**
     * Removes the specified task from the list of active tasks.
     *
     * @param task The task to remove
     */
    public void removeTask(Task task);

    /**
     * Retrieves the list of removed tasks
     *
     * @return The list of <code>Task</code> instances that have been removed
     */
    public List<Task> getRemovedTasks();

    /**
     * Returns the number of tasks that have been removed.
     *
     * @return The number of tasks that have been removed.
     */
    public int getNumberOfRemovedTasks();
}
