/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher;

import java.util.Collection;

/**
 * Simple registry where tasks may store objects to be used during runtime.
 * <b>Warning</b>:  This is currenly backed by an in-memory store, which means that heavy use of this may cause
 * a degradation in performance over time.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public interface EntityRegistry
{
    /**
     * Register a certain type as having been created.  The <code>entityIdentifier</code> is a key into a collection
     * of <code>entity</code> instances being held by the <code>EntityRegistry</code>.
     *
     * @param entityIdentifier The type of the entity being registered
     * @param entity           The entity being registered
     */
    public void register(String entityIdentifier, Object entity);

    /**
     * Retrieves all registered <code>entity</code> instances for a particular <code>entityIdentifier</code>.
     *
     * @param entityIdentifier The entity type to retrive the entity instances for
     * @return A collection of <code>entity</code>.
     */
    public Collection getAllRegistered(String entityIdentifier);

    /**
     * Retrieves the number of <code>entity</code> instances registered for the specified <code>entityIdentifier</code>.
     *
     * @param entityIdentifier The entity type to count
     * @return The number of instances registered for the specified <code>entityIdentifier</code>.
     */
    public long getNumRegistered(String entityIdentifier);

    /**
     * Retrieves a randomly sized collection of <code>entity</code> instances registered for the specified
     * <code>entityIdentifier</code>.
     *
     * @param entityIdentifier The entity type to generate the random selection from
     * @param maxSize          The maximum number of entity instances in the collection.
     * @return A collection of <code>entity</code> instances.  The size of this collection may be less than
     *         the maxSize specified if there are less entity instances registered than is requested.
     */
    public Collection getRandomSelection(String entityIdentifier, int maxSize);

    /**
     * Retrieves a random <code>entity</code> instance registered for the specified <code>entityIdentifier</code>.
     *
     * @param entityIdentifier The entity type to retrieve the instance for.
     * @return The entity instance or null if none could be found.
     */
    public Object getRandom(String entityIdentifier);

    /**
     * Removes the specified entity from the registry.
     *
     * @param entityIdentifier The entity type to remove.
     * @param entity           The entity to remove
     */
    public void unregister(String entityIdentifier, Object entity);
}
