/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher.internal.impl;

import java.util.Iterator;

import net.sourceforge.basher.Collector;
import net.sourceforge.basher.Task;
import net.sourceforge.basher.TaskManager;
import net.sourceforge.basher.TaskNotRunException;
import net.sourceforge.basher.internal.Randomizer;
import net.sourceforge.basher.internal.TaskInvoker;
import net.sourceforge.basher.internal.TimeSource;
import org.apache.commons.logging.Log;

/**
 * @author Johan Lindquist
 * @version 1.0
 */
public class TaskInvokerImpl implements TaskInvoker
{
    private Randomizer _randomizer;
    private Collector _collector;
    private Log _logger;
    private TaskManager _taskManager;
    private TimeSource _timeSource;

    public void setTimeSource(final TimeSource timeSource)
    {
        _timeSource = timeSource;
    }

    public void setLog(final Log log)
    {
        _logger = log;
    }

    public void setRandomizer(final Randomizer randomizer)
    {
        _randomizer = randomizer;
    }

    public void setCollector(final Collector collector)
    {
        _collector = collector;
    }

    public void setTaskManager(final TaskManager taskManager)
    {
        _taskManager = taskManager;
    }

    public void invokeTask(final Task task)
    {
        doInvokeTask(task, true);
    }

    public void invokeTask(final Task task, final boolean invokeFollowers)
    {
        doInvokeTask(task, invokeFollowers);
    }

    private void doInvokeTask(final Task task, final boolean invokeFollower)
    {
        final int invocations = task.getInvocations();

        if (task.getMaxInvocations() != 0 && invocations >= task.getMaxInvocations())
        {
            _logger.debug("Not running task: " + task.getName() + " (max invocations reached)");
            _taskManager.removeTask(task);
            return;
        }

        final long currentTime = _timeSource.getElapsedTime() / 1000;

        if (currentTime < task.getRunFrom() || currentTime > task.getStopAfter())
        {
            _logger.debug("Not running task " + task.getName() + " (not within time boundary " + currentTime + "/" + task.getRunFrom() + "/" + task.getStopAfter() + ")");
            _collector.notRun(task, 0);
            return;
        }

        final int weigher = _randomizer.getRandomInt(100);
        // We only run the task if the weigher is above the tasks threshold
        final int weight = task.getWeight();

        if (weigher < weight)
        {
            // Execute the task
            final long startTime = _timeSource.getCurrentTime();
            try
            {
                _logger.debug("Invoking task: " + task.getName());
                task.executeTask();
                // Should we invoke the subtasks?
                if (invokeFollower && task.getFollowers().size() != 0)
                {
                    _logger.debug("Invoking followers");
                    for (Iterator iterator = task.getFollowers().iterator(); iterator.hasNext();)
                    {
                        final Task follower = (Task) iterator.next();
                        invokeTask(follower, false);
                    }
                }
                _collector.success(task, _timeSource.getCurrentTime() - startTime);
            }
            catch (TaskNotRunException e)
            {
                // collect it!
                _logger.debug("Task not run: " + task.getName() + " (" + e.getMessage() + ")");
                _collector.notRun(task, _timeSource.getCurrentTime() - startTime);
            }
            catch (Throwable throwable)
            {
                // tell someone?
                _logger.debug("Task failed: " + task.getName(), throwable);
                _collector.fail(task, _timeSource.getCurrentTime() - startTime, throwable);
            }
        }
        else
        {
            _logger.debug("Not running task: " + task.getName() + " (weigth: " + weight + " / weigher: " + weigher + ")");
            _collector.notRun(task, 0);
        }
    }
}
