package net.sourceforge.basher.internal.impl;

import java.lang.reflect.Method;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.annotation.Annotation;

import net.sourceforge.basher.Task;
import net.sourceforge.basher.BasherException;
import net.sourceforge.basher.Phase;
import net.sourceforge.basher.annotations.*;
import net.sourceforge.basher.internal.TaskDecorator;
import org.ops4j.gaderian.service.ClassFactory;
import org.ops4j.gaderian.service.ClassFab;
import org.ops4j.gaderian.service.MethodSignature;
import org.ops4j.gaderian.service.BodyBuilder;
import org.ops4j.gaderian.util.PropertyAdaptor;
import org.apache.commons.logging.Log;

/**
 * @author Johan Lindquist
 * @version $Revision$
 */
public class TaskDecoratorImpl implements TaskDecorator
{
    private ClassFactory _classFactory;
    private Log _log;

    public void setLog(final Log log)
    {
        _log = log;
    }

    public void setClassFactory(final ClassFactory classFactory)
    {
        _classFactory = classFactory;
    }

    public Task decorateInstance(final Object taskInstance)
    {
        try
        {
            final String taskInstanceClassName = taskInstance.getClass().getName();
            final String className = taskInstanceClassName.substring(taskInstanceClassName.lastIndexOf('.'));

            final String taskClassName = className + "BasherDecoratedTask";

            // Determine the execution method name
            final String executionMethodName = determineExecutionMethodName(taskInstance);

            // Then create the decorated task
            final DecoratedTask decoratedTask = createDecoratedTask(taskClassName, executionMethodName, taskInstance);

            // Pass the instance to check for annotations and initialize the task
            initializeTask(taskInstance, decoratedTask);

            return decoratedTask;
        }
        catch (Exception e)
        {
            throw new BasherException(e.getMessage(), e);
        }
    }

    private void initializeTask(final Object taskInstance, final DecoratedTask decoratedTask)
    {
        final Class<? extends Object> taskInstanceClass = taskInstance.getClass();

        // First initialize from annotations
        processAnnotations(taskInstanceClass, decoratedTask);

        // Autowire any task dependencies

        

    }

    private void processAnnotations(final Class<? extends Object> taskInstanceClass, final DecoratedTask decoratedTask)
    {
        // First examine super classes for annotations
        // By doing depth-first here, we allow sub-classes to override super classes annotations
        final Class superClass = taskInstanceClass.getSuperclass();
        if (!superClass.equals(Object.class))
        {
            processAnnotations(superClass, decoratedTask);
        }
        if (taskInstanceClass.isAnnotationPresent(BasherMaxTime.class))
        {
            BasherMaxTime annotation = taskInstanceClass.getAnnotation(BasherMaxTime.class);
            decoratedTask.setMaxTime(annotation.time());
        }
        if (taskInstanceClass.isAnnotationPresent(BasherMaxInvocations.class))
        {
            BasherMaxInvocations annotation = taskInstanceClass.getAnnotation(BasherMaxInvocations.class);
            decoratedTask.setMaxInvocations(annotation.max());
        }
        if (taskInstanceClass.isAnnotationPresent(BasherInertia.class))
        {
            BasherInertia annotation = taskInstanceClass.getAnnotation(BasherInertia.class);
            decoratedTask.setInertia(annotation.inertia());
        }
        if (taskInstanceClass.isAnnotationPresent(BasherName.class))
        {
            BasherName annotation = taskInstanceClass.getAnnotation(BasherName.class);
            decoratedTask.setName(((BasherName) annotation).name());
        }
        if (taskInstanceClass.isAnnotationPresent(BasherStopAfter.class))
        {
            BasherStopAfter annotation = taskInstanceClass.getAnnotation(BasherStopAfter.class);
            decoratedTask.setStopAfter(((BasherStopAfter) annotation).time());
        }
        if (taskInstanceClass.isAnnotationPresent(BasherWeight.class))
        {
            BasherWeight annotation = taskInstanceClass.getAnnotation(BasherWeight.class);
            decoratedTask.setWeight(((BasherWeight) annotation).weight());
        }
        if (taskInstanceClass.isAnnotationPresent(BasherRunFrom.class))
        {
            BasherRunFrom annotation = taskInstanceClass.getAnnotation(BasherRunFrom.class);
            decoratedTask.setRunFrom(((BasherRunFrom) annotation).time());
        }

        // Process the NOT phase annotations first, since the phase annotations overrides the NOTs
        if (taskInstanceClass.isAnnotationPresent(BasherNotPhaseRun.class))
        {
            decoratedTask.removePhase(Phase.RUN);
        }
        if (taskInstanceClass.isAnnotationPresent(BasherNotPhaseSetup.class))
        {
            decoratedTask.removePhase(Phase.SETUP);
        }
        if (taskInstanceClass.isAnnotationPresent(BasherNotPhaseCooldown.class))
        {
            decoratedTask.removePhase(Phase.COOLDOWN);
        }

        if (taskInstanceClass.isAnnotationPresent(BasherPhaseRun.class))
        {
            decoratedTask.addPhase(Phase.RUN);
        }
        if (taskInstanceClass.isAnnotationPresent(BasherPhaseSetup.class))
        {
            decoratedTask.addPhase(Phase.SETUP);
        }
        if (taskInstanceClass.isAnnotationPresent(BasherPhaseCooldown.class))
        {
            decoratedTask.addPhase(Phase.COOLDOWN);
        }

        if (decoratedTask.applicablePhases().size() == 0)
        {
            _log.warn("No applicable phase specified for task");
        }

    }

    private DecoratedTask createDecoratedTask(final String taskClassName, final String executionMethodName, final Object taskInstance) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException, InstantiationException
    {
        final ClassFab fab = _classFactory.newClass(taskClassName, DecoratedTask.class);
        fab.addField("_taskInstance", taskInstance.getClass());

        final BodyBuilder bodyBuilder = new BodyBuilder();
        bodyBuilder.begin();
        bodyBuilder.addln("_taskInstance." + executionMethodName + "();");
        bodyBuilder.end();

        fab.addMethod(java.lang.reflect.Modifier.PUBLIC, new MethodSignature(void.class, "exectuteTask", new Class[0], new Class[]{Throwable.class}), bodyBuilder.toString());

        bodyBuilder.clear();

        bodyBuilder.begin();
        bodyBuilder.addln("super();");
        bodyBuilder.addln("_taskInstance = $1;");
        bodyBuilder.end();

        fab.addConstructor(new Class[]{taskInstance.getClass()}, new Class[0], bodyBuilder.toString());

        final Class decoratedTask = fab.createClass();

        final Constructor constructor = decoratedTask.getConstructor(taskInstance.getClass());

        return (DecoratedTask) constructor.newInstance(taskInstance);

    }

    String determineExecutionMethodName(final Object taskInstance)
    {
        final Method[] methods = taskInstance.getClass().getMethods();

        Method selectedMethod = null;

        for (final Method method : methods)
        {
            if (method.getAnnotation(BasherExecuteMethod.class) != null)
            {
                if (selectedMethod == null)
                {
                    // Found an execution method
                    selectedMethod = method;
                }
                else
                {
                    throw new BasherException("Found more than 1 execute method", null);
                }
            }
        }

        if (selectedMethod != null)
        {
            validateExecutionMethod(selectedMethod);
            return selectedMethod.getName();
        }

        // Ok, look for the default method.
        try
        {
            taskInstance.getClass().getMethod("executeTask");
            return "executeTask";
        }
        catch (NoSuchMethodException e)
        {
            throw new BasherException("Could not find executeTask", e);
        }

    }

    private void validateExecutionMethod(Method method)
    {
        if (method.getParameterTypes().length != 0)
        {
            throw new BasherException("Execution method '" + method.getName() + "' requires parameters", null);
        }

    }
}
