/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.sourceforge.basher;

import java.util.List;

/**
 * Defines a task that is to run within the Basher framework.  This task defines operations suitable for a scheduler
 * to determine whether or not the task should run as well as some operations for retrieving statistics about the
 * task itself.<br/>
 * <br/>
 * Implementations of this interface are encouraged to maintain the tasks small and to the point.  Overall, a collection
 * of small tasks should emulate a complete system more accurately.
 *
 * @author Johan Lindquist
 * @version 1.0
 */
public interface Task
{
    /**
     * Main method called by a component to execute this <code>Task</code>.  This should ensure that
     * counters of failures/successes are kept up to date.
     *
     * @throws Throwable The re-thrown exception thrown by the task
     */
    public void executeTask() throws Throwable;

    /**
     * Retrives the name of this <code>Task</code> instance.  Mainly used for display and tracking purposes.
     *
     * @return The name of this <code>Task</code>.
     */
    public String getName();

    /**
     * Retrieves the maximum time this task should take to execute.  If the task takes longer, this should be considered
     * an error and should be reported as such.  Returning -1 indicates no maximum time.
     *
     * @return The maximum time the task should take to execute (in seconds).
     */
    public int getMaxTime();

    /**
     * Retrieves the weight of this <code>Task</code>.  This could be used by certain schedulers to allow for
     * more fine grained control of task scheduling.
     *
     * @return A value between 1 and 100 (higher signals a wish to run more frequently)
     */
    public int getWeight();

    /**
     * Retrieves the number of times this task was invoked (successes+failures).
     *
     * @return The number of invocations.
     */
    public int getInvocations();

    /**
     * Retrieves the number of failures of this task.
     *
     * @return The nunber of failures
     */
    public int getFailures();

    /**
     * Retrieves the number of successes of this task.
     *
     * @return The number of successes.
     */
    public int getSuccesses();

    /**
     * Retrieves the number of not run of this task.
     *
     * @return The number of times this task didn't run.
     */
    public int getNotRun();

    /**
     * Retrieves the inertia of this <code>Task</code>.
     * See {@link net.sourceforge.basher.Task#setInertia(float)}}.
     *
     * @return The rate of change for this instance.
     */
    public float getInertia();

    /**
     * Sets the inertia of this <code>Task</code>.  This is used to control the change of
     * the weight assocaited with this task, calculated for each invocation of the task (if it is
     * not 1.0).
     *
     * @param inertia The inertia to give this instance.  This should be a value between 0.1 and 2.0.
     */
    public void setInertia(float inertia);

    /**
     * Sets the weight for this task.  This <i>could</i> be used by a scheduler to control the invocations of this
     * <code>Task</code>.
     *
     * @param weight The weight of this task.  Should be a value between 0 (task will not run) and 100 (always run).
     */
    public void setWeight(int weight);

    /**
     * Sets the maximum number of invocations that this task should do.
     *
     * @param maxInvocations The number of invocations to do.  If this is less than or equal to 0 (zero), the task
     *                       will always be invoked.
     */
    public void setMaxInvocations(int maxInvocations);

    /**
     * Retrieves the maximum number of invocations that this task should do.  The number returned <i>could</i> be used
     * by a scheduler to control the invocation of this <code>Task</code> instance.
     *
     * @return The maximum number of invocations
     */
    public int getMaxInvocations();

    /**
     * Retrives a list of <code>Task</code> instances that logically may follow the successful execution of this
     * <code>Task</code>.  These <i>could</i> be used by the scheduler to run series of tasks.
     *
     * @return An unmodifiable list of <code>Task</code> instances.
     */
    public List<Task> getFollowers();

    /**
     * Sets the list of <code>Task</code> instances that may logically follow this <code>Task</code>.  See
     * {@link net.sourceforge.basher.Task#getFollowers()} for more information.
     *
     * @param followers The list of <code>Task</code> instances.
     */
    public void setFollowers(List<Task> followers);

    /**
     * Specifies the point in time (basher internal run time) from which this <code>Task</code> should
     * start to execute.  Specified in seconds since basher startup.
     *
     * @return The point in time from which this task should start to execute (time in seconds)
     */
    public long getRunFrom();

    /**
     * Specifies the point in time (basher internal run time) from which this <code>Task</code> should
     * stop to execute.  Specified in seconds since basher startup
     *
     * @return The point in time from which this task should stop executing (time in seconds).  A value of
     *         0 specifies run-forever.
     */
    public long getStopAfter();

    /** Resets the tasks invocation counters.
     */
    public void reset();

    /**
     * Sets the point in time (basher internal run time) from which this <code>Task</code> should
     * start to execute.  Specified in seconds since basher startup.
     *
     * @param runFrom The point in time from which this task should start to execute (time in seconds)
     */
    public void setRunFrom(long runFrom);

    /**
     * Sets the point in time (basher internal run time) from which this <code>Task</code> should
     * stop to execute.  Specified in seconds since basher startup
     *
     * @param stopAfter The point in time from which this task should stop executing (time in seconds).  A value of
     *         0 specifies run-forever.
     */
    public void setStopAfter(long stopAfter);

    /** Returns the list of applicable phases in which this <code>Task</code> should be run.
     *
     * @return List of phases to run the task in.
     */
    public List<Phase> applicablePhases();
}
