package net.sourceforge.basher.booter;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;

import net.sourceforge.basher.BasherContext;
import net.sourceforge.basher.internal.impl.BasherBootHelper;
import org.apache.maven.surefire.booter.IsolatedClassLoader;
import org.apache.maven.surefire.booter.SurefireBooterForkException;
import org.apache.maven.surefire.booter.SurefireExecutionException;
import org.apache.maven.surefire.booter.output.*;
import org.apache.maven.surefire.booter.shade.org.codehaus.plexus.util.IOUtil;
import org.apache.maven.surefire.booter.shade.org.codehaus.plexus.util.StringUtils;
import org.apache.maven.surefire.booter.shade.org.codehaus.plexus.util.cli.CommandLineException;
import org.apache.maven.surefire.booter.shade.org.codehaus.plexus.util.cli.CommandLineUtils;
import org.apache.maven.surefire.booter.shade.org.codehaus.plexus.util.cli.Commandline;
import org.apache.maven.surefire.booter.shade.org.codehaus.plexus.util.cli.StreamConsumer;
import org.apache.maven.surefire.util.NestedRuntimeException;
import org.apache.maven.surefire.util.UrlUtils;

/** Helper class to help boot Basher in a clean JVM.  The original functionality of this plugin was copied from the
 * Maven SureFire plugin.
 *
 * @author Johan Lindquist
 */
public class BasherBooter
{
    private static final String REPORT_PROPERTY_PREFIX = "report.";
    private static final String PARAMS_SUFFIX = ".params";
    private static final String TYPES_SUFFIX = ".types";

    private List _reports = new ArrayList();

    private List<String> _classPathUrls = new ArrayList<String>();

    private List<String> _basherClassPathUrls = new ArrayList<String>();

    private List<String> _basherBootClassPathUrls = new ArrayList<String>();

    private boolean _failIfNoTasks = false;

    private int _forkedProcessTimeoutInSeconds = 0;

    private boolean _redirectTasksOutputToFile = false;

    private BasherForkConfiguration _forkConfiguration;

    public static final int BASHER_SUCCEEDED_EXIT_CODE = 0;

    public static final int BASHER_FAILED_EXIT_CODE = 255;

    public static final int NO_TASKS_EXIT_CODE = 254;

    private static Method _assertionStatusMethod;

    private File _reportsDirectory;

    /**
     * This field is set to true if it's running from main. It's used to help decide what classloader to use.
     */
    private final boolean _isForked;

    /**
     * Whether to enable assertions or not (can be affected by the fork arguments, and the ability to do so based on the
     * JVM).
     */
    private boolean _enableAssertions;
    private List<BasherContext> _basherContexts = new ArrayList<BasherContext>();
    private String _activeBasherContext;

    static
    {
        try
        {
            _assertionStatusMethod =
                    ClassLoader.class.getMethod("setDefaultAssertionStatus", boolean.class);
        }
        catch (NoSuchMethodException e)
        {
            _assertionStatusMethod = null;
        }
    }

    public BasherBooter()
    {
        _isForked = false;
    }

    private BasherBooter(final boolean isForked)
    {
        _isForked = isForked;
    }

    /** Executes the Basher run.
     *
     * @return 0 if successful, otherwise non-zero.
     * @throws SurefireBooterForkException
     * @throws SurefireExecutionException
     */
    public int run()
            throws SurefireBooterForkException, SurefireExecutionException
    {
        int globalResult = 0;


        final int result = forkBasherRun();
        if (result > globalResult)
        {
            globalResult = result;
        }

        return globalResult;
    }

    private int runBasherInProcess(final String activeBasherContext, final Properties results)
            throws SurefireExecutionException
    {
        // noinspection CatchGenericClass,OverlyBroadCatchBlock
        final ClassLoader oldContextClassLoader = Thread.currentThread().getContextClassLoader();
        try
        {
            final ClassLoader tasksClassLoader = useSystemClassLoader() ? ClassLoader.getSystemClassLoader() : createClassLoader(_classPathUrls, null);

            final ClassLoader basherClassLoader = createClassLoader(_basherClassPathUrls, tasksClassLoader);

            final Class basherBootHelperClass = basherClassLoader.loadClass(BasherBootHelper.class.getName());

            final Object basherBootHelper = basherBootHelperClass.newInstance();

            final Method run =
                    basherBootHelperClass.getMethod("run", String.class,Properties.class, Boolean.class);

            Thread.currentThread().setContextClassLoader(tasksClassLoader);

            final Integer result =
                    (Integer) run.invoke(basherBootHelper, activeBasherContext,results, new Boolean(_failIfNoTasks));

            return result.intValue();
        }
        catch (InvocationTargetException e)
        {
            throw new SurefireExecutionException(e.getTargetException().getMessage(), e.getTargetException());
        }
        catch (Exception e)
        {
            throw new SurefireExecutionException("Unable to instantiate and execute Basher", e);
        }
        finally
        {
            Thread.currentThread().setContextClassLoader(oldContextClassLoader);
        }
    }


    private int forkBasherRun()
            throws SurefireBooterForkException
    {
        final Properties properties = new Properties();
        setForkProperties(properties);

        return fork(properties, true, true);
    }

    private void setForkProperties(final Properties properties)
    {
        addPropertiesForTypeHolder(_reports, properties, REPORT_PROPERTY_PREFIX);
        serializeBasherContexts(properties);

        for (int i = 0; i < _classPathUrls.size(); i++)
        {
            final String url = _classPathUrls.get(i);
            properties.setProperty("classPathUrl." + i, url);
        }

        for (int i = 0; i < _basherClassPathUrls.size(); i++)
        {
            final String url = _basherClassPathUrls.get(i);
            properties.setProperty("basherClassPathUrl." + i, url);
        }

        properties.setProperty("reportingDirectory", _reportsDirectory.getAbsolutePath());
        if (_activeBasherContext != null)
        {
            properties.setProperty("activeBasherContext", _activeBasherContext);
        }
        properties.setProperty("enableAssertions", String.valueOf(_enableAssertions));
        properties.setProperty("useSystemClassLoader", String.valueOf(useSystemClassLoader()));
        properties.setProperty("useManifestOnlyJar", String.valueOf(useManifestOnlyJar()));
        properties.setProperty("failIfNoTasks", String.valueOf(_failIfNoTasks));
    }

    private File writePropertiesFile(final String name, final Properties properties)
            throws IOException
    {
        final File file = File.createTempFile(name, "tmp");
        if (!_forkConfiguration.isDebug())
        {
            file.deleteOnExit();
        }

        writePropertiesFile(file, name, properties);

        return file;
    }

    private void writePropertiesFile(final File file, final String name, final Properties properties)
            throws IOException
    {
        final FileOutputStream out = new FileOutputStream(file);

        try
        {
            properties.store(out, name);
        }
        finally
        {
            IOUtil.close(out);
        }
    }

    public void serializeBasherContexts(final Properties properties)
    {
        for (final BasherContext basherContext : _basherContexts)
        {
            final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
            final XMLEncoder xmlEncoder = new XMLEncoder(new BufferedOutputStream(byteArrayOutputStream));
            xmlEncoder.writeObject(basherContext);
            xmlEncoder.flush();
            final String basherContextSerialized = new String(byteArrayOutputStream.toByteArray());
            properties.put("basherContext." + basherContext.getName(),basherContextSerialized);
        }
    }

    private static List<BasherContext> deserializeBasherContexts(final Properties properties)
    {
        final List<BasherContext> basherContexts = new ArrayList<BasherContext>();

        for (final Object o : properties.keySet())
        {
            final String key = (String) o;

            if (key.startsWith("basherContext"))
            {
                final XMLDecoder xmlDecoder = new XMLDecoder(new ByteArrayInputStream(properties.getProperty(key).getBytes()));
                final BasherContext basherContext = (BasherContext) xmlDecoder.readObject();

                if (basherContext.getReportDirectory() == null)
                {
                    // FIXME: Validate this
                    basherContext.setReportDirectory(properties.getProperty("reportingDirectory"));
                }


                basherContexts.add(basherContext);
            }

        }
        return basherContexts;
    }

    private void addPropertiesForTypeHolder(final List typeHolderList, final Properties properties, final String propertyPrefix)
    {
        for (int i = 0; i < typeHolderList.size(); i++)
        {
            final Object[] report = (Object[]) typeHolderList.get(i);

            final String className = (String) report[0];
            final Object[] params = (Object[]) report[1];

            properties.setProperty(propertyPrefix + i, className);

            if (params != null)
            {
                String paramProperty = convert(params[0]);
                String typeProperty = params[0].getClass().getName();
                for (int j = 1; j < params.length; j++)
                {
                    paramProperty += "|";
                    typeProperty += "|";
                    if (params[j] != null)
                    {
                        paramProperty += convert(params[j]);
                        typeProperty += params[j].getClass().getName();
                    }
                }
                properties.setProperty(propertyPrefix + i + PARAMS_SUFFIX, paramProperty);
                properties.setProperty(propertyPrefix + i + TYPES_SUFFIX, typeProperty);
            }
        }
    }

    private static String convert(final Object param)
    {
        if (param instanceof File[])
        {
            final File[] files = (File[]) param;
            return "[" + StringUtils.join(files, ",") + "]";
        }
        else if (param instanceof Properties)
        {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            try
            {
                ((Properties) param).store(baos, "");
                return new String(baos.toByteArray(), "8859_1");
            }
            catch (Exception e)
            {
                throw new RuntimeException("bug in property conversion", e);
            }
        }
        else
        {
            return param.toString();
        }
    }

    private int fork(final Properties properties, final boolean showHeading, final boolean showFooter)
            throws SurefireBooterForkException
    {
        final File basherProperties;
        File systemProperties = null;
        try
        {
            basherProperties = writePropertiesFile("basher", properties);
            if (_forkConfiguration.getSystemProperties() != null)
            {
                systemProperties = writePropertiesFile("basher", _forkConfiguration.getSystemProperties());
            }
        }
        catch (IOException e)
        {
            throw new SurefireBooterForkException("Error creating properties files for forking", e);
        }

        final List<String> bootClasspath = new ArrayList<String>(_basherBootClassPathUrls.size() + _classPathUrls.size());

        bootClasspath.addAll(_basherBootClassPathUrls);

        if (useSystemClassLoader())
        {
            bootClasspath.addAll(_classPathUrls);
        }

        final Commandline cli = _forkConfiguration.createCommandLine(bootClasspath, useManifestOnlyJar());

        cli.createArg().setFile(basherProperties);

        if (systemProperties != null)
        {
            cli.createArg().setFile(systemProperties);
        }


        final ForkingStreamConsumer out = getForkingStreamConsumer(showHeading, showFooter, _redirectTasksOutputToFile);

        final StreamConsumer err;

        if (_redirectTasksOutputToFile)
        {
            err = out;
        }
        else
        {
            err = getForkingStreamConsumer(showHeading, showFooter, _redirectTasksOutputToFile);
        }

        if (_forkConfiguration.isDebug())
        {
            System.out.println("Forking command line: " + cli);
        }

        final int returnCode;

        try
        {
            returnCode = CommandLineUtils.executeCommandLine(cli, out, err, _forkedProcessTimeoutInSeconds);
        }
        catch (CommandLineException e)
        {
            throw new SurefireBooterForkException("Error while executing forked tasks.", e);
        }

        if (_redirectTasksOutputToFile)
        {
            // ensure the FileOutputConsumerProxy flushes/closes the output file
            try
            {
                out.getOutputConsumer().testSetCompleted();
            }
            catch (Exception e)
            {
                // the FileOutputConsumerProxy might throw an IllegalStateException but that's not of interest now
            }
        }

        if (basherProperties != null && basherProperties.exists())
        {
            FileInputStream inStream = null;
            try
            {
                inStream = new FileInputStream(basherProperties);

                properties.load(inStream);
            }
            catch (FileNotFoundException e)
            {
                throw new SurefireBooterForkException("Unable to reload properties file from forked process", e);
            }
            catch (IOException e)
            {
                throw new SurefireBooterForkException("Unable to reload properties file from forked process", e);
            }
            finally
            {
                IOUtil.close(inStream);
            }
        }

        return returnCode;
    }

    private ClassLoader createClassLoader(final List<String> classPathUrls, final ClassLoader parent)
            throws MalformedURLException
    {
        return createClassLoader(classPathUrls, parent, false);
    }

    private ClassLoader createClassLoader(final List<String> classPathUrls, final ClassLoader parent, final boolean childDelegation)
            throws MalformedURLException
    {
        final List<URL> urls = new ArrayList<URL>();

        for (Iterator i = classPathUrls.iterator(); i.hasNext();)
        {
            final String url = (String) i.next();

            if (url != null)
            {
                final File f = new File(url);
                urls.add(UrlUtils.getURL(f));
            }
        }

        final IsolatedClassLoader classLoader = new IsolatedClassLoader(parent, childDelegation);
        if (_assertionStatusMethod != null)
        {
            try
            {
                if (parent != null)
                {
                    _assertionStatusMethod.invoke(parent, _enableAssertions ? Boolean.TRUE : Boolean.FALSE);
                }
                _assertionStatusMethod.invoke(classLoader, _enableAssertions ? Boolean.TRUE : Boolean.FALSE);
            }
            catch (IllegalAccessException e)
            {
                throw new NestedRuntimeException("Unable to access the assertion enablement method", e);
            }
            catch (InvocationTargetException e)
            {
                throw new NestedRuntimeException("Unable to invoke the assertion enablement method", e);
            }
        }
        for (Iterator iter = urls.iterator(); iter.hasNext();)
        {
            final URL url = (URL) iter.next();
            classLoader.addURL(url);
        }
        return classLoader;
    }

    private static List processStringList(final String stringList)
    {
        String sl = stringList;

        if (sl.startsWith("[") && sl.endsWith("]"))
        {
            sl = sl.substring(1, sl.length() - 1);
        }

        final List<String> list = new ArrayList<String>();

        final String[] stringArray = StringUtils.split(sl, ",");

        for (int i = 0; i < stringArray.length; i++)
        {
            list.add(stringArray[i].trim());
        }
        return list;
    }

    private static Properties loadProperties(final File file)
            throws IOException
    {
        final Properties p = new Properties();

        if (file != null && file.exists())
        {
            final FileInputStream inStream = new FileInputStream(file);
            try
            {
                p.load(inStream);
            }
            finally
            {
                IOUtil.close(inStream);
            }
        }

        return p;
    }

    /**
     * This method is invoked when Surefire is forked - this method parses and organizes the arguments passed to it and
     * then calls the Surefire class' run method. <p/> The system exit code will be 1 if an exception is thrown.
     *
     * @param args
     */
    public static void main(final String[] args)
            throws Throwable
    {
        // noinspection CatchGenericClass,OverlyBroadCatchBlock
        try
        {
            if (args.length > 1)
            {
                setSystemProperties(new File(args[1]));
            }

            final File basherPropertiesFile = new File(args[0]);
            final Properties p = loadProperties(basherPropertiesFile);

            final SortedMap<Integer, String> classPathUrls = new TreeMap<Integer, String>();

            final SortedMap<Integer, String> basherClassPathUrls = new TreeMap<Integer, String>();

            final BasherBooter basherBooter = new BasherBooter(true);

            final BasherForkConfiguration basherForkConfiguration = new BasherForkConfiguration();

            // FIXME: Should use fork mode configuration setting
            basherForkConfiguration.setForkMode("never");
            basherBooter.setForkConfiguration(basherForkConfiguration);

            for (Enumeration e = p.propertyNames(); e.hasMoreElements();)
            {
                final String name = (String) e.nextElement();

                if (name.startsWith(REPORT_PROPERTY_PREFIX) && !name.endsWith(PARAMS_SUFFIX) && !name.endsWith(TYPES_SUFFIX))
                {
                    final String className = p.getProperty(name);

                    final String params = p.getProperty(name + PARAMS_SUFFIX);
                    final String types = p.getProperty(name + TYPES_SUFFIX);
                    basherBooter.addReport(className, constructParamObjects(params, types));
                }
                else if (name.startsWith("classPathUrl."))
                {
                    classPathUrls.put(Integer.valueOf(name.substring(name.indexOf('.') + 1)), p.getProperty(name));
                }
                else if (name.startsWith("basherClassPathUrl."))
                {
                    basherClassPathUrls.put(Integer.valueOf(name.substring(name.indexOf('.') + 1)), p.getProperty(name));
                }
                else if (name.startsWith("basherBootClassPathUrl."))
                {
                    basherBooter.addBasherBootClassPathUrl(p.getProperty(name));
                }
                else if ("enableAssertions".equals(name))
                {
                    basherBooter._enableAssertions =
                            Boolean.valueOf(p.getProperty("enableAssertions")).booleanValue();
                }
                else if ("useSystemClassLoader".equals(name))
                {
                    basherBooter._forkConfiguration.setUseSystemClassLoader(Boolean.valueOf(
                            p.getProperty("useSystemClassLoader")).booleanValue());
                }
                else if ("useManifestOnlyJar".equals(name))
                {
                    basherBooter._forkConfiguration.setUseManifestOnlyJar(Boolean.valueOf(
                            p.getProperty("useManifestOnlyJar")).booleanValue());
                }
                else if ("failIfNoTasks".equals(name))
                {
                    basherBooter.setFailIfNoTasks(Boolean.valueOf(p.getProperty("failIfNoTasks")).booleanValue());
                }
            }

            for (Iterator<Integer> cpi = classPathUrls.keySet().iterator(); cpi.hasNext();)
            {
                final String url = classPathUrls.get(cpi.next());
                basherBooter.addClassPathUrl(url);
            }

            for (Iterator<Integer> scpi = basherClassPathUrls.keySet().iterator(); scpi.hasNext();)
            {
                final String url = basherClassPathUrls.get(scpi.next());
                basherBooter.addBasherClassPathUrl(url);
            }

            // Load up the basher context as well as the active one.
            final String activeBasherContext = p.getProperty("activeBasherContext", "default");
            basherBooter.setBasherContexts(deserializeBasherContexts(p));

            // Finally, start up basher
            final int result = basherBooter.runBasherInProcess(activeBasherContext, p);

            // noinspection CallToSystemExit
            System.exit(result);
        }
        catch (Throwable t)
        {
            // Just throwing does getMessage() and a local trace - we want to call printStackTrace for a full trace
            // noinspection UseOfSystemOutOrSystemErr
            t.printStackTrace(System.err);
            // noinspection ProhibitedExceptionThrown,CallToSystemExit
            System.exit(1);
        }
    }

    private static Object[] constructParamObjects(final String paramProperty, final String typeProperty)
    {
        Object[] paramObjects = null;
        if (paramProperty != null)
        {
            // bit of a glitch that it need sto be done twice to do an odd number of vertical bars (eg |||, |||||).
            final String[] params =
                    StringUtils.split(
                            StringUtils.replace(StringUtils.replace(paramProperty, "||", "| |"), "||", "| |"),
                            "|");
            final String[] types =
                    StringUtils.split(
                            StringUtils.replace(StringUtils.replace(typeProperty, "||", "| |"), "||", "| |"),
                            "|");

            paramObjects = new Object[params.length];

            for (int i = 0; i < types.length; i++)
            {
                if (types[i].trim().length() == 0)
                {
                    params[i] = null;
                }
                else if (types[i].equals(String.class.getName()))
                {
                    paramObjects[i] = params[i];
                }
                else if (types[i].equals(File.class.getName()))
                {
                    paramObjects[i] = new File(params[i]);
                }
                else if (types[i].equals(File[].class.getName()))
                {
                    final List<String> stringList = processStringList(params[i]);
                    final File[] fileList = new File[stringList.size()];
                    for (int j = 0; j < stringList.size(); j++)
                    {
                        fileList[j] = new File(stringList.get(j));
                    }
                    paramObjects[i] = fileList;
                }
                else if (types[i].equals(ArrayList.class.getName()))
                {
                    paramObjects[i] = processStringList(params[i]);
                }
                else if (types[i].equals(Boolean.class.getName()))
                {
                    paramObjects[i] = Boolean.valueOf(params[i]);
                }
                else if (types[i].equals(Integer.class.getName()))
                {
                    paramObjects[i] = Integer.valueOf(params[i]);
                }
                else if (types[i].equals(Properties.class.getName()))
                {
                    final Properties result = new Properties();
                    final String value = params[i];
                    try
                    {
                        final ByteArrayInputStream bais = new ByteArrayInputStream(value.getBytes("8859_1"));
                        result.load(bais);
                    }
                    catch (Exception e)
                    {
                        throw new RuntimeException("bug in property conversion", e);
                    }
                    paramObjects[i] = result;
                }
                else
                {
                    // TODO: could attempt to construct with a String constructor if needed
                    throw new IllegalArgumentException("Unknown parameter type: " + types[i]);
                }
            }
        }
        return paramObjects;
    }

    private ForkingStreamConsumer getForkingStreamConsumer(final boolean showHeading, final boolean showFooter,
                                                           final boolean redirectTasksOutputToFile)
    {
        OutputConsumer outputConsumer = new StandardOutputConsumer();

        if (redirectTasksOutputToFile)
        {
            outputConsumer = new FileOutputConsumerProxy(outputConsumer, getReportsDirectory());
        }

        if (!showHeading)
        {
            outputConsumer = new SupressHeaderOutputConsumerProxy(outputConsumer);
        }
        if (!showFooter)
        {
            outputConsumer = new SupressFooterOutputConsumerProxy(outputConsumer);
        }

        return new ForkingStreamConsumer(outputConsumer);
    }

    // ----------------------------------------------------------------------
    // Accessors
    // ----------------------------------------------------------------------

    public void addReport(final String report)
    {
        addReport(report, null);
    }

    public void addReport(final String report, final Object[] constructorParams)
    {
        _reports.add(new Object[]{report, constructorParams});
    }

    public void addClassPathUrl(final String path)
    {
        if (!_classPathUrls.contains(path))
        {
            _classPathUrls.add(path);
        }
    }

    public void addBasherClassPathUrl(final String path)
    {
        if (!_basherClassPathUrls.contains(path))
        {
            _basherClassPathUrls.add(path);
        }
    }

    public void addBasherBootClassPathUrl(final String path)
    {
        if (!_basherBootClassPathUrls.contains(path))
        {
            _basherBootClassPathUrls.add(path);
        }
    }

    /**
     * Setting this to true will cause a failure if there are no tasks to run
     *
     * @param failIfNoTasks If true and no tasks are found, Basher will fail
     */
    public void setFailIfNoTasks(final boolean failIfNoTasks)
    {
        this._failIfNoTasks = failIfNoTasks;
    }

    /**
     * When forking, setting this to true will make the task output to be saved in a file instead of showing it on the
     * standard output
     *
     * @param redirectTasksOutputToFile If true, any output will be redirected to a file
     */
    public void setRedirectTasksOutputToFile(final boolean redirectTasksOutputToFile)
    {
        _redirectTasksOutputToFile = redirectTasksOutputToFile;
    }

    /**
     * Set the directory where reports will be saved
     *
     * @param reportsDirectory The directory where reports will be saved
     */
    public void setReportsDirectory(final File reportsDirectory)
    {
        _reportsDirectory = reportsDirectory;
    }

    /**
     * Get the directory where reports will be saved
     *
     * @return The file representing the directory where files are saved
     */
    public File getReportsDirectory()
    {
        return _reportsDirectory;
    }

    /**
     * Sets the forking configuration to use.
     *
     * @param forkConfiguration The fork configuration
     */
    public void setForkConfiguration(final BasherForkConfiguration forkConfiguration)
    {
        _forkConfiguration = forkConfiguration;
    }

    private final boolean useSystemClassLoader()
    {
        return _forkConfiguration.isUseSystemClassLoader() && (_isForked || _forkConfiguration.isForking());
    }

    private final boolean useManifestOnlyJar()
    {
        return _forkConfiguration.isUseSystemClassLoader() && _forkConfiguration.isUseManifestOnlyJar();
    }

    private static void setSystemProperties(final File file)
            throws IOException
    {
        final Properties p = loadProperties(file);

        for (Iterator i = p.keySet().iterator(); i.hasNext();)
        {
            final String key = (String) i.next();

            System.setProperty(key, p.getProperty(key));
        }
    }

    public void setEnableAssertions(final boolean enableAssertions)
    {
        _enableAssertions = enableAssertions;
    }

    public void setForkedProcessTimeoutInSeconds(final int forkedProcessTimeoutInSeconds)
    {
        _forkedProcessTimeoutInSeconds = forkedProcessTimeoutInSeconds;
    }

    public void setBasherContexts(final List<BasherContext> basherContexts)
    {
        _basherContexts = basherContexts;
    }

    public void setActiveBasherContext(final String activeBasherContext)
    {
        _activeBasherContext = activeBasherContext;
    }

    public boolean isForking()
    {
        return _forkConfiguration.isForking();
    }

}
