/**
 * Copyright 2010, CSIRO Australia.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * 
 */
package au.csiro.pidclient.business;

import java.text.MessageFormat;

/**
 * Represents the identify information required when calling the ANDS Persistent Identifier service.
 * 
 * Copyright 2010, CSIRO Australia All rights reserved.
 * 
 * @author Robert Bridle on 08/02/2010
 * @version $Revision: 7131 $ $Date: 2010-06-09 14:25:15 +1000 (Wed, 09 Jun 2010) $
 */
public class AndsPidIdentity
{
    /**
     * The unique Id provided to the caller upon IP registration with the ANDS Persistent Identifier service.
     */
    private String appId;
    
    /**
     * The identifier or name of the repository calling the service.
     */
    private String identifier;
    
    /**
     * The domain of the organisation calling the service.
     */
    private String authDomain;

    /**
     * 
     */
    public AndsPidIdentity()
    {
    }

    /**
     * Constructor
     * 
     * @param appId
     *            the unique Id provided to the caller upon IP registration with the ANDS Persistent Identifier service.
     * @param identifier
     *            the identifier or name of the repository calling the service.
     * @param authDomain
     *            the domain of the organisation calling the service.
     */
    @SuppressWarnings(value = "all")
    public AndsPidIdentity(String appId, String identifier, String authDomain)
    {
        this.setAppId(appId);
        this.setIdentifier(identifier);
        this.setAuthDomain(authDomain);
    }

    /**
     * @return the appId
     */
    public String getAppId()
    {
        return escape(appId);
    }

    /**
     * @param appId
     *            the appId to set
     */
    public void setAppId(String appId)
    {
        this.appId = appId;
    }

    /**
     * @return the identifier
     */
    public String getIdentifier()
    {
        return escape(identifier);
    }

    /**
     * @param identifier
     *            the identifier to set
     */
    public void setIdentifier(String identifier)
    {
        this.identifier = identifier;
    }

    /**
     * @return the authDomain
     */
    public String getAuthDomain()
    {
        return escape(authDomain);
    }

    /**
     * @param authDomain
     *            the authDomain to set
     */
    public void setAuthDomain(String authDomain)
    {
        this.authDomain = authDomain;
    }

    /**
     * Formats the identity information into a XML format that the ANDS Persistent Identifier service understands.
     * 
     * @param methodName
     *            the name of the method to be called using this identity information.
     * @return the identity XML string representing the identity information.
     */
    @SuppressWarnings(value = "all")
    public String toXML(String methodName)
    {
        return MessageFormat.format("<request name=\"{0}\">" 
            + "<properties>" 
            + "<property name=\"appId\" value=\"" + this.getAppId() + "\"/>" 
            + "<property name=\"identifier\" value=\"" + this.getIdentifier() + "\"/>"
            + "<property name=\"authDomain\" value=\"" + this.getAuthDomain() + "\"/>" 
            + "</properties>" 
            + "</request>",
            new Object[] { methodName });
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        return "RequestorIdentity [appId=" + appId + ", authDomain=" + authDomain + ", identifier=" + identifier + "]";
    }

    /**
     * Escape reserved characters, assumes UTF-8 or UTF-16 as encoding.
     * 
     * @param in
     *            the String whose reserved characters we want to remove.
     * @return the in String, stripped of reserved characters.
     */
    public String escape(String in)
    {
        if(in == null)
        {
            return in;
        }
        
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < in.length(); i++)
        {
            char c = in.charAt(i);
            if (c == '<')
            {
                buffer.append("&lt;");
            }
            else if (c == '>')
            {
                buffer.append("&gt;");
            }
            else if (c == '&')
            {
                buffer.append("&amp;");
            }
            else if (c == '"')
            {
                buffer.append("&quot;");
            }
            else if (c == '\'')
            {
                buffer.append("&apos;");
            }
            else
            {
                buffer.append(c);
            }
        }
        return buffer.toString();
    }

}
