/**
 * Copyright 2010, CSIRO Australia.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package au.csiro.pidclient;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpsURL;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.RequestEntity;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import au.csiro.pidclient.business.AndsPidIdentity;
import au.csiro.pidclient.business.AndsPidResponseProperty;

/**
 * This is the main interface to the ANDS PID Client library. It allows the caller to 
 * interact with the 
 * <a href="http://www.ands.org.au/services/pid-m2m-identifiers.html">Persistent IDentifier Service</a> 
 * provided by the 
 * <a href="http://www.ands.org.au/">Australian National Data Service</a>. You will need to 
 * register with ANDS to be able to use the service. 
 * <p><b>Usage:</b>
 * </p><p>
 * Create a new instance of the class using either the empty constructor and 
 * then calling the four setters, or using the constructors. This should be 
 * compatible with use as a singleton bean in Spring or other DI (dependency 
 * injection) frameworks. Example:
 * </p>
 * <UL>
 * <li>pidServiceHost - test.ands.org.au</li>
 * <li>pidServicePort - 8443</li>
 * <li>pidServicePath - /pids</li>
 * <li>requestorIdentity.appId - 5d9a4da3580c528ba98d8e6f088dab93f680dd6b</li>
 * <li>requestorIdentity.identifier - scott</li>
 * <li>requestorIdentity.authDomain - mycomputer.edu.au</li>
 * </UL>
 * <p>
 * You can then mint new handles using the {@link #mintHandle(HandleType, String) mintHandle} 
 * method, and manage the properties associated with the handle
 * {@link #addValue(String, HandleType, String) addValue}, 
 * {@link #modifyValueByIndex(String, int, String) modifyValueByIndex} and
 * {@link #deleteValueByIndex(String, int) deleteValueByIndex} methods. Note that these 
 * methods return the raw XML response as a string. You can also use the FormattedResponse 
 * versions to get back an interpreted result as a standard Java bean.
 * 
 * </p>
 * Copyright 2010, CSIRO Australia All rights reserved.
 * 
 * @author Robert Bridle on 05/02/2010
 * @version $Revision: 7131 $ $Date: 2010-06-09 14:25:15 +1000 (Wed, 09 Jun 2010) $
 */
public class AndsPidClient
{
    /**
     * Constant that defines the size of an array for storing triplet data.
     */
    private static final int TRIPLE = 3;

    /**
     * Constant that defines the logger to be used.
     */
    private static final Logger LOG = Logger.getLogger(AndsPidClient.class.getName());

    /**
     * Constant that defines the name of properties file to be used.
     */
    private static final String PROPERTIES_FILENAME = "/ands-pid-client.properties";

    /**
     * The name of the application, to be used by the HTTP client.
     */
    private static String applicationName;

    /**
     * The name of the method (RESTful web service) to call when minting a new handle.
     */
    private static String mintMethodName;
    
    /**
     * The name of the method (RESTful web service) to call when associating a new value with an existing handle.
     */
    private static String addValueMethodName;
    
    /**
     * The name of the method (RESTful web service) to call when associating a new value into the index of an existing
     * handle.
     */
    private static String addValueByIndexMethodName;

    /**
     * The name of the method (RESTful web service) to call when modifying an existing handle.
     */
    private static String modifyValueByIndexMethodName;

    /**
     * The name of the method (RESTful web service) to call when deleting an existing handle value.
     */
    private static String deleteValueByIndexMethodName;
    
    /**
     * The name of the method (RESTful web service) to call when listing the handles owned by the requestor.
     */
    private static String listHandlesMethodName;
    
    /**
     * The name of the method (RESTful web service) to call when retrieving the values associated with a handle.
     */
    private static String getHandleMethodName;

    /**
     * Represents the identity information of the caller.
     */
    private AndsPidIdentity requestorIdentity;

    /**
     * The ANDS Persistent Identifier host name.
     */
    private String pidServiceHost;

    /**
     * The ANDS Persistent Identifier port number.
     */
    private int pidServicePort;

    /**
     * The ANDS Persistent Identifier path name (web application context name).
     */
    private String pidServicePath;

    /**
     * The possible types of properties that can be associated with a handle. 
     * Each handle (PID) can carry multiple (max of around 100) properties. 
     */
    public enum HandleType
    {
        /**
         * A property with no associated value. 
         */
        EMPTY(""),
        /**
         * A property with a URL value.
         */
        URL("URL"),
        /**
         * A property with a descriptive text value.
         */
        DESC("DESC");

        /**
         * The value of the enumeration understood by the ANDS PID service.
         */
        private final String value;

        HandleType(String value)
        {
            this.value = value;
        }

        /**
         * @return the value of the enumeration.
         */
        public String value()
        {
            return this.value;
        }
        
        /**
         * @return whether this enumeration is empty.
         */
        public boolean isEmpty()
        {
            return EMPTY.value().equals(this.value);
        }
    }

    /**
     * Loads the specified properties file.
     */
    private static void loadProperties()
    {
        InputStream is = AndsPidResponse.class.getResourceAsStream(PROPERTIES_FILENAME);
        Properties props = new Properties();
        try
        {
            props.load(is);
            applicationName = props.getProperty("application.name");
            mintMethodName = props.getProperty("method.mint");
            addValueMethodName = props.getProperty("method.addValue");
            addValueByIndexMethodName = props.getProperty("method.addValueByIndex");
            modifyValueByIndexMethodName = props.getProperty("method.modifyValueByIndex");
            deleteValueByIndexMethodName = props.getProperty("method.deleteValueByIndex");
            listHandlesMethodName = props.getProperty("method.listHandles");
            getHandleMethodName = props.getProperty("method.getHandle");
        }
        catch (IOException e)
        {
            LOG.error("Could not load properties file: " + PROPERTIES_FILENAME, e);
        }
    }

    /**
     * Default constructor. You will still need to supply the configuration data 
     * via the public setters.
     */
    public AndsPidClient()
    {
        this(null, 0, null, null, null, null);
    }

    /**
     * @param pidServiceHost
     *            the ANDS Persistent Identifier host name.
     * @param pidServicePort
     *            the ANDS Persistent Identifier port number.
     * @param pidServicePath
     *            the ANDS Persistent Identifier path name (web application context name).
     * @param appId
     *            the unique Id provided to the caller upon IP registration with the ANDS Persistent Identifier service.
     * @param identifier
     *            the identifier or name of the repository calling the service.
     * @param authDomain
     *            the domain of the organisation calling the service.
     */
    public AndsPidClient(String pidServiceHost, int pidServicePort, String pidServicePath, String appId,
            String identifier, String authDomain)
    {
        this(pidServiceHost, pidServicePort, pidServicePath, new AndsPidIdentity(appId, identifier, authDomain));
    }

    /**
     * @param pidServiceHost
     *            the ANDS Persistent Identifier host name.
     * @param pidServicePort
     *            the ANDS Persistent Identifier port number.
     * @param pidServicePath
     *            the ANDS Persistent Identifier path name (web application context name).
     * @param requestorIdentity
     *            represents the identity information of the caller {@link AndsPidIdentity}.
     */
    public AndsPidClient(String pidServiceHost, int pidServicePort, String pidServicePath,
            AndsPidIdentity requestorIdentity)
    {
        this.setPidServiceHost(pidServiceHost);
        this.setPidServicePort(pidServicePort);
        this.setPidServicePath(pidServicePath);
        this.setRequestorIdentity(requestorIdentity);
        loadProperties();
    }

    /**
     * Responsible for the creation of a handle.
     * <p>
     * If the type and value arguments are both empty, a handle with no values is created. The handle is assigned to an
     * owner, specified by the {@link AndsPidIdentity#getAppId()} value. If the owner is not known to the handle system,
     * an owner is created from the {@link AndsPidIdentity#getIdentifier()} and {@link AndsPidIdentity#getIdentifier()}
     * values.
     * 
     * @param type
     *            the type {@link HandleType} of value which will be associated with the newly minted handle.
     * @param value
     *            the value which will be associated with the newly minted handle.
     * @return a formatted XML response that contains a handle, the handle is associated with the value argument.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     */
    public String mintHandle(HandleType type, String value) throws IllegalStateException, IllegalArgumentException,
            HttpException, IOException
    {
        this.validateState();

        this.validateMintHandleArguments(type, value);

        NameValuePair[] queryParams = {
                new NameValuePair("type", type.value()),
                new NameValuePair("value", value)
              };

        return executeMethod(queryParams, mintMethodName);
    }

    /**
     * Responsible for the creation of a handle.
     * <p>
     * If the type and value arguments are both empty, a handle with no values is created. The handle is assigned to an
     * owner, specified by the {@link AndsPidIdentity#getAppId()} value. If the owner is not known to the handle system,
     * an owner is created from the {@link AndsPidIdentity#getIdentifier()} and {@link AndsPidIdentity#getIdentifier()}
     * values.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param type
     *            the type {@link HandleType} of value which will be associated with the newly minted handle.
     * @param value
     *            the value which will be associated with the newly minted handle.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse mintHandleFormattedResponse(HandleType type, String value) throws IllegalStateException,
            IllegalArgumentException, HttpException, IOException, XPathExpressionException,
            ParserConfigurationException, SAXException
    {
        return populateAndsPidResponse(mintHandle(type, value));
    }
    
    /**
     * Responsible for the creation of a handle with a value at a specific index.
     * <p>
     * If the type and value arguments are both empty, a handle with no values is created. The handle is assigned
     * to an owner, specified by the {@link AndsPidIdentity#getAppId()} value. If the owner is not known to the handle
     * system, an owner is created from the {@link AndsPidIdentity#getIdentifier()} and
     * {@link AndsPidIdentity#getIdentifier()} values.
     * 
     * @param type
     *            the type {@link HandleType} of value which will be associated with the newly minted handle.
     * @param index
     *            the index of the value which will be associated with the newly minted handle.
     * @param value
     *            the value which will be associated with the newly minted handle.
     * @return a formatted XML response that contains a handle, the handle is associated with the value argument.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     */
    public String mintHandle(HandleType type, int index, String value) throws IllegalStateException,
            IllegalArgumentException, HttpException, IOException
    {
        this.validateState();

        this.validateMintHandleArguments(type, value);

        NameValuePair[] queryParams = {
                new NameValuePair("type", type.value()),
                new NameValuePair("index", String.valueOf(index)),
                new NameValuePair("value", value)
              };
        return executeMethod(queryParams, mintMethodName);
    }
    
    /**
     * Responsible for the creation of a handle with a value at a specific index.
     * <p>
     * If the type and value arguments are both empty, a handle with no values is created. The handle is assigned
     * to an owner, specified by the {@link AndsPidIdentity#getAppId()} value. If the owner is not known to the handle
     * system, an owner is created from the {@link AndsPidIdentity#getIdentifier()} and
     * {@link AndsPidIdentity#getIdentifier()} values.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param type
     *            the type {@link HandleType} of value which will be associated with the newly minted handle.
     * @param index
     *            the index of the value which will be associated with the newly minted handle.
     * @param value
     *            the value which will be associated with the newly minted handle.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse mintHandleFormattedResponse(HandleType type, int index, String value)
            throws IllegalStateException, IllegalArgumentException, HttpException, IOException,
            XPathExpressionException, ParserConfigurationException, SAXException
    {
        return populateAndsPidResponse(mintHandle(type, index, value));
    }

    /**
     * Adds a value to an existing handle.
     * <p>
     * Only the owner of the handle is able to add a value to the handle and only values of type {@link HandleType#URL}
     * or {@link HandleType#DESC} can be added.
     * 
     * @param handle
     *            the handle to which a new value is to be associated.
     * @param type
     *            the type of the value to be added to the handle, must be either {@link HandleType#URL} or
     *            {@link HandleType#DESC}.
     * @param value
     *            the value to be added to the handle.
     * @return a formatted XML response that contains the details of the updated handle.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     */
    @SuppressWarnings(value = "all")
    public String addValue(String handle, HandleType type, String value) throws IllegalStateException,
            IllegalArgumentException, HttpException, IOException
    {
        this.validateState();

        if (StringUtils.isEmpty(handle) || StringUtils.isEmpty(value) || type == null || type.equals(HandleType.EMPTY))
        {
            throw new IllegalArgumentException(MessageFormat.format(
                    "The method addValue() can not be called with null or empty arguments:\n type= {0}\n value= {1}\n",
                    new Object[] { (type == null) ? null : type.value(), value }));
        }

        NameValuePair[] queryParams = {
                new NameValuePair("type", type.value()),
                new NameValuePair("handle", handle),
                new NameValuePair("value", value)
              };
        return executeMethod(queryParams, addValueMethodName);
    }

    /**
     * Adds a value to an existing handle.
     * <p>
     * Only the owner of the handle is able to add a value to the handle and only values of type {@link HandleType#URL}
     * or {@link HandleType#DESC} can be added.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param handle
     *            the handle to which a new value is to be associated.
     * @param type
     *            the type of the value to be added to the handle, must be either {@link HandleType#URL} or
     *            {@link HandleType#DESC}.
     * @param value
     *            the value to be minted.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse addValueFormattedResponse(String handle, HandleType type, String value)
            throws IllegalStateException, IllegalArgumentException, HttpException, IOException,
            XPathExpressionException, ParserConfigurationException, SAXException
    {
        // populate and return the response object
        return populateAndsPidResponse(this.addValue(handle, type, value));
    }
    
    /**
     * Adds a value to a particular index of an existing handle.
     * <p>
     * Only the owner of the handle is able to add a value to the handle and only values of type {@link HandleType#URL}
     * or {@link HandleType#DESC} can be added.
     * 
     * @param handle
     *            the handle to which a new value is to be associated.
     * @param index
     *            the index in which to add the value.
     * @param type
     *            the type of the value to be added to the handle, must be either {@link HandleType#URL} or
     *            {@link HandleType#DESC}.
     * @param value
     *            the value to be added to the handle.
     * @return a formatted XML response that contains the details of the updated handle.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     */
    @SuppressWarnings(value = "all")
    public String addValueByIndex(String handle, int index, HandleType type, String value)
            throws IllegalStateException, IllegalArgumentException, HttpException, IOException
    {
        this.validateState();

        if (StringUtils.isEmpty(handle) || StringUtils.isEmpty(value) || type == null || type.isEmpty())
        {
            throw new IllegalArgumentException(MessageFormat.format(
                    "The method addValueByIndex() can not be called with null or empty arguments:\n type= {0}\n value= {1}\n",
                    new Object[] { (type == null) ? null : type.value(), value }));
        }

        NameValuePair[] queryParams = {
                new NameValuePair("handle", handle),
                new NameValuePair("index", String.valueOf(index)),
                new NameValuePair("type", type.value()),                
                new NameValuePair("value", value)
              };
        return executeMethod(queryParams, addValueByIndexMethodName);
    }

    /**
     * Adds a value to a particular index of an existing handle.
     * <p>
     * Only the owner of the handle is able to add a value to the handle and only values of type {@link HandleType#URL}
     * or {@link HandleType#DESC} can be added.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param handle
     *            the handle to which a new value is to be associated.
     * @param index
     *            the index in which to add the value.
     * @param type
     *            the type of the value to be added to the handle, must be either {@link HandleType#URL} or
     *            {@link HandleType#DESC}.
     * @param value
     *            the value to be minted.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse addValueByIndexFormattedResponse(String handle, int index, HandleType type, String value)
            throws IllegalStateException, IllegalArgumentException, HttpException, IOException,
            XPathExpressionException, ParserConfigurationException, SAXException
    {
        // populate and return the response object
        return populateAndsPidResponse(this.addValueByIndex(handle, index, type, value));
    }

    /**
     * Changes a value associated with an existing handle.
     * <p>
     * Only the owner of the handle is able to modify a value associated with the handle and only values with the type
     * {@link HandleType#URL} or {@link HandleType#DESC} can be modified.
     * 
     * @param handle
     *            the handle that is to have one of its values modified.
     * @param index
     *            the index of the value to modify.
     * @param newValue
     *            the new value.
     * @return a formatted XML response that contains the details of the updated handle.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    @SuppressWarnings(value = "all")
    public String modifyValueByIndex(String handle, int index, String newValue) throws IllegalStateException,
            IllegalArgumentException, HttpException, IOException
    {
        this.validateState();

        if (StringUtils.isEmpty(handle) || StringUtils.isEmpty(newValue))
        {
            throw new IllegalArgumentException(
                    MessageFormat
                            .format(
                                    "The method modifyValueByIndex() can not be called with null or empty arguments:" +
                                    "\n handle={0}\n newValue={1}\n",
                                    new Object[] { handle, newValue }));
        }

        NameValuePair[] queryParams = {
                new NameValuePair("index", String.valueOf(index)),
                new NameValuePair("handle", handle),
                new NameValuePair("value", newValue)
              };
        return executeMethod(queryParams, modifyValueByIndexMethodName);
    }

    /**
     * Changes a value associated with an existing handle.
     * <p>
     * Only the owner of the handle is able to modify a value associated with the handle and only values with the type
     * {@link HandleType#URL} or {@link HandleType#DESC} can be modified.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param handle
     *            the handle that is to have one of its values modified.
     * @param index
     *            the index of the value to modify.
     * @param newValue
     *            the new value.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse modifyValueByIndexFormattedResponse(String handle, int index, String newValue)
            throws IllegalStateException, IllegalArgumentException, HttpException, IOException,
            XPathExpressionException, ParserConfigurationException, SAXException
    {
        return populateAndsPidResponse(modifyValueByIndex(handle, index, newValue));
    }

    /**
     * Deletes a value associated with an existing handle.
     * <p>
     * Only the owner of the handle is able to delete a value associated to the handle and only values of type
     * {@link HandleType#URL} or {@link HandleType#DESC} can be deleted.
     * 
     * @param handle
     *            the handle that is to have one of its values deleted.
     * @param index
     *            the index of the value to be deleted.
     * @return a formatted XML response that contains the details of the updated handle.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    public String deleteValueByIndex(String handle, int index) throws IllegalStateException, IllegalArgumentException,
            HttpException, IOException
    {
        this.validateState();

        if (StringUtils.isEmpty(handle))
        {
            throw new IllegalArgumentException(MessageFormat.format(
                    "The method deleteValueByIndex() can not be called with null or empty arguments:\n handle={0}\n",
                    new Object[] { handle }));
        }
      
        NameValuePair[] queryParams = {
                new NameValuePair("index", String.valueOf(index)),
                new NameValuePair("handle", handle)};
        return executeMethod(queryParams, deleteValueByIndexMethodName);
    }

    /**
     * Deletes a value associated with an existing handle. Only values with the type {@link HandleType#URL} or
     * {@link HandleType#DESC} can be deleted.
     * <p>
     * Only the owner of the handle is able to delete a value associated to the handle and only values of type
     * {@link HandleType#URL} or {@link HandleType#DESC} can be deleted.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param handle
     *            the handle that is to have one of its indexes deleted.
     * @param index
     *            the index to be deleted.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse deleteValueByIndexFormattedResponse(String handle, int index) throws IllegalStateException,
            IllegalArgumentException, HttpException, IOException, XPathExpressionException,
            ParserConfigurationException, SAXException
    {
        return populateAndsPidResponse(deleteValueByIndex(handle, index));
    }

    /**
     * List the handles owned by the caller of the ANDS PID service.
     * <p>
     * This service is intended for listing a small number of handles in a GUI environment, therefore its response is
     * limited in the number of handles returned.
     * 
     * @return a formatted XML response that contains the details of the handles owned by the caller.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    @SuppressWarnings(value = "all")
    public String listHandles() throws IllegalStateException, HttpException, IOException
    {
        this.validateState();
        NameValuePair[] queryParams = {new NameValuePair()};
        return executeMethod(queryParams, listHandlesMethodName);
    }

    /**
     * List the handles owned by the caller of the ANDS PID service.
     * <p>
     * This service is intended for listing a small number of handles in a GUI environment, therefore its response is
     * limited in the number of handles returned.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse listHandlesFormattedResponse() throws IllegalStateException, HttpException, IOException,
            XPathExpressionException, ParserConfigurationException, SAXException
    {
        return populateAndsPidResponse(listHandles());
    }
    
    /**
     * Retrieves the values associated with a given handle.
     * 
     * @param handle
     *            the handle whose details are to be retrieved.
     * @return a formatted XML response that contains the details of the handle.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    @SuppressWarnings(value = "all")
    public String getHandle(String handle) throws IllegalStateException, IllegalArgumentException,
            HttpException, IOException
    {
        this.validateState();

        if (StringUtils.isEmpty(handle))
        {
            throw new IllegalArgumentException(MessageFormat.format(
                    "The method getHandle() can not be called with null or empty arguments:\n handle={0}\n",
                    new Object[] { handle }));
        }

        NameValuePair[] queryParams = {new NameValuePair("handle", handle)};
        return executeMethod(queryParams, getHandleMethodName);
    }

    /**
     * Retrieves the values associated with a given handle.
     * <p>
     * The XML response is parsed and returned by the object {@link AndsPidResponse}.
     * 
     * @param handle
     *            the handle whose details are to be retrieved.
     * @return the ANDS Persistent Identifier service response represented by the object {@link AndsPidResponse}.
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     * @throws IllegalArgumentException
     *             thrown when method is called with invalid arguments.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     */
    public AndsPidResponse getHandleFormattedResponse(String handle) throws IllegalStateException,
            IllegalArgumentException, HttpException, IOException, XPathExpressionException,
            ParserConfigurationException, SAXException
    {
        return populateAndsPidResponse(getHandle(handle));
    }

    /**
     * Constructs and executes an HTTP POST call.
     * 
     * @param queryParams
     *            the array of query parameters to provide the POST call.
     * @param methodName
     *            the method to call.
     * @return a formatted XML response.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     */
    private String executeMethod(NameValuePair[] queryParams, String methodName) throws HttpException, IOException
    {
        HttpsURL url = new HttpsURL(/* String user */"", /* String password */"", this.getPidServiceHost(), this
                .getPidServicePort(), this.getPidServicePath());

        // if no path has been specified (i.e. path value is just '/') then we have:
        // http://test.org.au:80/mint?type=URL&value=<some value>
        if ("/".equals(url.getPath()))
        {
            url.setPath(url.getPath() + methodName);
        }
        else
        // if a path has been specified then we have: http://test.org.au:80/<some path>/mint?type=URL&value=<some value>
        {
            url.setPath(url.getPath() + "/" + methodName);
        }

        String identityXML = this.getRequestorIdentity().toXML(methodName);

        return executePostMethod(url.toString(), queryParams, identityXML);

    }

    /**
     * Calls a POST method of the ANDS Persistent Identifier service in a RESTful web service manner. The query string
     * of the URI defines the type of operation that is to be performed. The request body contains an XML fragment that
     * identifies the caller.
     * 
     * @param postMethodURL
     *            the URI of the RESTful ANDS Persistent Identifier web service.
     * @param params
     *            the array of query parameters to provide the POST call.
     * @param identityXML
     *            an XML fragment that details the identity of the caller.
     * @return a formatted XML response.
     * @throws IOException
     *             thrown when attempting to read response.
     * @throws HttpException
     *             thrown when attempting to execute method call.
     */
    private String executePostMethod(String postMethodURL, NameValuePair[] params, String identityXML) throws HttpException, IOException
    {
        LOG.debug("Post method URL: " + postMethodURL);
        LOG.debug("Identity XML: " + identityXML);

        HttpClient client = new HttpClient();
        client.getParams().setParameter("http.useragent", applicationName);
        client.getParams().setParameter("Content-Type", "text/xml");
        client.getParams().setParameter("Content-Encoding", "UTF-8");

        PostMethod method = new PostMethod(postMethodURL);

        BufferedReader br = null;
        StringBuffer strBuf = new StringBuffer();

        try
        {
            RequestEntity entity = new StringRequestEntity(identityXML, "text/xml", "UTF-8");
            client.getParams().setParameter("Content-Length", entity.getContentLength());
            method.setRequestEntity(entity);
            method.setQueryString(params);
            int returnCode = client.executeMethod(method);
            if (returnCode == HttpStatus.SC_NOT_IMPLEMENTED)
            {
                LOG.error("The post method is not implemented by this URI");
                // still consume the response body
                method.getResponseBodyAsString();
            }
            else
            {
                br = new BufferedReader(new InputStreamReader(method.getResponseBodyAsStream()));
                String readLine;
                while (((readLine = br.readLine()) != null))
                {
                    LOG.debug(readLine);
                    strBuf.append(readLine);
                }
            }
        }
        finally
        {
            if (br != null)
            {
                br.close();
            }
            method.releaseConnection();
        }

        return strBuf.toString();
    }

    /**
     * Parses an XML document which contains an ANDS PID service response for a node that specifies whether the service
     * call was successful.
     * 
     * @param doc
     *            an XML document of the ANDS PID service response.
     * @return whether the response represents a success
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    @SuppressWarnings(value = "all")
    private boolean parseForSuccess(Document doc) throws XPathExpressionException, ParserConfigurationException,
            SAXException, IOException
    {
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();

        // whether <response type="success"> was returned
        XPathExpression expr = xpath.compile("//response[@type]");
        Object result = expr.evaluate(doc, XPathConstants.NODESET);
        NodeList nodes = (NodeList) result;
        for (int i = 0; i < nodes.getLength(); i++)
        {
            for (int j = 0; j < nodes.item(i).getAttributes().getLength(); j++)
            {
                Node node = (Node) nodes.item(i).getAttributes().item(j);
                return "success".equals(node.getNodeValue());
            }
        }
        return false;
    }

    /**
     * Parses an XML document which contains an ANDS PID service response 
     * for a node that specifies what the response message is.
     * 
     * @param doc
     *            an XML document of the ANDS PID service response.
     * @return the response message
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    @SuppressWarnings(value = "all")
    private String parseForMessage(Document doc) throws XPathExpressionException, ParserConfigurationException,
            SAXException, IOException
    {
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();

        XPathExpression expr = xpath.compile("//response/message");
        Object result = expr.evaluate(doc, XPathConstants.NODESET);
        NodeList nodes = (NodeList) result;
        for (int i = 0; i < nodes.getLength(); i++)
        {
            return nodes.item(i).getTextContent();
        }
        return null;
    }
    
    /**
     * Parses an XML document which contains an ANDS PID service response for the nodes specifying the handles that were
     * used in the service call.
     * 
     * @param doc
     *            an XML document of the ANDS PID service response.
     * @return the handles associated with the service call.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    private List<String> parseForHandles(Document doc) throws XPathExpressionException, ParserConfigurationException,
            SAXException, IOException
    {
        List<String> handles = new ArrayList<String>();

        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();

        // look for the tag: <identifier handle="XXXX/X">
        XPathExpression expr = xpath.compile("//identifier[@handle]");
        Object result = expr.evaluate(doc, XPathConstants.NODESET);
        NodeList nodes = (NodeList) result;
        for (int i = 0; i < nodes.getLength(); i++)
        {
            for (int j = 0; j < nodes.item(i).getAttributes().getLength(); j++)
            {
                Node node = (Node) nodes.item(i).getAttributes().item(j);
                String handle = node.getNodeValue();
                handles.add(handle);
            }
        }
        return handles;
    }

    /**
     * Parses an XML document which contains an ANDS PID service response for the nodes specifying the properties
     * associated with a handle.
     * 
     * @param doc
     *            an XML document of the ANDS PID service response.
     * @return the properties associated with the handle used in the service call.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    private static List<AndsPidResponseProperty> parseForProperties(Document doc)
            throws XPathExpressionException, ParserConfigurationException, SAXException, IOException
    {
        List<AndsPidResponseProperty> properties = new ArrayList<AndsPidResponseProperty>();

        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();

        // look for the tags: <property index="X" type="XXXX" values="XXXX"/>
        XPathExpression expr = xpath.compile("//property[@index]");
        Object result = expr.evaluate(doc, XPathConstants.NODESET);
        NodeList nodes = (NodeList) result;
        for (int i = 0; i < nodes.getLength(); i++)
        {
            String[] triple = new String[TRIPLE];
            for (int j = 0; j < Math.min(TRIPLE, nodes.item(i).getAttributes().getLength()); j++)
            {
                Node node = (Node) nodes.item(i).getAttributes().item(j);
                triple[j] = node.getNodeValue();
            }
            properties.add(new AndsPidResponseProperty(Integer.valueOf(triple[0]), triple[1], triple[2]));
        }
        return properties;
    }

    /**
     * Helper method for converting formatted XML response into an XML document.
     * 
     * @param xmlResponse
     *            a formatted XML response.
     * @return an XML document.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    private static Document parseXML(String xmlResponse) throws ParserConfigurationException, SAXException, IOException
    {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        DocumentBuilder builder = factory.newDocumentBuilder();
        StringReader reader = new StringReader(xmlResponse);
        InputSource inputSource = new InputSource(reader);
        Document doc = builder.parse(inputSource);
        reader.close();
        return doc;
    }

    /**
     * Populate a AndsPidResponse object from the XML response of an ANDS Persistent Identifier service call.
     * 
     * @param xmlResponse
     *            the raw xmlResponse from an ANDS Persistent Identifier service call.
     * @return an ANDS Persistent Identifier service response {@link AndsPidResponse}.
     * @throws XPathExpressionException
     *             thrown when attempting to execute XPath on XML response.
     * @throws ParserConfigurationException
     *             thrown when attempting to convert response to an XML document.
     * @throws SAXException
     *             thrown when attempting to convert response to an XML document.
     * @throws IOException
     *             thrown when attempting to read response.
     */
    private AndsPidResponse populateAndsPidResponse(String xmlResponse) throws XPathExpressionException,
            ParserConfigurationException, SAXException, IOException
    {
        AndsPidResponse response = new AndsPidResponse();
        response.setXmlResponse(xmlResponse);
        
        Document doc = parseXML(xmlResponse);
        response.setMessage(parseForMessage(doc));
        response.setSuccess(parseForSuccess(doc));
        response.setHandles(parseForHandles(doc));
        response.setProperties(parseForProperties(doc));
        return response;
    }

    /**
     * Checks that this class is in a valid state to call the ANDS PID service.
     * 
     * @throws IllegalStateException
     *             thrown if the parameters need to call the ANDS PID service have not been provided.
     */
    private void validateState() throws IllegalStateException
    {
        StringBuffer errorMsg = new StringBuffer();
        if (StringUtils.isEmpty(this.getPidServiceHost()))
        {
            errorMsg.append("The host name of the ANDS PID service has not been provided. e.g. test.org.au\n");
        }
        if (getRequestorIdentity() == null || StringUtils.isEmpty(this.getRequestorIdentity().getAppId()))
        {
            errorMsg
                    .append("The appID of the caller has not been provided. " +
                            "e.g. unique Id provided by ANDS upon IP registration.\n");
        }
        if (getRequestorIdentity() == null || StringUtils.isEmpty(this.getRequestorIdentity().getIdentifier()))
        {
            errorMsg
                    .append("The identifier of the caller has not been provided. " +
                            "e.g. identifier or name of the repository calling the service.\n");
        }
        if (getRequestorIdentity() == null || StringUtils.isEmpty(this.getRequestorIdentity().getAuthDomain()))
        {
            errorMsg
                    .append("The authDomain of the caller has not been provided. " +
                            "e.g. the domain of the organisation calling the service.");
        }
        // if we have error messages, throw the exception
        if (errorMsg.length() != 0)
        {
            throw new IllegalStateException(errorMsg.toString());
        }
    }
    

    /**
     * Checks the arguments passed to the {@link #mintHandle(HandleType, String)} and the
     * {@link #mintHandleByIndex(HandleType, int, String)} methods.
     * 
     * @param type
     *            the type of the value to be minted {@link HandleType}.
     * @param value
     *            the value to be minted.
     * @throws IllegalStateException
     *             thrown if the arguments provided to {@link #mintHandle(HandleType, String)} or to
     *             {@link #mintHandleByIndex(HandleType, int, String)} are not valid.
     */
    private void validateMintHandleArguments(HandleType type, String value) throws IllegalArgumentException
    {
        // we should never accept a null HandleType
        if (type == null)
        {
            throw new IllegalArgumentException(
                    "The method mintHandle() can not be called with a null HandleType argument.\n");
        }

        // we can only accept the arguments if and only if both arguments are empty or both arguments are not empty
        // (logical equality).
        if (type.isEmpty() != StringUtils.isEmpty(value))
        {
            throw new IllegalArgumentException(MessageFormat.format(
                    "The method mintHandle() can only be called if both arguments are empty "
                            + "or both arguments are not empty:\n type= {0}\n value={1}\n", new Object[] {
                            (type == null) ? null : type.value(), value }));
        }
    }

    /**
     * Retrieve the current ANDS Persistent Identifier host name
     * @return the ANDS Persistent Identifier host name
     */
    public String getPidServiceHost()
    {
        return pidServiceHost;
    }

    /**
     * Set the ANDS Persistent Identifier host name.
     * @param pidServiceHost
     *            the ANDS Persistent Identifier host name to set
     */
    public void setPidServiceHost(String pidServiceHost)
    {
        this.pidServiceHost = pidServiceHost;
    }

    /**
     * Retrieve the ANDS Persistent Identifier port number
     * @return the ANDS Persistent Identifier port number
     */
    public int getPidServicePort()
    {
        return pidServicePort;
    }

    /**
     * Set the ANDS Persistent Identifier port number
     * @param pidServicePort
     *            the ANDS Persistent Identifier port number to set
     */
    public void setPidServicePort(int pidServicePort)
    {
        this.pidServicePort = pidServicePort;
    }

    /**
     * Retrieve the current ANDS Persistent Identifier path name (web application context name)
     * @return the ANDS Persistent Identifier path name
     */
    public String getPidServicePath()
    {
        return pidServicePath;
    }

    /**
     * Set the current ANDS Persistent Identifier path name (web application context name)
     * @param pidServicePath
     *            the ANDS Persistent Identifier path name to set
     */
    public void setPidServicePath(String pidServicePath)
    {
        this.pidServicePath = pidServicePath;
    }

    /**
     * Retrieve the identity information of the calling application/organisation.
     * @return the identity of the caller
     */
    public AndsPidIdentity getRequestorIdentity()
    {
        return requestorIdentity;
    }

    /**
     * Set the identity information of the calling application/organisation.
     * @param requestorIdentity
     *            the identity object to set
     */
    public void setRequestorIdentity(AndsPidIdentity requestorIdentity)
    {
        this.requestorIdentity = requestorIdentity;
    }

}