package net.scattersphere.data.message;

import net.scattersphere.data.DataSerializer;

import java.io.Serializable;

/**
 * This is a packet message that is sent when a job status is requested.
 *
 * Created by kenji on 1/25/15.
 */
public class JobResponseMessage implements Serializable {

    private static final long serialVersionUID = 8965863129042005132L;

    private final String jobId;
    private final String jobDescription;
    private final String jobStatus;
    private final String jobMessage;
    private final Exception jobException;

    /**
     * Constructor.
     *
     * @param jobId The Job ID.
     * @param jobStatus The status of the job.
     * @param jobMessage The job interruption message, if one occurred.
     * @param jobException The {@code Exception} for the job if one occurred.
     */
    public JobResponseMessage(String jobId, String jobStatus, String jobMessage, Exception jobException) {
        this.jobId = jobId;
        this.jobStatus = jobStatus;
        this.jobMessage = jobMessage;
        this.jobException = jobException;
        this.jobDescription = null;
    }

    /**
     * Constructor that takes the job ID and status.
     *
     * @param jobId The Job ID.
     * @param jobStatus The status of the job.
     */
    public JobResponseMessage(String jobId, String jobStatus) {
        this(jobId, jobStatus, null, null);
    }

    public JobResponseMessage(String jobId, String jobStatus, String jobDescription) {
        this.jobId = jobId;
        this.jobStatus = jobStatus;
        this.jobMessage = null;
        this.jobException = null;
        this.jobDescription = jobDescription;
    }

    /**
     * Retrieves the job ID.
     *
     * @return {@link String} containing the job ID.
     */
    public String getJobId() {
        return jobId;
    }

    /**
     * Returns the description of the job.
     *
     * @return {@link String} containing the job description.
     */
    public String getJobDescription() {
        return jobDescription;
    }

    /**
     * Retrieves the job status.
     *
     * @return {@link String} containing the status of the job.
     */
    public String getJobStatus() {
        return jobStatus;
    }

    /**
     * Retrieves the job cancellation/interruption message if one exists.
     *
     * @return {@link String} containing the job interruption message.
     */
    public String getJobMessage() {
        return jobMessage;
    }

    /**
     * Returns an {@link Exception}, if one occurred for this job.
     *
     * @return {@link Exception} for the job, {@code null} if not set.
     */
    public Exception getJobException() {
        return jobException;
    }

    /**
     * Returns a serialized version of this class.
     *
     * @return {@code byte[]} array containing the serialized payload.
     */
    public byte[] toByteArray() {
        return DataSerializer.serialize(this);
    }

    /**
     * Deserializes the payload sent, returning a new {@link net.scattersphere.data.message.JobResponseMessage} on success.
     *
     * @param array {@code byte[]} array of data to deserialize.
     * @return {@link net.scattersphere.data.message.JobResponseMessage} on success, {@code null} otherwise.
     */
    public static JobResponseMessage fromByteArray(byte[] array) {
        return (JobResponseMessage) DataSerializer.deserialize(array);
    }

}
