/*
 * Scattersphere
 * Copyright 2014-2015, Scattersphere Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.scattersphere.data.message;

import net.scattersphere.data.DataSerializer;
import org.nustaq.serialization.FSTObjectInput;
import org.nustaq.serialization.FSTObjectOutput;

import javax.xml.crypto.Data;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Serializable;

/**
 * This class is used to indicate that a job needs to be started, or stopped on a server.
 *
 * Created by kenji on 12/5/14.
 */
public class JobParametersMessage implements Serializable {

    private final String jobName;
    private final String jobMessage;
    private final String[] jobParameters;

    /**
     * Constructor.
     *
     * @param jobName The name of the job.
     * @param jobMessage The job reason message for stopping, if applicable.
     * @param jobParameters The {@code String} array of parameters to send to the job.
     */
    public JobParametersMessage(String jobName, String jobMessage, String[] jobParameters) {
        this.jobName = jobName;
        this.jobMessage = jobMessage;
        this.jobParameters = jobParameters;
    }

    /**
     * Constructor that sets the job name and its parameters.
     *
     * @param jobName The name of the job.
     * @param jobParameters The {@code String} array of parameters to send to the job.
     */
    public JobParametersMessage(String jobName, String[] jobParameters) {
        this(jobName, null, jobParameters);
    }

    /**
     * Returns the name of the job.
     *
     * @return {@code String} containing the job name.
     */
    public String getJobName() {
        return jobName;
    }

    /**
     * Retrieves the job reason message, if set.
     *
     * @return {@code String} containing the job message.
     */
    public String getJobMessage() {
        return jobMessage;
    }

    /**
     * Retrieves the list of parameters set.
     *
     * @return {@code String[]} array containing the parameters list, {@code null} if not set.
     */
    public String[] getJobParameters() {
        return jobParameters;
    }

    /**
     * Returns a serialized version of this class.
     *
     * @return {@code byte[]} array containing the serialized payload.
     */
    public byte[] toByteArray() {
        return DataSerializer.serialize(this);
    }

    /**
     * Deserializes the payload sent, returning a new {@link net.scattersphere.data.message.JobParametersMessage} on success.
     *
     * @param array {@code byte[]} array of data to deserialize.
     * @return {@link net.scattersphere.data.message.JobParametersMessage} on success, {@code null} otherwise.
     */
    public static JobParametersMessage fromByteArray(byte[] array) {
        return (JobParametersMessage) DataSerializer.deserialize(array);
    }

    @Override
    public String toString() {
        StringBuilder out = new StringBuilder("JobParametersMessage: jobName=");

        out.append(jobName)
           .append(";jobMessage=")
           .append(jobMessage)
           .append(";jobParameters=")
           .append(jobParameters);

        return out.toString();
    }

}
