/*
 * Scattersphere
 * Copyright 2014-2015, Scattersphere Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.scattersphere.data.message;

import net.scattersphere.data.DataSerializer;
import org.nustaq.serialization.FSTObjectInput;
import org.nustaq.serialization.FSTObjectOutput;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Serializable;

/**
 * This is a transport message class that is used by Scattersphere to send messages between servers and clients.  It contains
 * a message, an action, and a payload.
 *
 * Created by kenji on 11/30/14.
 */
public class JobMessage implements Serializable {

    private final String message;
    private final String action;
    private final byte[] payload;

    /**
     * Constructor.
     *
     * @param message The message to store.
     * @param action The action associated with the message.
     * @param payload The serialized payload to send.
     */
    public JobMessage(String message, String action, byte[] payload) {
        this.message = message;
        this.action = action;
        this.payload = payload;
    }

    /**
     * Constructor that only stores a message.
     *
     * @param message The message to store.
     */
    public JobMessage(String message) {
        this(message, null, null);
    }

    /**
     * Constructor that takes a message and action with no payload.
     *
     * @param message The message to store.
     * @param action The action associated with the message.
     */
    public JobMessage(String message, String action) {
        this(message, action, null);
    }

    /**
     * Constructor that takes a message and payload with no action.
     *
     * @param message The message to store.
     * @param payload The payload to attach.
     */
    public JobMessage(String message, byte[] payload) {
        this(message, null, payload);
    }

    /**
     * Retrieves the message.
     *
     * @return {@code String} containing the message.
     */
    public String getMessage() {
        return message;
    }

    /**
     * Retrieves the action.
     *
     * @return {@code String} containing the action.
     */
    public String getAction() {
        return action;
    }

    /**
     * Retrieves the payload.
     *
     * @return {@code byte[]} containing the payload of data.
     */
    public byte[] getPayload() {
        return payload;
    }

    /**
     * Serializes this object.
     *
     * @return {@code byte[]} containing the serialized data.
     */
    public byte[] toByteArray() {
        return DataSerializer.serialize(this);
    }

    /**
     * Deserializes a {@link JobMessage} object from a payload.
     *
     * @param array {@code byte[]} of data to deserialize.
     * @return {@link JobMessage} after deserialization, {@code null} if it could not be deserialized.
     */
    public static JobMessage fromByteArray(byte[] array) {
        return (JobMessage) DataSerializer.deserialize(array);
    }

    @Override
    public String toString() {
        StringBuilder response = new StringBuilder("JobMessage: message=");

        response.append(message)
                .append(";action=")
                .append(action)
                .append(";payload=")
                .append(payload == null ? 0 : payload.length)
                .append(" bytes");

        return response.toString();
    }

}
