/*
 * Scattersphere
 * Copyright 2014-2015, Scattersphere Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.scattersphere.data;

import org.nustaq.serialization.FSTObjectInput;
import org.nustaq.serialization.FSTObjectOutput;
import org.nustaq.serialization.util.FSTInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.Objects;

/**
 * This is a helper class that (de)serializes data that is sent and received over the wire.  It includes a packetize
 * function, which forms a packet with an expected packet size at the header of the message.
 *
 * Created by kenji on 11/22/14.
 */
public class DataSerializer {

    /**
     * Serializes a {@link java.io.Serializable} object using FST.
     *
     * @param obj The {@link java.io.Serializable} object to serialize.
     * @return {@code byte[]} array of data.
     */
    public static byte[] serialize(Object obj) {
        Objects.requireNonNull(obj);

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        FSTObjectOutput foo = new FSTObjectOutput(baos);

        try {
            foo.writeObject(obj);
            foo.close();
            baos.close();
        } catch(Exception ex) {
            return null;
        }

        return baos.toByteArray();
    }

    /**
     * Deserializes a {@code byte[]} buffer into a {@link java.io.Serializable} obhect.
     *
     * @param data The {@code byte[]} buffer to deserialize.
     * @return {@link Object}.
     */
    public static Object deserialize(byte[] data) {
        Objects.requireNonNull(data);

        ByteArrayInputStream bais = new ByteArrayInputStream(data);
        FSTObjectInput ois = null;

        try {
            ois = new FSTObjectInput(bais);

            Object obj = ois.readObject();

            ois.close();
            bais.close();

            return obj;
        } catch(Exception ex) {
            Logger LOG = LoggerFactory.getLogger(DataSerializer.class);

            LOG.warn("Deserialize failed.", ex);
        }

        return null;
    }

    /**
     * Packetizes a {@code byte[]} array of data to send over the wire.  This format is (packet size)(bytes[]).
     *
     * @param data {@code byte[]} array to packetize.
     * @return {@code byte[]} array containing the size of the payload in the header.
     */
    public static byte[] packetize(byte[] data) {
        Objects.requireNonNull(data);

        byte[] copiedData = new byte[data.length + 4];

        copiedData[0] = (byte)(0xff & (data.length >> 24));
        copiedData[1] = (byte)(0xff & (data.length >> 16));
        copiedData[2] = (byte)(0xff & (data.length >> 8));
        copiedData[3] = (byte)(0xff & data.length);
        System.arraycopy(data, 0, copiedData, 4, data.length);

        return copiedData;
    }

}
