package net.scattersphere.api;

import net.scattersphere.data.DataSerializer;
import net.scattersphere.data.message.JobMessage;
import net.scattersphere.data.message.JobParametersMessage;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * This API handles the action of subscribing to a stream by job ID to servers.  It handles the retrieval of data from a
 * job in real time, in packets at a time.
 *
 * <p>Packets are only sent to the {@code onStream} function as they are received in full.  This API will buffer all data
 * from a connection.
 *
 * Created by kenji on 2/8/15.
 */
public class StreamClient {

    private Consumer<String> onConnect;
    private Consumer<String> onDisconnect;
    private BiConsumer<String, byte[]> onStream;

    private final org.slf4j.Logger LOG = LoggerFactory.getLogger(StreamClient.class);

    /**
     * This is a storage class used to store connections to servers.
     */
    private class StreamClientConnectionPair {
        private String host;
        private String pairName;
        private ClientConnection client;

        public StreamClientConnectionPair(String pairName, String host) {
            this.pairName = pairName;
            this.host = host;
            this.client = new ClientConnection(true);
        }

        public String getHost() {
            return host;
        }

        public void connect() {
            client.onConnect(x -> {
                if (x == ClientConnectionState.CONNECTED) {
                    onConnect.accept(pairName);
                    LOG.info("New connection state: {}", x);
                }

                if (x == ClientConnectionState.CLOSED) {
                    onDisconnect.accept(pairName);
                    LOG.info("Connection closed: {}", x);
                }

                if (x == ClientConnectionState.FAILED) {
                    onDisconnect.accept(pairName);
                    LOG.info("Connection failed: {}", host);
                }
            })
            .streamReceived(msg -> onStream.accept(pairName, msg))
            .messageReceived(msg -> LOG.info("Unexpected message received: {}", msg))
            .connect(host);
        }

        public ClientConnection getClientConnection() {
            return client;
        }
    }

    private Map<String, StreamClientConnectionPair> connections;

    public StreamClient() {
        this.connections = new HashMap<>();
        this.onConnect = null;
    }

    /**
     * Adds a connection to a server, assigning a name to that connection.
     *
     * @param clientName The name of the connection to assign.
     * @param clientAddress The address to connect to.
     */
    public void addClient(String clientName, String clientAddress) {
        Objects.requireNonNull(clientName);
        Objects.requireNonNull(clientAddress);

        StreamClientConnectionPair clientConnectionPair = new StreamClientConnectionPair(clientName, clientAddress);

        clientConnectionPair.connect();

        connections.put(clientName, clientConnectionPair);
    }

    /**
     * Retrieves the underlying {@link ClientConnection}, which is the communication layer to the endpoint.  This is available
     * for use if required.
     *
     * @param clientName The name of the connection to retrieve.
     * @return {@link ClientConnection} if found.
     */
    public ClientConnection getClientConnection(String clientName) {
        Objects.requireNonNull(clientName);

        if (connections.get(clientName) == null) {
            throw new NullPointerException("Connection for client by name does not exist: " + clientName);
        }

        return connections.get(clientName).getClientConnection();
    }

    /**
     * Subscribes to a stream by ID.
     *
     * @param clientName The client connection to connect to.
     * @param streamId {@code String} containing the stream ID.
     */
    public void openStream(String clientName, String streamId) {
        JobParametersMessage parameters = new JobParametersMessage(streamId, "open", new String[0] );
        JobMessage message = new JobMessage(JobMessage.STREAM_COMMAND, parameters.toByteArray());

        getClientConnection(clientName).sendMessage(DataSerializer.packetize(message.toByteArray()));
    }

    /**
     * Unsubscribes from a stream by ID.
     *
     * @param clientName The client connection to connect to.
     * @param streamId {@code String} containing the stream ID.
     */
    public void closeStream(String clientName, String streamId) {
        JobParametersMessage parameters = new JobParametersMessage(streamId, "close", new String[0] );
        JobMessage message = new JobMessage(JobMessage.STREAM_COMMAND, parameters.toByteArray());

        getClientConnection(clientName).sendMessage(DataSerializer.packetize(message.toByteArray()));
    }

    /**
     * Lambda function called when a connection is successful.
     *
     * @param function The {@link Consumer} function to set.
     */
    public void onConnect(Consumer<String> function) {
        this.onConnect = function;
    }

    /**
     * Lambda function called when a connection is lost, or dropped.
     *
     * @param function The {@link Consumer} function to set.
     */
    public void onDisconnect(Consumer<String> function) {
        this.onDisconnect = function;
    }

    /**
     * Lambda function called when a stream packet is receive from a server.  If a stream has closed, this function will no
     * longer be called.
     *
     * @param function The {@link BiConsumer} of {@code streamId, byte[]} data received from the server for the specified stream.
     */
    public void onStream(BiConsumer<String, byte[]> function) {
        this.onStream = function;
    }

}
