/*
 * Decompiled with CFR 0.152.
 */
package net.sandrohc.schematic4j.parser;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import net.querz.nbt.io.NamedTag;
import net.querz.nbt.tag.CompoundTag;
import net.querz.nbt.tag.DoubleTag;
import net.querz.nbt.tag.IntTag;
import net.querz.nbt.tag.ListTag;
import net.querz.nbt.tag.LongTag;
import net.querz.nbt.tag.StringTag;
import net.querz.nbt.tag.Tag;
import net.sandrohc.schematic4j.SchematicUtil;
import net.sandrohc.schematic4j.exception.ParsingException;
import net.sandrohc.schematic4j.parser.Parser;
import net.sandrohc.schematic4j.schematic.Schematic;
import net.sandrohc.schematic4j.schematic.SchematicSponge;
import net.sandrohc.schematic4j.schematic.types.SchematicBiome;
import net.sandrohc.schematic4j.schematic.types.SchematicBlock;
import net.sandrohc.schematic4j.schematic.types.SchematicBlockEntity;
import net.sandrohc.schematic4j.schematic.types.SchematicEntity;
import net.sandrohc.schematic4j.schematic.types.SchematicPosDouble;
import net.sandrohc.schematic4j.schematic.types.SchematicPosInteger;
import net.sandrohc.schematic4j.utils.TagUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SpongeSchematicParser
implements Parser {
    public static final String NBT_ROOT = "Schematic";
    public static final String NBT_VERSION = "Version";
    public static final String NBT_DATA_VERSION = "DataVersion";
    public static final String NBT_METADATA = "Metadata";
    public static final String NBT_METADATA_NAME = "Name";
    public static final String NBT_METADATA_AUTHOR = "Author";
    public static final String NBT_METADATA_DATE = "Date";
    public static final String NBT_METADATA_REQUIRED_MODS = "RequiredMods";
    public static final String NBT_WIDTH = "Width";
    public static final String NBT_HEIGHT = "Height";
    public static final String NBT_LENGTH = "Length";
    public static final String NBT_OFFSET = "Offset";
    public static final String NBT_PALETTE = "Palette";
    public static final String NBT_PALETTE_MAX = "PaletteMax";
    public static final String NBT_BLOCK_DATA = "BlockData";
    public static final String NBT_BLOCK_ENTITIES = "BlockEntities";
    public static final String NBT_BLOCK_ENTITIES_ID = "Id";
    public static final String NBT_BLOCK_ENTITIES_POS = "Pos";
    public static final String NBT_TILE_ENTITIES = "TileEntities";
    public static final String NBT_BIOME_PALETTE = "BiomePalette";
    public static final String NBT_BIOME_PALETTE_MAX = "BiomePaletteMax";
    public static final String NBT_BIOME_DATA = "BiomeData";
    public static final String NBT_ENTITIES = "Entities";
    public static final String NBT_ENTITIES_ID = "Id";
    public static final String NBT_ENTITIES_POS = "Pos";
    private static final Logger log = LoggerFactory.getLogger(SpongeSchematicParser.class);

    @Override
    public Schematic parse(NamedTag root) throws ParsingException {
        log.debug("Parsing Sponge schematic");
        CompoundTag rootTag = (CompoundTag)root.getTag();
        SchematicSponge.Builder builder = new SchematicSponge.Builder();
        int version = TagUtils.getIntOrThrow(rootTag, NBT_VERSION);
        builder.version(version);
        if (version > 2) {
            log.warn("Sponge Schematic version {} is not officially supported. Use at your own risk", (Object)version);
        }
        this.parseDataVersion(rootTag, builder, version);
        this.parseMetadata(rootTag, builder);
        this.parseOffset(rootTag, builder);
        this.parseBlocks(rootTag, builder);
        this.parseBlockEntities(rootTag, builder, version);
        this.parseEntities(rootTag, builder);
        this.parseBiomes(rootTag, builder);
        return builder.build();
    }

    private void parseDataVersion(CompoundTag root, SchematicSponge.Builder builder, int version) throws ParsingException {
        log.trace("Parsing data version");
        if (version == 1) {
            TagUtils.getInt(root, NBT_DATA_VERSION).ifPresent(builder::dataVersion);
        } else {
            builder.dataVersion(TagUtils.getIntOrThrow(root, NBT_DATA_VERSION));
        }
    }

    private void parseMetadata(CompoundTag root, SchematicSponge.Builder builder) {
        log.trace("Parsing metadata");
        String name = null;
        String author = null;
        LocalDateTime date = null;
        String[] requiredMods = new String[]{};
        LinkedHashMap<String, Object> extra = new LinkedHashMap<String, Object>();
        Optional<CompoundTag> metadataTag = TagUtils.getCompound(root, NBT_METADATA);
        if (metadataTag.isPresent()) {
            block12: for (Map.Entry entry : metadataTag.get()) {
                String key = (String)entry.getKey();
                Tag tag = (Tag)entry.getValue();
                switch (key) {
                    case "Name": {
                        name = ((StringTag)tag).getValue();
                        continue block12;
                    }
                    case "Author": {
                        author = ((StringTag)tag).getValue();
                        continue block12;
                    }
                    case "Date": {
                        long dateEpochMillis = ((LongTag)tag).asLong();
                        date = LocalDateTime.ofInstant(Instant.ofEpochMilli(dateEpochMillis), ZoneId.systemDefault());
                        continue block12;
                    }
                    case "RequiredMods": {
                        ListTag stringTags = ((ListTag)tag).asStringTagList();
                        requiredMods = (String[])StreamSupport.stream(stringTags.spliterator(), false).map(StringTag::getValue).toArray(String[]::new);
                        continue block12;
                    }
                }
                extra.put(key, SchematicUtil.unwrap(tag));
            }
        } else {
            log.debug("No metadata found");
        }
        builder.metadata(new SchematicSponge.Metadata(name, author, date, requiredMods, extra));
    }

    private void parseOffset(CompoundTag root, SchematicSponge.Builder builder) {
        log.trace("Parsing offset");
        TagUtils.getIntArray(root, NBT_OFFSET).ifPresent(offset -> {
            builder.offset((int[])offset);
            log.debug("Loaded offset is {}", offset);
        });
    }

    private void parseBlocks(CompoundTag root, SchematicSponge.Builder builder) throws ParsingException {
        log.trace("Parsing blocks");
        if (!SchematicUtil.containsAllTags(root, NBT_BLOCK_DATA, NBT_PALETTE)) {
            log.trace("Did not have block data");
            builder.blocks(new SchematicBlock[0][0][0]);
            return;
        }
        short width = TagUtils.getShortOrThrow(root, NBT_WIDTH);
        short height = TagUtils.getShortOrThrow(root, NBT_HEIGHT);
        short length = TagUtils.getShortOrThrow(root, NBT_LENGTH);
        builder.width(width).height(height).length(length);
        log.trace("Dimensions: width={}, height={}, length={}", new Object[]{width, height, length});
        CompoundTag palette = TagUtils.getCompoundOrThrow(root, NBT_PALETTE);
        log.trace("Palette size: {}", (Object)palette.size());
        Map<Integer, SchematicBlock> blockById = palette.entrySet().stream().collect(Collectors.toMap(entry -> ((IntTag)entry.getValue()).asInt(), entry -> new SchematicBlock((String)entry.getKey())));
        int paletteMax = root.getInt(NBT_PALETTE_MAX);
        if (palette.size() != paletteMax) {
            log.warn("Palette actual size does not match expected size. Expected {} but got {}", (Object)paletteMax, (Object)palette.size());
        }
        byte[] blockDataRaw = TagUtils.getByteArrayOrThrow(root, NBT_BLOCK_DATA);
        SchematicBlock[][][] blockData = new SchematicBlock[width][height][length];
        int expectedBlocks = width * height * length;
        if (blockDataRaw.length != expectedBlocks) {
            log.warn("Number of blocks does not match expected. Expected {} blocks, but got {}", (Object)expectedBlocks, (Object)blockDataRaw.length);
        }
        int index = 0;
        int i = 0;
        int value = 0;
        int varint_length = 0;
        while (i < blockDataRaw.length) {
            SchematicBlock block;
            value = 0;
            varint_length = 0;
            while (true) {
                value |= (blockDataRaw[i] & 0x7F) << varint_length++ * 7;
                if (varint_length > 5) {
                    throw new RuntimeException("VarInt too big (probably corrupted data)");
                }
                if ((blockDataRaw[i] & 0x80) != 128) {
                    ++i;
                    break;
                }
                ++i;
            }
            int y = index / (width * length);
            int z = index % (width * length) / width;
            int x = index % (width * length) % width;
            blockData[x][y][z] = block = blockById.get(value);
            ++index;
        }
        builder.blocks(blockData);
        log.debug("Loaded {} blocks", (Object)(width * height * length));
    }

    private void parseBlockEntities(CompoundTag root, SchematicSponge.Builder builder, int version) throws ParsingException {
        List<SchematicBlockEntity> blockEntities;
        log.trace("Parsing block entities");
        Optional<ListTag<CompoundTag>> blockEntitiesListTag = TagUtils.getCompoundList(root, version == 1 ? NBT_TILE_ENTITIES : NBT_BLOCK_ENTITIES);
        if (blockEntitiesListTag.isPresent()) {
            ListTag<CompoundTag> blockEntitiesTag = blockEntitiesListTag.get();
            blockEntities = new ArrayList<SchematicBlockEntity>(blockEntitiesTag.size());
            for (CompoundTag blockEntity : blockEntitiesTag) {
                String id = TagUtils.getStringOrThrow(blockEntity, "Id");
                int[] pos = TagUtils.getIntArray(blockEntity, "Pos").orElseGet(() -> new int[]{0, 0, 0});
                Map<String, Object> extra = blockEntity.entrySet().stream().filter(tag -> !((String)tag.getKey()).equals("Id") && !((String)tag.getKey()).equals("Pos")).collect(Collectors.toMap(Map.Entry::getKey, e -> SchematicUtil.unwrap((Tag)e.getValue())));
                blockEntities.add(new SchematicBlockEntity(id, SchematicPosInteger.from(pos), extra));
            }
            log.debug("Loaded {} block entities", (Object)blockEntities.size());
        } else {
            log.trace("No block entities found");
            blockEntities = Collections.emptyList();
        }
        builder.blockEntities(blockEntities);
    }

    private void parseEntities(CompoundTag root, SchematicSponge.Builder builder) throws ParsingException {
        List<SchematicEntity> entities;
        log.trace("Parsing entities");
        Optional<ListTag<CompoundTag>> entitiesListTag = TagUtils.getCompoundList(root, NBT_ENTITIES);
        if (entitiesListTag.isPresent()) {
            ListTag<CompoundTag> entitiesTag = entitiesListTag.get();
            entities = new ArrayList<SchematicEntity>(entitiesTag.size());
            for (CompoundTag entity : entitiesTag) {
                String id = TagUtils.getStringOrThrow(entity, "Id");
                double[] pos = new double[]{0.0, 0.0, 0.0};
                TagUtils.getDoubleList(entity, "Pos").ifPresent(posTag -> {
                    pos[0] = ((DoubleTag)posTag.get(0)).asDouble();
                    pos[1] = ((DoubleTag)posTag.get(1)).asDouble();
                    pos[2] = ((DoubleTag)posTag.get(2)).asDouble();
                });
                Map<String, Object> extra = entity.entrySet().stream().filter(tag -> !((String)tag.getKey()).equals("Id") && !((String)tag.getKey()).equals("Pos")).collect(Collectors.toMap(Map.Entry::getKey, e -> SchematicUtil.unwrap((Tag)e.getValue())));
                entities.add(new SchematicEntity(id, SchematicPosDouble.from(pos), extra));
            }
            log.debug("Loaded {} entities", (Object)entities.size());
        } else {
            log.trace("No entities found");
            entities = Collections.emptyList();
        }
        builder.entities(entities);
    }

    private void parseBiomes(CompoundTag root, SchematicSponge.Builder builder) throws ParsingException {
        log.trace("Parsing biomes");
        if (!SchematicUtil.containsAllTags(root, NBT_BIOME_DATA, NBT_BIOME_PALETTE)) {
            log.trace("Did not have biome data");
            builder.biomes(new SchematicBiome[0][0]);
            return;
        }
        int width = TagUtils.getShortOrThrow(root, NBT_WIDTH);
        int length = TagUtils.getShortOrThrow(root, NBT_LENGTH);
        CompoundTag palette = TagUtils.getCompoundOrThrow(root, NBT_BIOME_PALETTE);
        log.trace("Biome palette size: {}", (Object)palette.size());
        Map<Integer, SchematicBiome> biomeById = palette.entrySet().stream().collect(Collectors.toMap(entry -> ((IntTag)entry.getValue()).asInt(), entry -> new SchematicBiome((String)entry.getKey())));
        int paletteMax = root.getInt(NBT_BIOME_PALETTE_MAX);
        if (palette.size() != paletteMax) {
            log.warn("Biome palette actual size does not match expected size. Expected {} but got {}", (Object)paletteMax, (Object)palette.size());
        }
        byte[] biomeDataRaw = TagUtils.getByteArrayOrThrow(root, NBT_BLOCK_DATA);
        SchematicBiome[][] biomeData = new SchematicBiome[width][length];
        int expectedBlocks = width * length;
        if (biomeDataRaw.length != expectedBlocks) {
            log.warn("Number of blocks does not match expected. Expected {} blocks, but got {}", (Object)expectedBlocks, (Object)biomeDataRaw.length);
        }
        for (int x = 0; x < width; ++x) {
            for (int z = 0; z < length; ++z) {
                SchematicBiome block;
                int index = x + z * width;
                int blockId = biomeDataRaw[index] & 0xFF;
                biomeData[x][z] = block = biomeById.get(blockId);
            }
        }
        builder.biomes(biomeData);
        log.debug("Loaded {} biomes", (Object)(width * length));
    }

    public String toString() {
        return "SpongeSchematicParser()";
    }
}

