package net.risesoft.service.relation.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import lombok.RequiredArgsConstructor;

import net.risesoft.entity.Y9Person;
import net.risesoft.entity.relation.Y9PersonsToGroups;
import net.risesoft.id.Y9IdGenerator;
import net.risesoft.manager.relation.Y9PersonsToGroupsManager;
import net.risesoft.repository.relation.Y9PersonsToGroupsRepository;
import net.risesoft.service.relation.Y9PersonsToGroupsService;
import net.risesoft.y9.Y9Context;
import net.risesoft.y9.pubsub.event.Y9EntityCreatedEvent;
import net.risesoft.y9.pubsub.event.Y9EntityDeletedEvent;
import net.risesoft.y9.pubsub.event.Y9EntityUpdatedEvent;
import net.risesoft.y9.util.Y9BeanUtil;

/**
 * @author dingzhaojun
 * @author qinman
 * @author mengjuhua
 * @date 2022/2/10
 */
@Transactional(value = "rsTenantTransactionManager", readOnly = true)
@Service
@RequiredArgsConstructor
public class Y9PersonsToGroupsServiceImpl implements Y9PersonsToGroupsService {

    private final Y9PersonsToGroupsManager y9PersonsToGroupsManager;

    private final Y9PersonsToGroupsRepository y9PersonsToGroupsRepository;

    @Override
    @Transactional(readOnly = false)
    public void addGroups(String personId, String[] groupIds) {
        for (int i = 0; i < groupIds.length; i++) {
            String groupId = groupIds[i];
            if (y9PersonsToGroupsRepository.findByGroupIdAndPersonId(groupId, personId).isPresent()) {
                continue;
            }
            Integer maxGroupsOrder = getNextGroupOrderByPersonId(personId);
            Integer maxPersonsOrder = getNextPersonOrderByGroupId(groupId);
            addY9PersonsToGroups(personId, groupId, maxGroupsOrder + i, maxPersonsOrder);
        }
    }

    @Override
    @Transactional(readOnly = false)
    public void addPersons(String groupId, String[] personIds) {
        for (int i = 0; i < personIds.length; i++) {
            String personId = personIds[i];
            if (y9PersonsToGroupsRepository.findByGroupIdAndPersonId(groupId, personId).isPresent()) {
                continue;
            }
            Integer maxPersonsOrder = getNextPersonOrderByGroupId(groupId);
            Integer maxGroupsOrder = getNextGroupOrderByPersonId(personId);
            addY9PersonsToGroups(personId, groupId, maxGroupsOrder, maxPersonsOrder + i);
        }
    }

    @Override
    @Transactional(readOnly = false)
    public void deleteByGroupId(String groupId) {
        y9PersonsToGroupsRepository.deleteByGroupId(groupId);
    }

    @Override
    @Transactional(readOnly = false)
    public void deleteByPersonId(String personId) {
        y9PersonsToGroupsManager.deleteByPersonId(personId);
    }

    @Override
    public List<Y9PersonsToGroups> findByGroupId(String groupId) {
        return y9PersonsToGroupsRepository.findByGroupId(groupId);
    }

    @Override
    public Integer getNextGroupOrderByPersonId(String personId) {
        return y9PersonsToGroupsRepository.findTopByPersonIdOrderByGroupOrderDesc(personId)
            .map(Y9PersonsToGroups::getGroupOrder).orElse(-1) + 1;
    }

    @Override
    public Integer getNextPersonOrderByGroupId(String groupId) {
        return y9PersonsToGroupsRepository.findTopByGroupIdOrderByPersonOrderDesc(groupId)
            .map(Y9PersonsToGroups::getPersonOrder).orElse(-1) + 1;
    }

    @Override
    public List<Y9PersonsToGroups> listByGroupId(String groupId) {
        return y9PersonsToGroupsRepository.findByGroupIdOrderByPersonOrder(groupId);
    }

    @Override
    public List<String> listGroupIdsByPersonId(String personId) {
        return y9PersonsToGroupsRepository.findByPersonIdOrderByGroupOrder(personId).stream()
            .map(Y9PersonsToGroups::getGroupId).collect(Collectors.toList());
    }

    @Override
    @Transactional(readOnly = false)
    public List<Y9PersonsToGroups> orderGroups(String personId, String[] groupIds) {
        List<Y9PersonsToGroups> personsGroupsList = new ArrayList<>();
        for (int i = 0; i < groupIds.length; i++) {

            Optional<Y9PersonsToGroups> optionalY9PersonsToGroups =
                y9PersonsToGroupsRepository.findByGroupIdAndPersonId(groupIds[i], personId);

            if (optionalY9PersonsToGroups.isPresent()) {
                Y9PersonsToGroups updatedPersonToGroups = optionalY9PersonsToGroups.get();
                Y9PersonsToGroups originalPersonToGroups = new Y9PersonsToGroups();
                Y9BeanUtil.copyProperties(updatedPersonToGroups, originalPersonToGroups);
                updatedPersonToGroups.setGroupOrder(i);
                final Y9PersonsToGroups savedPersonToGroups = y9PersonsToGroupsRepository.save(updatedPersonToGroups);

                Y9Context.publishEvent(new Y9EntityUpdatedEvent<>(originalPersonToGroups, savedPersonToGroups));

                personsGroupsList.add(savedPersonToGroups);
            }
        }
        return personsGroupsList;
    }

    @Override
    @Transactional(readOnly = false)
    public List<Y9PersonsToGroups> orderPersons(String groupId, String[] personIds) {
        List<Y9PersonsToGroups> personsGroupsList = new ArrayList<>();
        for (int i = 0; i < personIds.length; i++) {

            Optional<Y9PersonsToGroups> optionalY9PersonsToGroups =
                y9PersonsToGroupsRepository.findByGroupIdAndPersonId(groupId, personIds[i]);

            if (optionalY9PersonsToGroups.isPresent()) {
                Y9PersonsToGroups updatedPersonToGroups = optionalY9PersonsToGroups.get();
                Y9PersonsToGroups originalPersonToGroups = new Y9PersonsToGroups();
                Y9BeanUtil.copyProperties(updatedPersonToGroups, originalPersonToGroups);
                updatedPersonToGroups.setPersonOrder(i);
                final Y9PersonsToGroups savedPersonToGroups = y9PersonsToGroupsRepository.save(updatedPersonToGroups);

                Y9Context.publishEvent(new Y9EntityUpdatedEvent<>(originalPersonToGroups, savedPersonToGroups));

                personsGroupsList.add(savedPersonToGroups);
            }
        }
        return personsGroupsList;
    }

    @Override
    @Transactional(readOnly = false)
    public void removeGroups(String personId, String[] groupIds) {
        for (String groupId : groupIds) {
            remove(personId, groupId);
        }
    }

    @Override
    @Transactional(readOnly = false)
    public void removePersons(String groupId, String[] personIds) {
        for (String personId : personIds) {
            remove(personId, groupId);
        }
    }

    @Transactional(readOnly = false)
    public Y9PersonsToGroups addY9PersonsToGroups(String personId, String groupId, Integer maxGroupsOrder,
        Integer maxPersonsOrder) {
        Y9PersonsToGroups y9PersonsToGroups = new Y9PersonsToGroups();
        y9PersonsToGroups.setId(Y9IdGenerator.genId());
        y9PersonsToGroups.setGroupId(groupId);
        y9PersonsToGroups.setPersonId(personId);
        y9PersonsToGroups.setGroupOrder(maxGroupsOrder);
        y9PersonsToGroups.setPersonOrder(maxPersonsOrder);
        y9PersonsToGroups = y9PersonsToGroupsRepository.save(y9PersonsToGroups);

        Y9Context.publishEvent(new Y9EntityCreatedEvent<>(y9PersonsToGroups));

        return y9PersonsToGroups;
    }

    @EventListener
    @Transactional(readOnly = false)
    public void onPersonDeleted(Y9EntityDeletedEvent<Y9Person> event) {
        Y9Person person = event.getEntity();
        y9PersonsToGroupsRepository.deleteByPersonId(person.getId());
    }

    @Transactional(readOnly = false)
    public void remove(String personId, String groupId) {
        Optional<Y9PersonsToGroups> optionalY9PersonsToGroups =
            y9PersonsToGroupsRepository.findByGroupIdAndPersonId(groupId, personId);
        if (optionalY9PersonsToGroups.isPresent()) {
            Y9PersonsToGroups y9PersonsToGroups = optionalY9PersonsToGroups.get();

            y9PersonsToGroupsManager.delete(y9PersonsToGroups);
        }
    }
}
