/*
 * Decompiled with CFR 0.152.
 */
package net.reini.rabbitmq.cdi;

import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.ShutdownSignalException;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import javax.annotation.PreDestroy;
import javax.enterprise.context.ApplicationScoped;
import net.reini.rabbitmq.cdi.ConnectionConfig;
import net.reini.rabbitmq.cdi.ConnectionListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ApplicationScoped
public class ConnectionProducer {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionProducer.class);
    public static final int CONNECTION_HEARTBEAT_IN_SEC = 3;
    public static final int CONNECTION_TIMEOUT_IN_MS = 1000;
    public static final int CONNECTION_ESTABLISH_INTERVAL_IN_MS = 500;
    private final Supplier<ConnectionFactory> factorySupplier;
    private final Map<ConnectionConfig, ConnectionState> connectionStates;

    public ConnectionProducer() {
        this(ConnectionFactory::new);
    }

    ConnectionProducer(Supplier<ConnectionFactory> factorySupplier) {
        this.factorySupplier = factorySupplier;
        this.connectionStates = new ConcurrentHashMap<ConnectionConfig, ConnectionState>();
    }

    public Connection getConnection(ConnectionConfig config) throws IOException, TimeoutException, NoSuchAlgorithmException {
        return this.connectionStates.computeIfAbsent(config, ConnectionState::new).getConnection(this.factorySupplier);
    }

    @PreDestroy
    public void close() {
        this.connectionStates.values().forEach(ConnectionState::close);
    }

    public void registerConnectionListener(ConnectionConfig config, ConnectionListener listener) {
        this.connectionStates.computeIfAbsent(config, ConnectionState::new).listeners().add(listener);
    }

    public void removeConnectionListener(ConnectionConfig config, ConnectionListener listener) {
        this.connectionStates.computeIfAbsent(config, ConnectionState::new).listeners().remove(listener);
    }

    static final class ConnectionState {
        private final ConnectionConfig config;
        private final Set<ConnectionListener> listeners;
        private volatile Connection connection;
        private volatile State state;

        ConnectionState(ConnectionConfig config) {
            this.config = config;
            this.state = State.NEVER_CONNECTED;
            this.listeners = ConcurrentHashMap.newKeySet();
        }

        Set<ConnectionListener> listeners() {
            return this.listeners;
        }

        void changeState(State newState) {
            this.state = newState;
            this.notifyListenersOnStateChange();
        }

        void notifyListenersOnStateChange() {
            LOGGER.debug("Notifying connection listeners about state change to {}", (Object)this.state);
            for (ConnectionListener listener : this.listeners) {
                switch (this.state) {
                    case CONNECTED: {
                        listener.onConnectionEstablished(this.connection);
                        break;
                    }
                    case CONNECTING: {
                        listener.onConnectionLost(this.connection);
                        break;
                    }
                    case CLOSED: {
                        listener.onConnectionClosed(this.connection);
                        break;
                    }
                }
            }
        }

        Connection getConnection(Supplier<ConnectionFactory> factorySupplier) throws IOException, TimeoutException, NoSuchAlgorithmException {
            if (this.state == State.CLOSED) {
                throw new IOException("Attempt to retrieve a connection from a closed connection factory");
            }
            if (this.state == State.NEVER_CONNECTED) {
                this.establishConnection(factorySupplier);
            }
            if (this.connection.isOpen()) {
                return this.connection;
            }
            LOGGER.error("Unable to retrieve connection");
            throw new IOException("Unable to retrieve connection");
        }

        synchronized void establishConnection(Supplier<ConnectionFactory> factorySupplier) throws IOException, TimeoutException, NoSuchAlgorithmException {
            if (this.state == State.CLOSED) {
                throw new IOException("Attempt to establish a connection with a closed connection factory");
            }
            if (this.state == State.CONNECTED) {
                LOGGER.warn("Establishing new connection although a connection is already established");
            }
            LOGGER.debug("Trying to establish connection using {}", (Object)this.config);
            ConnectionFactory connectionFactory = factorySupplier.get();
            connectionFactory.setRequestedHeartbeat(3);
            connectionFactory.setConnectionTimeout(1000);
            this.connection = this.config.createConnection(connectionFactory);
            this.connection.addShutdownListener(cause -> this.shutdownCompleted(cause, factorySupplier));
            LOGGER.debug("Established connection successfully");
            this.changeState(State.CONNECTED);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void shutdownCompleted(ShutdownSignalException cause, Supplier<ConnectionFactory> factorySupplier) {
            if (!cause.isHardError()) {
                return;
            }
            ConnectionState connectionState = this;
            synchronized (connectionState) {
                if (this.state == State.CLOSED || this.state == State.CONNECTING) {
                    return;
                }
                this.changeState(State.CONNECTING);
            }
            LOGGER.error("Connection lost");
            int attemptInterval = 500;
            while (this.state == State.CONNECTING) {
                try {
                    this.establishConnection(factorySupplier);
                    return;
                }
                catch (IOException | NoSuchAlgorithmException | TimeoutException e) {
                    LOGGER.debug("Next reconnect attempt in {} ms", (Object)attemptInterval);
                    try {
                        Thread.sleep(attemptInterval);
                        if (attemptInterval >= 600000) continue;
                        attemptInterval = 2 * attemptInterval;
                    }
                    catch (InterruptedException ie) {
                        return;
                    }
                }
            }
        }

        synchronized void close() {
            if (this.state == State.CLOSED) {
                LOGGER.warn("Attempt to close connection factory which is already closed");
                return;
            }
            LOGGER.info("Closing connection factory");
            if (this.connection != null) {
                try {
                    this.connection.close();
                    this.connection = null;
                }
                catch (IOException e) {
                    if (!this.connection.isOpen()) {
                        LOGGER.warn("Attempt to close an already closed connection");
                    }
                    LOGGER.error("Unable to close current connection", (Throwable)e);
                }
            }
            this.changeState(State.CLOSED);
            LOGGER.info("Closed connection factory");
        }
    }

    static enum State {
        NEVER_CONNECTED,
        CONNECTING,
        CONNECTED,
        CLOSED;

    }
}

