/*
 * Decompiled with CFR 0.152.
 */
package net.minestom.server.extensions;

import com.google.gson.Gson;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import net.minestom.dependencies.DependencyGetter;
import net.minestom.dependencies.ResolvedDependency;
import net.minestom.dependencies.maven.MavenRepository;
import net.minestom.server.ServerProcess;
import net.minestom.server.extensions.DiscoveredExtension;
import net.minestom.server.extensions.Extension;
import net.minestom.server.extensions.ExtensionClassLoader;
import net.minestom.server.utils.validate.Check;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExtensionManager {
    public static final Logger LOGGER = LoggerFactory.getLogger(ExtensionManager.class);
    public static final String INDEV_CLASSES_FOLDER = "minestom.extension.indevfolder.classes";
    public static final String INDEV_RESOURCES_FOLDER = "minestom.extension.indevfolder.resources";
    private static final Gson GSON = new Gson();
    private final ServerProcess serverProcess;
    private final Map<String, Extension> extensions = new LinkedHashMap<String, Extension>();
    private final Map<String, Extension> immutableExtensions = Collections.unmodifiableMap(this.extensions);
    private final File extensionFolder = new File(System.getProperty("minestom.extension.folder", "extensions"));
    private final File dependenciesFolder = new File(this.extensionFolder, ".libs");
    private Path extensionDataRoot = this.extensionFolder.toPath();
    private State state = State.NOT_STARTED;

    public ExtensionManager(ServerProcess serverProcess) {
        this.serverProcess = serverProcess;
    }

    public boolean shouldLoadOnStartup() {
        return this.state != State.DO_NOT_START;
    }

    public void setLoadOnStartup(boolean loadOnStartup) {
        Check.stateCondition(this.state.ordinal() > State.NOT_STARTED.ordinal(), "Extensions have already been initialized");
        this.state = loadOnStartup ? State.NOT_STARTED : State.DO_NOT_START;
    }

    @NotNull
    public File getExtensionFolder() {
        return this.extensionFolder;
    }

    @NotNull
    public Path getExtensionDataRoot() {
        return this.extensionDataRoot;
    }

    public void setExtensionDataRoot(@NotNull Path dataRoot) {
        this.extensionDataRoot = dataRoot;
    }

    @NotNull
    public Collection<Extension> getExtensions() {
        return this.immutableExtensions.values();
    }

    @Nullable
    public Extension getExtension(@NotNull String name2) {
        return this.extensions.get(name2.toLowerCase());
    }

    public boolean hasExtension(@NotNull String name2) {
        return this.extensions.containsKey(name2);
    }

    @ApiStatus.Internal
    public void start() {
        if (this.state == State.DO_NOT_START) {
            LOGGER.warn("Extension loadOnStartup option is set to false, extensions are therefore neither loaded or initialized.");
            return;
        }
        Check.stateCondition(this.state != State.NOT_STARTED, "ExtensionManager has already been started");
        this.loadExtensions();
        this.state = State.STARTED;
    }

    @ApiStatus.Internal
    public void gotoPreInit() {
        if (this.state == State.DO_NOT_START) {
            return;
        }
        Check.stateCondition(this.state != State.STARTED, "Extensions have already done pre initialization");
        this.extensions.values().forEach(Extension::preInitialize);
        this.state = State.PRE_INIT;
    }

    @ApiStatus.Internal
    public void gotoInit() {
        if (this.state == State.DO_NOT_START) {
            return;
        }
        Check.stateCondition(this.state != State.PRE_INIT, "Extensions have already done initialization");
        this.extensions.values().forEach(Extension::initialize);
        this.state = State.INIT;
    }

    @ApiStatus.Internal
    public void gotoPostInit() {
        if (this.state == State.DO_NOT_START) {
            return;
        }
        Check.stateCondition(this.state != State.INIT, "Extensions have already done post initialization");
        this.extensions.values().forEach(Extension::postInitialize);
        this.state = State.POST_INIT;
    }

    private void loadExtensions() {
        if (!this.extensionFolder.exists() && !this.extensionFolder.mkdirs()) {
            LOGGER.error("Could not find or create the extension folder, extensions will not be loaded!");
            return;
        }
        if (!this.dependenciesFolder.exists() && !this.dependenciesFolder.mkdirs()) {
            LOGGER.error("Could not find nor create the extension dependencies folder, extensions will not be loaded!");
            return;
        }
        List<DiscoveredExtension> discoveredExtensions = this.discoverExtensions();
        if (discoveredExtensions.isEmpty()) {
            return;
        }
        Iterator<DiscoveredExtension> extensionIterator = discoveredExtensions.iterator();
        while (extensionIterator.hasNext()) {
            DiscoveredExtension discoveredExtension = extensionIterator.next();
            try {
                discoveredExtension.createClassLoader();
            }
            catch (Exception e) {
                discoveredExtension.loadStatus = DiscoveredExtension.LoadStatus.FAILED_TO_SETUP_CLASSLOADER;
                this.serverProcess.exception().handleException(e);
                LOGGER.error("Failed to load extension {}", (Object)discoveredExtension.getName());
                LOGGER.error("Failed to load extension", e);
                extensionIterator.remove();
            }
        }
        discoveredExtensions = this.generateLoadOrder(discoveredExtensions);
        this.loadDependencies(discoveredExtensions);
        discoveredExtensions.removeIf(ext -> ext.loadStatus != DiscoveredExtension.LoadStatus.LOAD_SUCCESS);
        for (DiscoveredExtension discoveredExtension : discoveredExtensions) {
            try {
                this.loadExtension(discoveredExtension);
            }
            catch (Exception e) {
                discoveredExtension.loadStatus = DiscoveredExtension.LoadStatus.LOAD_FAILED;
                LOGGER.error("Failed to load extension {}", (Object)discoveredExtension.getName());
                this.serverProcess.exception().handleException(e);
            }
        }
    }

    public boolean loadDynamicExtension(@NotNull File jarFile) throws FileNotFoundException {
        if (!jarFile.exists()) {
            throw new FileNotFoundException("File '" + jarFile.getAbsolutePath() + "' does not exists. Cannot load extension.");
        }
        LOGGER.info("Discover dynamic extension from jar {}", (Object)jarFile.getAbsolutePath());
        DiscoveredExtension discoveredExtension = this.discoverFromJar(jarFile);
        List<DiscoveredExtension> extensionsToLoad = Collections.singletonList(discoveredExtension);
        return this.loadExtensionList(extensionsToLoad);
    }

    @Nullable
    private Extension loadExtension(@NotNull DiscoveredExtension discoveredExtension) {
        Constructor<Extension> constructor;
        Class<Extension> extensionClass;
        Class<?> jarClass;
        String extensionName = discoveredExtension.getName();
        String mainClass = discoveredExtension.getEntrypoint();
        ExtensionClassLoader loader = discoveredExtension.getClassLoader();
        if (this.extensions.containsKey(extensionName.toLowerCase())) {
            LOGGER.error("An extension called '{}' has already been registered.", (Object)extensionName);
            return null;
        }
        try {
            jarClass = Class.forName(mainClass, true, loader);
        }
        catch (ClassNotFoundException e) {
            LOGGER.error("Could not find main class '{}' in extension '{}'.", mainClass, extensionName, e);
            return null;
        }
        try {
            extensionClass = jarClass.asSubclass(Extension.class);
        }
        catch (ClassCastException e) {
            LOGGER.error("Main class '{}' in '{}' does not extend the 'Extension' superclass.", mainClass, extensionName, e);
            return null;
        }
        try {
            constructor = extensionClass.getDeclaredConstructor(new Class[0]);
            constructor.setAccessible(true);
        }
        catch (NoSuchMethodException e) {
            LOGGER.error("Main class '{}' in '{}' does not define a no-args constructor.", mainClass, extensionName, e);
            return null;
        }
        Extension extension = null;
        try {
            extension = constructor.newInstance(new Object[0]);
        }
        catch (InstantiationException e) {
            LOGGER.error("Main class '{}' in '{}' cannot be an abstract class.", mainClass, extensionName, e);
            return null;
        }
        catch (IllegalAccessException e) {
        }
        catch (InvocationTargetException e) {
            LOGGER.error("While instantiating the main class '{}' in '{}' an exception was thrown.", mainClass, extensionName, e.getTargetException());
            return null;
        }
        for (String dependencyName : discoveredExtension.getDependencies()) {
            Extension dependency = this.extensions.get(dependencyName.toLowerCase());
            if (dependency == null) {
                LOGGER.warn("Dependency {} of {} is null? This means the extension has been loaded without its dependency, which could cause issues later.", (Object)dependencyName, (Object)discoveredExtension.getName());
                continue;
            }
            dependency.getDependents().add(discoveredExtension.getName());
        }
        this.extensions.put(extensionName.toLowerCase(), extension);
        return extension;
    }

    @NotNull
    private List<DiscoveredExtension> discoverExtensions() {
        LinkedList<DiscoveredExtension> extensions = new LinkedList<DiscoveredExtension>();
        File[] fileList = this.extensionFolder.listFiles();
        if (fileList != null) {
            for (File file : fileList) {
                DiscoveredExtension extension;
                if (file.isDirectory() || !file.getName().endsWith(".jar") || (extension = this.discoverFromJar(file)) == null || extension.loadStatus != DiscoveredExtension.LoadStatus.LOAD_SUCCESS) continue;
                extensions.add(extension);
            }
        }
        if (System.getProperty(INDEV_CLASSES_FOLDER) != null && System.getProperty(INDEV_RESOURCES_FOLDER) != null) {
            LOGGER.info("Found indev folders for extension. Adding to list of discovered extensions.");
            String extensionClasses = System.getProperty(INDEV_CLASSES_FOLDER);
            String extensionResources = System.getProperty(INDEV_RESOURCES_FOLDER);
            try (InputStreamReader reader = new InputStreamReader(new FileInputStream(new File(extensionResources, "extension.json")));){
                DiscoveredExtension extension = GSON.fromJson((Reader)reader, DiscoveredExtension.class);
                extension.files.add(new File(extensionClasses).toURI().toURL());
                extension.files.add(new File(extensionResources).toURI().toURL());
                extension.setDataDirectory(this.getExtensionDataRoot().resolve(extension.getName()));
                DiscoveredExtension.verifyIntegrity(extension);
                if (extension.loadStatus == DiscoveredExtension.LoadStatus.LOAD_SUCCESS) {
                    extensions.add(extension);
                }
            }
            catch (IOException e) {
                this.serverProcess.exception().handleException(e);
            }
        }
        return extensions;
    }

    @Nullable
    private DiscoveredExtension discoverFromJar(@NotNull File file) {
        DiscoveredExtension discoveredExtension;
        ZipFile f = new ZipFile(file);
        try {
            ZipEntry entry2 = f.getEntry("extension.json");
            if (entry2 == null) {
                throw new IllegalStateException("Missing extension.json in extension " + file.getName() + ".");
            }
            InputStreamReader reader = new InputStreamReader(f.getInputStream(entry2));
            DiscoveredExtension extension = GSON.fromJson((Reader)reader, DiscoveredExtension.class);
            extension.setOriginalJar(file);
            extension.files.add(file.toURI().toURL());
            extension.setDataDirectory(this.getExtensionDataRoot().resolve(extension.getName()));
            DiscoveredExtension.verifyIntegrity(extension);
            discoveredExtension = extension;
        }
        catch (Throwable throwable) {
            try {
                try {
                    f.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                this.serverProcess.exception().handleException(e);
                return null;
            }
        }
        f.close();
        return discoveredExtension;
    }

    @NotNull
    private List<DiscoveredExtension> generateLoadOrder(@NotNull List<DiscoveredExtension> discoveredExtensions) {
        List<Map.Entry> loadableExtensions;
        HashMap dependencyMap = new HashMap();
        HashMap<String, DiscoveredExtension> extensionMap = new HashMap<String, DiscoveredExtension>();
        for (DiscoveredExtension discoveredExtension : discoveredExtensions) {
            extensionMap.put(discoveredExtension.getName().toLowerCase(), discoveredExtension);
        }
        block1: for (DiscoveredExtension discoveredExtension : discoveredExtensions) {
            ArrayList<DiscoveredExtension> dependencies = new ArrayList<DiscoveredExtension>(discoveredExtension.getDependencies().length);
            for (String dependencyName : discoveredExtension.getDependencies()) {
                DiscoveredExtension dependencyExtension = (DiscoveredExtension)extensionMap.get(dependencyName.toLowerCase());
                if (dependencyExtension == null) {
                    if (this.extensions.containsKey(dependencyName.toLowerCase())) {
                        dependencies.add(this.extensions.get(dependencyName.toLowerCase()).getOrigin());
                        continue;
                    }
                    LOGGER.error("Extension {} requires an extension called {}.", (Object)discoveredExtension.getName(), (Object)dependencyName);
                    LOGGER.error("However the extension {} could not be found.", (Object)dependencyName);
                    LOGGER.error("Therefore {} will not be loaded.", (Object)discoveredExtension.getName());
                    discoveredExtension.loadStatus = DiscoveredExtension.LoadStatus.MISSING_DEPENDENCIES;
                    continue block1;
                }
                dependencies.add(dependencyExtension);
            }
            dependencyMap.put(discoveredExtension, dependencies);
        }
        LinkedList<DiscoveredExtension> sortedList = new LinkedList<DiscoveredExtension>();
        while (!(loadableExtensions = dependencyMap.entrySet().stream().filter(entry2 -> this.isLoaded((List)entry2.getValue())).toList()).isEmpty()) {
            for (Map.Entry entry3 : loadableExtensions) {
                sortedList.add((DiscoveredExtension)entry3.getKey());
                dependencyMap.remove(entry3.getKey());
                for (List dependencies : dependencyMap.values()) {
                    dependencies.remove(entry3.getKey());
                }
            }
        }
        if (!dependencyMap.isEmpty()) {
            LOGGER.error("Minestom found {} cyclic extensions.", (Object)dependencyMap.size());
            LOGGER.error("Cyclic extensions depend on each other and can therefore not be loaded.");
            for (Map.Entry entry3 : dependencyMap.entrySet()) {
                DiscoveredExtension discoveredExtension = (DiscoveredExtension)entry3.getKey();
                LOGGER.error("{} could not be loaded, as it depends on: {}.", (Object)discoveredExtension.getName(), (Object)((List)entry3.getValue()).stream().map(DiscoveredExtension::getName).collect(Collectors.joining(", ")));
            }
        }
        return sortedList;
    }

    private boolean isLoaded(@NotNull List<DiscoveredExtension> extensions) {
        return extensions.isEmpty() || extensions.stream().allMatch(ext -> this.extensions.containsKey(ext.getName().toLowerCase()));
    }

    private void loadDependencies(@NotNull List<DiscoveredExtension> extensions) {
        LinkedList<DiscoveredExtension> allLoadedExtensions = new LinkedList<DiscoveredExtension>(extensions);
        for (Extension extension : this.immutableExtensions.values()) {
            allLoadedExtensions.add(extension.getOrigin());
        }
        for (DiscoveredExtension discoveredExtension : extensions) {
            try {
                DependencyGetter getter = new DependencyGetter();
                DiscoveredExtension.ExternalDependencies externalDependencies = discoveredExtension.getExternalDependencies();
                LinkedList<MavenRepository> repoList2 = new LinkedList<MavenRepository>();
                for (DiscoveredExtension.ExternalDependencies.Repository repository : externalDependencies.repositories) {
                    if (repository.name == null || repository.name.isEmpty()) {
                        throw new IllegalStateException("Missing 'name' element in repository object.");
                    }
                    if (repository.url == null || repository.url.isEmpty()) {
                        throw new IllegalStateException("Missing 'url' element in repository object.");
                    }
                    repoList2.add(new MavenRepository(repository.name, repository.url));
                }
                getter.addMavenResolver(repoList2);
                for (String artifact : externalDependencies.artifacts) {
                    ResolvedDependency resolved = getter.get(artifact, this.dependenciesFolder);
                    this.addDependencyFile(resolved, discoveredExtension);
                    LOGGER.trace("Dependency of extension {}: {}", (Object)discoveredExtension.getName(), (Object)resolved);
                }
                ExtensionClassLoader extensionClassLoader = discoveredExtension.getClassLoader();
                for (String dependencyName : discoveredExtension.getDependencies()) {
                    DiscoveredExtension resolved = extensions.stream().filter(ext -> ext.getName().equalsIgnoreCase(dependencyName)).findFirst().orElseThrow(() -> new IllegalStateException("Unknown dependency '" + dependencyName + "' of '" + discoveredExtension.getName() + "'"));
                    ExtensionClassLoader dependencyClassLoader = resolved.getClassLoader();
                    extensionClassLoader.addChild(dependencyClassLoader);
                    LOGGER.trace("Dependency of extension {}: {}", (Object)discoveredExtension.getName(), (Object)resolved);
                }
            }
            catch (Exception e) {
                discoveredExtension.loadStatus = DiscoveredExtension.LoadStatus.MISSING_DEPENDENCIES;
                LOGGER.error("Failed to load dependencies for extension {}", (Object)discoveredExtension.getName());
                LOGGER.error("Extension '{}' will not be loaded", (Object)discoveredExtension.getName());
                LOGGER.error("This is the exception", e);
            }
        }
    }

    private void addDependencyFile(@NotNull ResolvedDependency dependency, @NotNull DiscoveredExtension extension) {
        URL location = dependency.getContentsLocation();
        extension.files.add(location);
        extension.getClassLoader().addURL(location);
        LOGGER.trace("Added dependency {} to extension {} classpath", (Object)location.toExternalForm(), (Object)extension.getName());
        if (!dependency.getSubdependencies().isEmpty()) {
            LOGGER.trace("Dependency {} has subdependencies, adding...", (Object)location.toExternalForm());
            for (ResolvedDependency sub : dependency.getSubdependencies()) {
                this.addDependencyFile(sub, extension);
            }
            LOGGER.trace("Dependency {} has had its subdependencies added.", (Object)location.toExternalForm());
        }
    }

    private boolean loadExtensionList(@NotNull List<DiscoveredExtension> extensionsToLoad) {
        LOGGER.debug("Reorder extensions to ensure proper load order");
        extensionsToLoad = this.generateLoadOrder(extensionsToLoad);
        this.loadDependencies(extensionsToLoad);
        for (DiscoveredExtension toReload : extensionsToLoad) {
            LOGGER.debug("Setting up classloader for extension {}", (Object)toReload.getName());
        }
        LinkedList<Extension> newExtensions = new LinkedList<Extension>();
        for (DiscoveredExtension toReload : extensionsToLoad) {
            LOGGER.info("Actually load extension {}", (Object)toReload.getName());
            Extension loadedExtension = this.loadExtension(toReload);
            if (loadedExtension == null) continue;
            newExtensions.add(loadedExtension);
        }
        if (newExtensions.isEmpty()) {
            LOGGER.error("No extensions to load, skipping callbacks");
            return false;
        }
        LOGGER.info("Load complete, firing preinit, init and then postinit callbacks");
        newExtensions.forEach(Extension::preInitialize);
        newExtensions.forEach(Extension::initialize);
        newExtensions.forEach(Extension::postInitialize);
        return true;
    }

    public void shutdown() {
        HashSet<String> extensionNames = new HashSet<String>(this.extensions.keySet());
        for (String ext : extensionNames) {
            if (!this.extensions.containsKey(ext)) continue;
            this.unloadExtension(ext);
        }
    }

    private void unloadExtension(@NotNull String extensionName) {
        Extension ext = this.extensions.get(extensionName.toLowerCase());
        if (ext == null) {
            throw new IllegalArgumentException("Extension " + extensionName + " is not currently loaded.");
        }
        LinkedList<String> dependents = new LinkedList<String>(ext.getDependents());
        for (String dependentID : dependents) {
            Extension dependentExt = this.extensions.get(dependentID.toLowerCase());
            if (dependentExt == null) continue;
            LOGGER.info("Unloading dependent extension {} (because it depends on {})", (Object)dependentID, (Object)extensionName);
            this.unload(dependentExt);
        }
        LOGGER.info("Unloading extension {}", (Object)extensionName);
        this.unload(ext);
    }

    private void unload(@NotNull Extension ext) {
        ext.preTerminate();
        ext.terminate();
        ext.getExtensionClassLoader().terminate();
        ext.postTerminate();
        String id2 = ext.getOrigin().getName().toLowerCase();
        this.extensions.remove(id2);
    }

    private static enum State {
        DO_NOT_START,
        NOT_STARTED,
        STARTED,
        PRE_INIT,
        INIT,
        POST_INIT;

    }
}

