﻿package pragma.protoc.plugin.custom
import com.google.protobuf.DescriptorProtos
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorRequest
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorResponse
import pragma.PragmaOptions

class UnitySdkGenerator(args: Array<String> = arrayOf()) : SdkGenerator(args) {
    override fun generate(
        target: String,
        request: CodeGeneratorRequest
    ): Iterable<CodeGeneratorResponse.File> {
        // Note: Unity type definitions are currently generated via the normal protobuf flow. See "proto-csharp-unity" in 4-demo/demo-protos/pom.xml
        // or equivalent pom.xml for your project.
        val filesWithApiTypes = filterSdkApiFiles(request)

        if (filesWithApiTypes.isEmpty()) return listOf()

        val generatedTypesFile = createGeneratedTypesFile(target, filesWithApiTypes)
        val responses = mutableListOf(generatedTypesFile)
        responses.addAll(rawServices(target, getFilesToGenerate(request)))
        responses.addAll(generateAnyMetadata(target, request))
        return responses
    }

    // This file provides the type map necessary to map /v1/types info to native C# types.
    private fun createGeneratedTypesFile(
        target: String,
        filesWithApiTypes: List<DescriptorProtos.FileDescriptorProto>
    ): CodeGeneratorResponse.File = CodeGeneratorResponse.File.newBuilder()
        .setName("$target/GeneratedTypes.generated.cs")
        .setContent(
            compileTemplate(
                "unity/GeneratedTypes.mustache",
                generatedTypesTemplateContext(filesWithApiTypes)
            )
        )
        .build()

    private fun generateAnyMetadata(
        target: String,
        request: CodeGeneratorRequest
    ): Iterable<CodeGeneratorResponse.File> {
        val filesWithTypeRegistryEntries = filterTypeRegistryFiles(request)

        val templateContext = mapOf(
            "backendTypes" to filesWithTypeRegistryEntries.flatMap { file ->
                file.messageTypeList.filter(::isTypeRegistryType).map { message ->
                    AnyMetadataTemplateContext(message.options.getExtension(PragmaOptions.typeRegistryId), csharpName(file, message))
                }
            }.sortedBy { it.id }
        )

        return listOf(
            CodeGeneratorResponse.File.newBuilder()
                .setName("$target/AnyUtil.generated.cs")
                .setContent(compileTemplate("unity/AnyUtilGenerated.cs.mustache", templateContext))
                .build()
        )

    }

    private fun getFilesToGenerate(request: CodeGeneratorRequest): List<DescriptorProtos.FileDescriptorProto> {
        // Only output files with messages or enums.
        val filesWithTypes = request.protoFileList
            .filter { file -> file.messageTypeCount > 0 || file.enumTypeCount > 0 }
            .map { file -> file.name }
            .toSet()
        val filesToGenerate = request.protoFileList.filter {
            filesWithTypes.contains(it.name)
        }.filter {
            request.fileToGenerateList.contains(it.name)
        }
        return filesToGenerate
    }

    private fun csharpName(file: DescriptorProtos.FileDescriptorProto, message: DescriptorProtos.DescriptorProto): String {
        val namespace =
            if (file.options.csharpNamespace.isNullOrBlank())
                file.`package`.split('.').map { path -> path.replaceFirstChar { it.uppercase() }}
            else
                listOf(file.options.csharpNamespace)
        return (namespace + message.name).joinToString(".")
    }

    private fun generatedTypesTemplateContext(filesWithApiTypes: List<DescriptorProtos.FileDescriptorProto>) =
        mapOf(
            "backendTypes" to filesWithApiTypes.flatMap { file ->
                file.messageTypeList.filter { isSdkApiType(it) }.map { message ->
                    errorIfForceIncludeAndHidden(message)
                    BackendTypeTemplateContext(
                        backendTypeName(file, message.name),
                        csharpName(file, message)
                    )
                }
            }.sortedBy { it.backendName }
        )

    override fun generatedServiceRegistration(target: String, playerServices: List<String>, partnerServices: List<String>): List<CodeGeneratorResponse.File> {
        val context = GeneratedServicesContext(playerServices, partnerServices)
        return listOf(
            generatedFile(
                "$target/GeneratedServiceRegistration.cs",
                compileTemplate("unity/GeneratedServiceRegistration.cs.mustache", context)
            )
        )
    }

    override fun createRawServiceFiles(
        packageRoot: String,
        serviceName: String,
        sessionName: String,
        target: String,
        context: RawServiceContext
    ): List<CodeGeneratorResponse.File> {
        val fileName = "${packageRoot}${serviceName}${sessionName}ServiceRaw"
        val sourceFile = Unreal4SdkGenerator.generatedFile(
            "$target/generated-services/$fileName.cs",
            compileTemplate("unity/RawService.cs.mustache", context)
        )
        val fakeSourceFile = Unreal4SdkGenerator.generatedFile(
            "$target/generated-fake-services/Fake$fileName.cs",
            compileTemplate("unity/FakeRawService.cs.mustache", context)
        )
        return listOf(sourceFile, fakeSourceFile)
    }
}
