package pragma.protoc.plugin.custom

import com.google.protobuf.DescriptorProtos
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorRequest
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorResponse
import pragma.PragmaOptions

fun unrealOutputHeaderFilepath(target: String, filename: String): String = "$target/$filename.h"
fun unrealOutputCppFilepath(target: String, filename: String): String = "$target/$filename.cpp"

// Unsupported types are not generated.
val ignoredTypeNames = listOf(".google.protobuf.Struct")

@Suppress("UNUSED_PARAMETER")
class Unreal4SdkGenerator(
    previousProtosPaths: Array<String> = arrayOf(),
    fileWrapper: PragmaProtoFileWrapper = PragmaProtoFileWrapper()
) : MustacheSdkGenerator(previousProtosPaths, fileWrapper) {
    class MessageTemplateContextUnreal(
        nativeType: String,
        fields: List<MessageFieldTemplateContext>,
        oneOfTypes: List<String>,
        val hasAddonSerialization: Boolean
    ) : MessageTemplateContext(nativeType, fields, oneOfTypes)
    class EnumTemplateContextUnreal(
        nativeType: String,
        enumValues: List<EnumValueTemplateContext>,
        val ue4Annotations: String,
        val underlyingType: String
    ) : EnumTemplateContext(nativeType, enumValues)

    override fun generate(
        target: String,
        request: CodeGeneratorRequest
    ): Iterable<CodeGeneratorResponse.File> {
        val dependencies = generateDependencyInfo(request)

        val responses = mutableListOf<CodeGeneratorResponse.File>()
        val filesToGenerate = request.protoFileList.filter { file ->
            request.fileToGenerateList.contains(file.name) &&
                // Don't output files if we have no messages or enums.
                (file.messageTypeCount + file.enumTypeCount) > 0
        }

        populateQualifiedToNativeTypeMap(filesToGenerate)

        //
        // Generate type definitions.
        //
        val allFilesWithOneOfs = mutableListOf<String>()
        val allOneOfContexts = mutableListOf<OneOfTemplateContext>()
        responses.addAll(
            filesToGenerate.map { file ->
                val dependencySortedMessages = getDependencySortedMessages(file)
                val outputFilename = outputFilename(file.name)
                val namespace = generateCustomNamespace(file)
                val oneofContexts = dependencySortedMessages.map { createOneOfTemplateContexts(namespace, it) }.flatten()
                val messageSnippets = dependencySortedMessages.map { createMessageSnippets(namespace, it) }
                val headerSnippets = messageSnippets.map { it.definitions }.flatten()
                val cppSnippets = messageSnippets.map { it.implementations }.flatten()
                val enumContexts = file.enumTypeList.map { createEnumTemplateContext(namespace, it) }
                if (oneofContexts.isNotEmpty()) {
                    allFilesWithOneOfs.add(outputFilename)
                    allOneOfContexts.addAll(oneofContexts)
                }
                val headerTemplateContext = mapOf(
                    // [PRAG-227]: Support for google/protobuf and potentially other external types.
                    "dependencies" to file.dependencyList.filter { !it.contains("google/protobuf") }.mapNotNull { dependencies[it] },
                    "sourceFile" to file.name,
                    "outputFileName" to outputFilename,
                    "messages" to headerSnippets,
                    "enums" to enumContexts,
                )
                val headerContent = compileTemplate("unreal4/Dto.h.mustache", headerTemplateContext)
                val header = CodeGeneratorResponse.File.newBuilder()
                    // Note we do not add "generated.h" for Unreal types, since the engine already generates "filename.generated.h"
                    // files for anything that uses its object macros (USTRUCT, UCLASS, etc.)
                    .setName(unrealOutputHeaderFilepath(target, outputFilename)).setContent(headerContent).build()
                val files = mutableListOf(header)

                if (cppSnippets.isNotEmpty()) {
                    val cppTemplateContext = mapOf(
                        "outputFileName" to outputFilename,
                        "messages" to cppSnippets,
                    )
                    val cppContent = compileTemplate("unreal4/Dto.cpp.mustache", cppTemplateContext)
                    val cpp = CodeGeneratorResponse.File.newBuilder()
                        .setName(unrealOutputCppFilepath(target, outputFilename)).setContent(cppContent).build()
                    files.add(cpp)
                }
                files
            }.flatten() // filesToGenerate.map
        ) // responses.addAll
        val rtti = generateRtti(target, allFilesWithOneOfs, allOneOfContexts)
        if (rtti != null) {
            responses.add(rtti)
        }

        responses.addAll(generateTypesMap(target, request, dependencies))
        return responses
    }

    private data class MessageSnippets(val definitions: List<String>, val implementations: List<String>)

    override fun createMessageTemplateContexts(namespace: String, message: DescriptorProtos.DescriptorProto): Pair<MessageTemplateContext, List<OneOfTemplateContext>> {
        // [PRAG-224]: Support nested protobuf type definitions.
        // Note that maps are specifically encoded as nested messages. We do support those.
        if (message.nestedTypeList.count { !it.options.mapEntry } > 0) error("Nested protobuf type defs are not currently supported in Unreal type generation")
        val oneofMessages = createOneOfTemplateContexts(namespace, message)
        val messageNativeType = if (message.name == "Fixed128") {
            // [PRAG-257]: This type is intentionally unused until we support custom deserialization. We write it as FString everywhere,
            // but the type should remain as an unused type with a valid name.
            "FPragma_Fixed128"
        } else {
            generatedType(namespace, message)
        }
        val fieldContexts = message.fieldList
            .filter { !it.hasOneofIndex() }
            .filter{ !ignoredTypeNames.contains(it.typeName) }
            .map { field ->
            val nativeType = generatedType(field)
            MessageFieldTemplateContext(
                nativeType,
                name = field.name.snakeToPascal(),
                defaultValue = generatedTypeDefaultValue(field),
                // Fixed128 uses uint64 which is not supported by BP and will fail to compile.
                annotations = if (message.name != "Fixed128") { "VisibleAnywhere, BlueprintReadOnly" } else { "" },
                isMap = nativeType.contains("TMap"),
            )
        } + message.oneofDeclList.mapIndexed { i, oneof ->
            val oneofType = qualifiedToNativeType[qualifiedOneofName(namespace, message.name, i)]
                ?: error("Oneof ${oneof.name} in message ${message.name} has no qualified name has created for it.")
            MessageFieldTemplateContext(
                nativeType = oneofType,
                name = oneof.name.snakeToPascal(),
                "",
                "Transient",
                false
            )
        }
        val primaryMessage = MessageTemplateContextUnreal(
            messageNativeType,
            fieldContexts,
            oneofMessages.map { it.oneOfType },
            hasAddonSerialization = oneofMessages.isNotEmpty()
        )
        return Pair(primaryMessage, oneofMessages)
    }

    private fun createMessageSnippets(namespace: String, message: DescriptorProtos.DescriptorProto): MessageSnippets {
        val (primaryMessage, oneofMessages) = createMessageTemplateContexts(namespace, message)
        val definitions = mutableListOf<String>()
        val implementations = mutableListOf<String>()
        if (oneofMessages.isNotEmpty()) {
            oneofMessages.forEach { oneof ->
                definitions.add(compileTemplate("unreal4/OneOfSnippet.h.mustache", oneof))
                implementations.add(compileTemplate("unreal4/OneOfSnippet.cpp.mustache", oneof))
            }
        }
        definitions.add(compileTemplate("unreal4/MessageSnippet.h.mustache", primaryMessage))
        implementations.add(compileTemplate("unreal4/MessageSnippet.cpp.mustache", primaryMessage))
        return MessageSnippets(definitions, implementations)
    }

    override fun createOneOfTemplateContexts(namespace: String, message: DescriptorProtos.DescriptorProto): List<OneOfTemplateContext> {
        val oneOfFields = message.fieldList.filter { it.hasOneofIndex() }.groupBy { it.oneofIndex }
        return message.oneofDeclList.mapIndexed { i, oneof ->
            val oneOfParentFieldName = generatedType(namespace, message)
            val oneOfParentField = oneof.name.snakeToPascal()
            val oneOfType = generatedTypeInternal(ProtobufType.Message, namespace, "${message.name}_${oneOfParentField}")
            val oneOfEnum = generatedTypeInternal(ProtobufType.Enum, namespace, "${message.name}_${oneOfParentField}Type")
            val fields = oneOfFields[i]?.map { field ->
                val fieldName = field.name.snakeToPascal()
                val isStructOrEnum = (field.type == DescriptorProtos.FieldDescriptorProto.Type.TYPE_ENUM
                    || field.type == DescriptorProtos.FieldDescriptorProto.Type.TYPE_MESSAGE)
                    && !field.isFixed128()
                OneOfFieldTemplateContext(
                    generatedType(field),
                    fieldName,
                    fieldName.replaceFirstChar { it.lowercase() },
                    jsonType(field),
                    isStructOrEnum,
                    isPrimitiveField(field)
                )
            } ?: listOf()
            // Since we're generating the oneof type name here, we also put it into the map so we can lookup the type later.
            qualifiedToNativeType[qualifiedOneofName(namespace, message.name, i)] = oneOfType
            OneOfTemplateContext(
                oneOfType,
                oneOfEnum,
                fields,
                oneOfParentFieldName,
                oneOfParentField,
            )
        }
    }

    private fun isBlueprintableEnum(enum: DescriptorProtos.EnumDescriptorProto): Boolean {
        // Most enums in unreal should be BlueprintType and uint8 because blueprints require it, but
        // the errors enums are bigger as we explicitly set numbers > 255.
        return enum.name != "PragmaError" && enum.name != "ExtError"
    }

    override fun createEnumTemplateContext(namespace: String, enum: DescriptorProtos.EnumDescriptorProto): EnumTemplateContext {
        return EnumTemplateContextUnreal(
            generatedType(namespace, enum),
            enum.valueList.map { EnumValueTemplateContext(it.name, it.number) },
            if (isBlueprintableEnum(enum)) "BlueprintType" else "",
            if (isBlueprintableEnum(enum)) " : uint8" else "",
        )
    }

    override fun getEnumName(type: DescriptorProtos.EnumDescriptorProto): String {
        val overrideName = type.options.getExtension(PragmaOptions.unrealEnumName)
        return if (overrideName.isNullOrEmpty()) type.name else overrideName
    }

    override fun getTypeName(type: DescriptorProtos.DescriptorProto): String {
        val overrideName = type.options.getExtension(PragmaOptions.unrealTypeName)
        return if (overrideName.isNullOrEmpty()) type.name else overrideName
    }

    private fun generateTypesMap(
        target: String,
        request: CodeGeneratorRequest,
        dependencies: GeneratedDependencyInfoTemplateContextMap
    ): Iterable<CodeGeneratorResponse.File> {
        val filesWithApiTypes = filterSdkApiFiles(request)

        if (filesWithApiTypes.isEmpty()) return listOf()

        val templateContext = mapOf(
            "backendTypes" to filesWithApiTypes.flatMap { file ->
                file.messageTypeList.filter { isSdkApiType(it) }.map { message ->
                        val nativeType = generatedType(generateCustomNamespace(file), message)
                        BackendTypeTemplateContext(backendTypeName(file, message.name), nativeType)
                    }
            },
            "dependencies" to filesWithApiTypes.mapNotNull { dependencies[it.name] }
        )

        return listOf(CodeGeneratorResponse.File.newBuilder()
            .setName("$target/PragmaGeneratedTypes.h")
            .setContent(compileTemplate("unreal4/GeneratedTypes.mustache", templateContext))
            .build())
    }

    private fun generateRtti(
        target: String,
        dependencies: List<String>,
        oneofTemplateContexts: List<OneOfTemplateContext>
    ): CodeGeneratorResponse.File? {
        if (oneofTemplateContexts.isEmpty()) {
            return null
        }
        val content = compileTemplate(
            "unreal4/GeneratedRtti.cpp.mustache", mapOf(
                "dependencies" to dependencies.map { GeneratedDependencyInfoTemplateContext(it) },
                "oneofs" to oneofTemplateContexts,
            )
        )
        return CodeGeneratorResponse.File.newBuilder()
            .setName("$target/PragmaGeneratedRtti.cpp")
            .setContent(content)
            .build()
    }

    /**
     * Returns a generated type name for a Pragma proto message appropriate for Unreal.
     * e.g.
     * pragma.account.LoginV1Request -> FPragma_Account_LoginV1Request
     *
     * Notes:
     * - Unreal requires prefixing types with a letter to indicate type.
     * - Unreal object macros (USTRUCT, etc.) do not work with namespaces, so we bake namespaced into the typenames
     *   to avoid conflicts with customer code.
     */
    override fun generatedTypeInternal(protoType: ProtobufType, namespace: String, typeName: String): String {
        val prefix = when (protoType) {
            ProtobufType.Enum -> "E"
            ProtobufType.Message -> "F"
        }
        val namespaceSeparator = if (namespace == "") {
            ""
        } else {
            "_"
        }
        val unrealTypeName = "${prefix}Pragma_${namespace}${namespaceSeparator}$typeName"
        return if (unrealTypeName == "FPragma_Fixed128") {
            // [PRAG-257]: Custom serialization callbacks for Fixed128 so we can handle the type conversion to string uuid for json path
            // For now, this only supports the json path, which returns a string uuid.
            "FString"
        } else {
            unrealTypeName
        }
    }

    override fun generatedType(valueType: DescriptorProtos.FieldDescriptorProto?, keyType: DescriptorProtos.FieldDescriptorProto?) : String {
        return "TMap<${keyType?.let { generatedType(it) }}, ${valueType?.let { generatedType(it) }}>"
    }

    override fun generatedType(field: DescriptorProtos.FieldDescriptorProto): String {
        val unreal4Type = when (field.type) {
            // See https://developers.google.com/protocol-buffers/docs/proto for baseline.
            // Note that some use Unreal-specific types.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_DOUBLE -> "double"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FLOAT -> "float"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT64 -> "int64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT64 -> "uint64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT32 -> "int32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED64 -> "uint64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED32 -> "uint32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BOOL -> "bool"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_STRING -> "FString"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT32 -> "uint32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED32 -> "int32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED64 -> "int64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT32 -> "int32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT64 -> "int64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_ENUM,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_MESSAGE -> buildMessageType(field)
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_GROUP -> error("Proto 'groups' are deprecated and unsupported. https://developers.google.com/protocol-buffers/docs/proto#groups")
            // [PRAG-257]: Support protobuf 'bytes' when we build proto path.
            // For the moment, we use FString for bytes, because IntAny comes across as an escaped json string.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BYTES -> "FString"
            null -> error("Proto type is null: '${field.name}', field number: ${field.number}. This shouldn't happen.")
        }
        return if (field.label == DescriptorProtos.FieldDescriptorProto.Label.LABEL_REPEATED &&
            // Maps are parsed as repeated *Entry types, so if it's a map, we shouldn't also wrap it in array.
            !unreal4Type.contains("TMap<.+>".toRegex())
        ) {
            "TArray<$unreal4Type>"
        } else {
            unreal4Type
        }
    }

    override fun generatedTypeDefaultValue(field: DescriptorProtos.FieldDescriptorProto): String {
        if (field.label == DescriptorProtos.FieldDescriptorProto.Label.LABEL_REPEATED) {
            return ""
        }
        return when (field.type) {
            // Appended directly after the variable name.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_DOUBLE -> "{0.0}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FLOAT -> "{0.0f}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT64 -> "{0}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BOOL -> "{false}"
            // Protos default to '0', so we know it always exists and should likewise default to 0.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_ENUM -> "{static_cast<${generatedType(field)}>(0)}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BYTES,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_STRING,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_MESSAGE -> ""
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_GROUP -> error("Proto 'groups' are deprecated and unsupported. https://developers.google.com/protocol-buffers/docs/proto#groups")
            null -> error("Unsupported proto type: ${field.typeName} on field ${field.name}")
        }
    }
}