package pragma.protoc.plugin.custom

import com.google.protobuf.DescriptorProtos
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorRequest
import com.google.protobuf.compiler.PluginProtos.CodeGeneratorResponse
import pragma.PragmaOptions

fun unrealOutputHeaderFilepath(target: String, filename: String): String = "$target/$filename.h"
fun unrealOutputCppFilepath(target: String, filename: String): String = "$target/$filename.cpp"

// Unsupported types are not generated.
val ignoredTypeNames = listOf(".google.protobuf.Struct")

@Suppress("UNUSED_PARAMETER")
class Unreal4SdkGenerator(
    args: Array<String> = arrayOf(),
    fileWrapper: PragmaProtoFileWrapper = PragmaProtoFileWrapper()
) : MustacheSdkGenerator(args, fileWrapper) {
    class MessageTemplateContextUnreal(
        nativeType: String,
        fields: List<MessageFieldTemplateContext>,
        oneOfTypes: List<String>,
        val hasAddonSerialization: Boolean
    ) : MessageTemplateContext(nativeType, fields, oneOfTypes)

    class EnumTemplateContextUnreal(
        nativeType: String,
        enumValues: List<EnumValueTemplateContext>,
        val ue4Annotations: String,
        val underlyingType: String
    ) : EnumTemplateContext(nativeType, enumValues)

    override fun generate(
        target: String,
        request: CodeGeneratorRequest
    ): Iterable<CodeGeneratorResponse.File> {
        // Only output files with messages or enums.
        val filesWithTypes = request.protoFileList
            .filter { file -> file.messageTypeCount > 0 || file.enumTypeCount > 0 }
            .map { file -> file.name }
            .toSet()
        val dependencies = generateDependencyInfo(request).filter { filesWithTypes.contains(it.key) }
        val filesToGenerate = request.protoFileList.filter {
            filesWithTypes.contains(it.name)
        }.filter {
            request.fileToGenerateList.contains(it.name)
        }

        populateQualifiedToNativeTypeMap(filesToGenerate)

        //
        // Generate type definitions.
        //
        val allFilenames = mutableListOf<String>()
        val allMessageContexts = mutableListOf<MessageTemplateContext>()
        val allOneOfContexts = mutableListOf<OneOfTemplateContext>()
        val responses = mutableListOf<CodeGeneratorResponse.File>()
        responses.addAll(
            filesToGenerate.map { file ->
                val dependencySortedMessages = getDependencySortedMessages(file).filter { hasExternalVisibility(it) }
                val outputFilename = outputFilename(file.name)
                val namespace = generateCustomNamespace(file)
                val oneofContexts =
                    dependencySortedMessages.map { createOneOfTemplateContexts(namespace, it) }.flatten()
                val messageSnippets = dependencySortedMessages.map { createMessageSnippets(namespace, it) }
                val forwardDeclarations = messageSnippets.map { it.forwardDeclarations }.flatten()
                val headerSnippets = messageSnippets.map { it.definitions }.flatten()
                val cppSnippets = messageSnippets.map { it.implementations }.flatten()
                val enumContexts = file.enumTypeList.map { createEnumTemplateContext(namespace, it) }
                allFilenames.add(outputFilename)
                allMessageContexts.addAll(dependencySortedMessages.map { createMessageTemplateContexts(namespace, it).first })
                if (oneofContexts.isNotEmpty()) {
                    allOneOfContexts.addAll(oneofContexts)
                }
                val headerTemplateContext = mapOf(
                    // [PRAG-227]: Support for google/protobuf and potentially other external types.
                    "dependencies" to file.dependencyList.filter { !it.contains("google/protobuf") }
                        .mapNotNull { dependencies[it] },
                    "sourceFile" to file.name,
                    "outputFileName" to outputFilename,
                    "forwardDeclarations" to forwardDeclarations,
                    "messages" to headerSnippets,
                    "enums" to enumContexts,
                )
                // Note we do not add "generated.h" for Unreal types, since the engine already generates "filename.generated.h"
                // files for anything that uses its object macros (USTRUCT, UCLASS, etc.)
                val headerName = unrealOutputHeaderFilepath(target, outputFilename)
                val headerContent = compileTemplate("unreal4/Dto.h.mustache", headerTemplateContext)
                val files = mutableListOf(generatedFile(headerName, headerContent))

                if (cppSnippets.isNotEmpty()) {
                    val cppTemplateContext = mapOf(
                        "outputFileName" to outputFilename,
                        "messages" to cppSnippets,
                    )
                    val cppName = unrealOutputCppFilepath(target, outputFilename)
                    val cppContent = compileTemplate("unreal4/Dto.cpp.mustache", cppTemplateContext)
                    files.add(generatedFile(cppName, cppContent))
                }
                files
            }.flatten() // filesToGenerate.map
        ) // responses.addAll
        val rtti = generateRtti(target, allFilenames, allMessageContexts, allOneOfContexts)
        if (rtti != null) {
            responses.add(rtti)
        }
        responses.addAll(rawServices(target, filesToGenerate))
        responses.addAll(generateTypesMap(target, request, dependencies))
        return responses
    }

    private data class MessageSnippets(val forwardDeclarations: List<String>, val definitions: List<String>, val implementations: List<String>)

    private data class RawServiceContext(val className: String, val dtoFileName: String, val packageRoot: String, val serviceName: String, val sessionName: String, val sdkSessionType: String, val backendType: String, val notifications: List<RpcContext>, val rpcs: List<RpcContext>)
    private data class RpcContext(val rpcName: String)

    private data class GeneratedServicesContext(val playerServices: List<String>, val partnerServices: List<String>)

    private data class RawService(
        val className: String,
        val files: List<CodeGeneratorResponse.File>
    )

    private fun rawServices(target: String, fileDescriptors: List<DescriptorProtos.FileDescriptorProto>): List<CodeGeneratorResponse.File> {
        val playerServices = mutableListOf<String>()
        val partnerServices = mutableListOf<String>()
        val files = fileDescriptors.map { file ->
            val rpcTypes = file.messageTypeList.filter { isSdkApiType(it) }
            val outFiles = mutableListOf<CodeGeneratorResponse.File>()
            val service = rawService("", target, file, rpcTypes.filter { isPlayerRpc(it) })
            if (service != null) {
                playerServices.add(service.className)
                outFiles.addAll(service.files)
            }
            val partnerService = rawService("Partner", target, file, rpcTypes.filter { isPartnerRpc(it) })
            if (partnerService != null) {
                partnerServices.add(partnerService.className)
                outFiles.addAll(partnerService.files)
            }
            outFiles
        }.flatten().toMutableList()
        files.addAll(generatedServiceRegistration(target, playerServices, partnerServices))
        return files
    }

    private fun generatedServiceRegistration(target: String, playerServices: List<String>, partnerServices: List<String>): List<CodeGeneratorResponse.File> {
        val context = GeneratedServicesContext(playerServices, partnerServices);
        return listOf(
            generatedFile("$target/PragmaGeneratedServiceRegistration.h", compileTemplate("unreal4/GeneratedServiceRegistration.h.mustache", context)),
            generatedFile("$target/PragmaGeneratedServiceRegistration.cpp", compileTemplate("unreal4/GeneratedServiceRegistration.cpp.mustache", context))
        )
    }

    private fun rawService(sessionName: String, target: String, file: DescriptorProtos.FileDescriptorProto, types: List<DescriptorProtos.DescriptorProto>): RawService? {
        if (types.isEmpty()) {
            return null
        }
        val packageRoot = packageRoot(file.`package`)
        val serviceName = generateCustomNamespace(file)
        if (isNoCodegenService(packageRoot, serviceName)) {
            return null
        }
        val className = rawServiceClassName(packageRoot, serviceName, sessionName)
        val context = RawServiceContext(
            className,
            dtoFileName = outputFilename(file.name),
            packageRoot = packageRoot,
            serviceName = serviceName,
            sessionName = sessionName,
            sdkSessionType = if (sessionName == "Partner") { "Server" } else { "Player" },
            backendType = getFileBackendType(file),
            notifications = types
                .filter { it.name.endsWith("Notification") }
                .map { RpcContext(it.name.replace("Notification", "")) },
            rpcs = types
                // Both request and response are in the list, but we create 1 method for each PAIR.
                .filter{ it.name.endsWith("Request") }
                .map{ RpcContext(it.name.removeSuffix("Request")) },
        )
        val fileName = "${packageRoot}${serviceName}${sessionName}ServiceRaw"
        val headerFile = generatedFile("$target/$fileName.h", compileTemplate("unreal4/RawService.h.mustache", context))
        val sourceFile = generatedFile("$target/$fileName.cpp", compileTemplate("unreal4/RawService.cpp.mustache", context))
        return RawService(className, listOf(headerFile, sourceFile))
    }



    private fun rawServiceClassName(packageRoot: String, serviceName: String, sessionName: String) = "$packageRoot$serviceName${sessionName}ServiceRaw"

    private fun getFileBackendType(file: DescriptorProtos.FileDescriptorProto): String {
        val overrideName = file.options.getExtension(PragmaOptions.backendType)
        return if (overrideName == PragmaOptions.PragmaBackendType.SOCIAL) "SOCIAL" else "GAME"
    }

    private fun generatedFile(name: String, content: String): CodeGeneratorResponse.File {
        return CodeGeneratorResponse.File.newBuilder()
            .setName(name).setContent(content).build()
    }

    override fun createMessageTemplateContexts(
        namespace: String,
        message: DescriptorProtos.DescriptorProto
    ): Pair<MessageTemplateContext, List<OneOfTemplateContext>> {
        // [PRAG-224]: Support nested protobuf type definitions.
        // Note that maps are specifically encoded as nested messages. We do support those.
        if (message.nestedTypeList.count { !it.options.mapEntry } > 0) error("Nested protobuf type defs are not currently supported in Unreal type generation")
        val oneofMessages = createOneOfTemplateContexts(namespace, message)
        val messageNativeType = if (message.name == "Fixed128") {
            // [PRAG-257]: This type is intentionally unused until we support custom deserialization. We write it as FString everywhere,
            // but the type should remain as an unused type with a valid name.
            "FPragma_Fixed128"
        } else {
            generatedType(namespace, message)
        }
        val fieldContexts = message.fieldList
            .filter { !it.hasOneofIndex() }
            .filter { !ignoredTypeNames.contains(it.typeName) }
            .map { field ->
                val nativeType = generatedType(field)
                MessageFieldTemplateContext(
                    nativeType,
                    name = field.name.snakeToPascal(),
                    defaultValue = generatedTypeDefaultValue(field),
                    // Fixed128 uses uint64 which is not supported by BP and will fail to compile.
                    annotations = if (message.name != "Fixed128") {
                        "VisibleAnywhere, BlueprintReadOnly, Category=Pragma"
                    } else {
                        ""
                    },
                    isMap = nativeType.contains("TMap"),
                )
            } + message.oneofDeclList.mapIndexed { i, oneof ->
            val oneofType = qualifiedToNativeType[qualifiedOneofName(namespace, message.name, i)]
                ?: error("Oneof ${oneof.name} in message ${message.name} has no qualified name has created for it.")
            MessageFieldTemplateContext(
                nativeFieldType = oneofType,
                name = oneof.name.snakeToPascal(),
                "",
                "Transient",
                false
            )
        }
        val primaryMessage = MessageTemplateContextUnreal(
            messageNativeType,
            fieldContexts,
            oneofMessages.map { it.oneOfType },
            hasAddonSerialization = oneofMessages.isNotEmpty()
        )
        return Pair(primaryMessage, oneofMessages)
    }

    private fun createMessageSnippets(namespace: String, message: DescriptorProtos.DescriptorProto): MessageSnippets {
        val (primaryMessage, oneofMessages) = createMessageTemplateContexts(namespace, message)
        val forwardDeclarations = listOf(primaryMessage.nativeType)
        val definitions = mutableListOf<String>()
        val implementations = mutableListOf<String>()
        if (oneofMessages.isNotEmpty()) {
            oneofMessages.forEach { oneof ->
                val snippetName = if (oneof.isOptional) "Optional" else "OneOf"
                definitions.add(compileTemplate("unreal4/${snippetName}Snippet.h.mustache", oneof))
                implementations.add(compileTemplate("unreal4/${snippetName}Snippet.cpp.mustache", oneof))
            }
        }
        definitions.add(compileTemplate("unreal4/MessageSnippet.h.mustache", primaryMessage))
        implementations.add(compileTemplate("unreal4/MessageSnippet.cpp.mustache", primaryMessage))
        return MessageSnippets(forwardDeclarations, definitions, implementations)
    }

    override fun createOneOfTemplateContexts(
        namespace: String,
        message: DescriptorProtos.DescriptorProto
    ): List<OneOfTemplateContext> {
        val oneOfFields = message.fieldList.filter { it.hasOneofIndex() }.groupBy { it.oneofIndex }
        return message.oneofDeclList.mapIndexed { i, oneof ->
            val oneOfParentFieldName = generatedType(namespace, message)
            val oneOfParentField = oneof.name.snakeToPascal()
            val oneOfType =
                generatedTypeInternal(ProtobufType.Message, namespace, "${message.name}_${oneOfParentField}")
            val oneOfEnum =
                generatedTypeInternal(ProtobufType.Enum, namespace, "${message.name}_${oneOfParentField}Type")
            val fields = oneOfFields[i]?.map { field ->
                val fieldName = field.name.snakeToPascal()
                val isStructOrEnum = (field.type == DescriptorProtos.FieldDescriptorProto.Type.TYPE_ENUM
                        || field.type == DescriptorProtos.FieldDescriptorProto.Type.TYPE_MESSAGE)
                        && !field.isFixed128()
                OneOfFieldTemplateContext(
                    generatedType(field),
                    fieldName,
                    fieldName.replaceFirstChar { it.lowercase() },
                    jsonType(field),
                    isStructOrEnum,
                    isPrimitiveField(field)
                )
            } ?: listOf()
            // Since we're generating the oneof type name here, we also put it into the map so we can lookup the type later.
            qualifiedToNativeType[qualifiedOneofName(namespace, message.name, i)] = oneOfType
            OneOfTemplateContext(
                oneOfType,
                oneOfEnum,
                fields,
                oneOfParentFieldName,
                oneOfParentField,
                isOptional = message.optionalOneOfIndexes().contains(i)
            )
        }
    }

    private fun isBlueprintableEnum(enum: DescriptorProtos.EnumDescriptorProto): Boolean {
        // Most enums in unreal should be BlueprintType and uint8 because blueprints require it, but
        // the errors enums are bigger as we explicitly set numbers > 255.
        return enum.name != "PragmaError" && enum.name != "ExtError"
    }

    override fun createEnumTemplateContext(
        namespace: String,
        enum: DescriptorProtos.EnumDescriptorProto
    ): EnumTemplateContext {
        return EnumTemplateContextUnreal(
            generatedType(namespace, enum),
            enum.valueList.map { EnumValueTemplateContext(it.name, it.number) },
            if (isBlueprintableEnum(enum)) "BlueprintType, Category=Pragma" else "",
            if (isBlueprintableEnum(enum)) " : uint8" else "",
        )
    }

    override fun getEnumName(type: DescriptorProtos.EnumDescriptorProto): String {
        val overrideName = type.options.getExtension(PragmaOptions.unrealEnumName)
        return if (overrideName.isNullOrEmpty()) type.name else overrideName
    }

    override fun getTypeName(type: DescriptorProtos.DescriptorProto): String {
        val overrideName = type.options.getExtension(PragmaOptions.unrealTypeName)
        return if (overrideName.isNullOrEmpty()) type.name else overrideName
    }

    private fun generateTypesMap(
        target: String,
        request: CodeGeneratorRequest,
        dependencies: GeneratedDependencyInfoTemplateContextMap
    ): Iterable<CodeGeneratorResponse.File> {
        val filesWithApiTypes = filterSdkApiFiles(request)

        if (filesWithApiTypes.isEmpty()) return listOf()

        val templateContext = mapOf(
            "backendTypes" to filesWithApiTypes.flatMap { file ->
                file.messageTypeList.filter { isSdkApiType(it) }.map { message ->
                    val nativeType = generatedType(generateCustomNamespace(file), message)
                    BackendTypeTemplateContext(backendTypeName(file, message.name), nativeType)
                }
            }.sortedBy { it.backendName },
            "dependencies" to filesWithApiTypes.mapNotNull { dependencies[it.name] }
        )

        return listOf(
            CodeGeneratorResponse.File.newBuilder()
                .setName("$target/PragmaGeneratedTypes.h")
                .setContent(compileTemplate("unreal4/GeneratedTypes.mustache", templateContext))
                .build()
        )
    }

    private fun generateRtti(
        target: String,
        dependencies: List<String>,
        messageTemplateContexts: List<MessageTemplateContext>,
        oneofTemplateContexts: List<OneOfTemplateContext>,
    ): CodeGeneratorResponse.File? {
        if (oneofTemplateContexts.isEmpty()) {
            return null
        }
        val content = compileTemplate(
            "unreal4/GeneratedRtti.cpp.mustache", mapOf(
                "dependencies" to dependencies.map { GeneratedDependencyInfoTemplateContext(it) },
                "messages" to messageTemplateContexts,
                "oneofs" to oneofTemplateContexts,
            )
        )
        return CodeGeneratorResponse.File.newBuilder()
            .setName("$target/PragmaGeneratedRtti.cpp")
            .setContent(content)
            .build()
    }

    /**
     * Returns a generated type name for a Pragma proto message appropriate for Unreal.
     * e.g.
     * pragma.account.LoginV1Request -> FPragma_Account_LoginV1Request
     *
     * Notes:
     * - Unreal requires prefixing types with a letter to indicate type.
     * - Unreal object macros (USTRUCT, etc.) do not work with namespaces, so we bake namespaced into the typenames
     *   to avoid conflicts with customer code.
     */
    override fun generatedTypeInternal(protoType: ProtobufType, namespace: String, typeName: String): String {
        val prefix = when (protoType) {
            ProtobufType.Enum -> "E"
            ProtobufType.Message -> "F"
        }
        val namespaceSeparator = if (namespace == "") {
            ""
        } else {
            "_"
        }
        val unrealTypeName = "${prefix}Pragma_${namespace}${namespaceSeparator}$typeName"
        return if (unrealTypeName == "FPragma_Fixed128") {
            // [PRAG-257]: Custom serialization callbacks for Fixed128 so we can handle the type conversion to string uuid for json path
            // For now, this only supports the json path, which returns a string uuid.
            "FString"
        } else {
            unrealTypeName
        }
    }

    override fun generatedType(
        valueType: DescriptorProtos.FieldDescriptorProto?,
        keyType: DescriptorProtos.FieldDescriptorProto?
    ): String {
        return "TMap<${keyType?.let { generatedType(it) }}, ${valueType?.let { generatedType(it) }}>"
    }

    override fun generatedType(field: DescriptorProtos.FieldDescriptorProto): String {
        var unreal4Type = when (field.type) {
            // See https://developers.google.com/protocol-buffers/docs/proto for baseline.
            // Note that some use Unreal-specific types.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_DOUBLE -> "double"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FLOAT -> "float"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT64 -> "int64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT64 -> "uint64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT32 -> "int32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED64 -> "uint64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED32 -> "uint32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BOOL -> "bool"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_STRING -> "FString"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT32 -> "uint32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED32 -> "int32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED64 -> "int64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT32 -> "int32"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT64 -> "int64"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_ENUM,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_MESSAGE -> buildMessageType(field)
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_GROUP -> error("Proto 'groups' are deprecated and unsupported. https://developers.google.com/protocol-buffers/docs/proto#groups")
            // [PRAG-257]: Support protobuf 'bytes' when we build proto path.
            // For the moment, we use FString for bytes, because IntAny comes across as an escaped json string.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BYTES -> "FString"
            null -> error("Proto type is null: '${field.name}', field number: ${field.number}. This shouldn't happen.")
        }
        if (field.options.getExtension(PragmaOptions.exportAsKey)) {
            unreal4Type = unreal4Type.replace("FString", "FName")
        }
        return if (field.label == DescriptorProtos.FieldDescriptorProto.Label.LABEL_REPEATED &&
            // Maps are parsed as repeated *Entry types, so if it's a map, we shouldn't also wrap it in array.
            !unreal4Type.contains("TMap<.+>".toRegex())
        ) {
            "TArray<$unreal4Type>"
        } else {
            unreal4Type
        }
    }

    override fun generatedTypeDefaultValue(field: DescriptorProtos.FieldDescriptorProto): String {
        if (field.label == DescriptorProtos.FieldDescriptorProto.Label.LABEL_REPEATED) {
            return ""
        }
        val isKey = field.options.getExtension(PragmaOptions.exportAsKey)
        if (isKey) {
            return "{NAME_None}"
        }
        return when (field.type) {
            // Appended directly after the variable name.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_DOUBLE -> "{0.0}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FLOAT -> "{0.0f}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_UINT64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_FIXED32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SINT64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_SFIXED64,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT32,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_INT64 -> "{0}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BOOL -> "{false}"
            // Protos default to '0', so we know it always exists and should likewise default to 0.
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_ENUM -> "{static_cast<${generatedType(field)}>(0)}"
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_BYTES,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_STRING,
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_MESSAGE -> ""
            DescriptorProtos.FieldDescriptorProto.Type.TYPE_GROUP -> error("Proto 'groups' are deprecated and unsupported. https://developers.google.com/protocol-buffers/docs/proto#groups")
            null -> error("Unsupported proto type: ${field.typeName} on field ${field.name}")
        }
    }
}