package pragma.protoc.plugin.custom

import com.google.protobuf.DescriptorProtos
import com.google.protobuf.compiler.PluginProtos
import pragma.rpc.Data
import pragma.rpc.Notification
import pragma.rpc.Request
import pragma.rpc.Response
import pragma.rpc.Rpc
import kotlin.reflect.KClass

@Suppress("UNUSED_PARAMETER")
class JavaPostGenerator(val ignorePackage: String = "") : ProtocPlugin() {
    override fun generate(
        target: String,
        request: PluginProtos.CodeGeneratorRequest
    ): Iterable<PluginProtos.CodeGeneratorResponse.File> {
        val response = mutableListOf<PluginProtos.CodeGeneratorResponse.File>()
        request.protoFileList.filter {
            request.fileToGenerateList.contains(it.name)
        }.forEach { protoFile ->
            // There is a "java_multiple_files" option which either makes a top level class,
            // and all the nested messages as nested classes, or promotes all messages in that file to their own top level class. Build these differently
            if (shouldIncludeFile(protoFile)) {
                // e.g fileDescriptorProto.name = pragma/account/accountRpc.proto
                val fileNameParts = protoFile.name.split("/")
                val lastPart = fileNameParts[fileNameParts.size - 1]
                // path is relative to project root (/platform)
                val actualFileName = "${protoFile.`package`.replace(".", "/")}/${
                    lastPart.substring(0, 1).uppercase()
                }${lastPart.substring(1).replace("proto", "java")}"
                // All RPC requests must be nested
                protoFile.messageTypeList.forEach { messageType ->
                    val qualifiedMessageName = "${protoFile.`package`}.${messageType.name}"
                    // Attach correct RPC interfaces
                    listOf(Request::class, Response::class, Notification::class, Data::class).forEach { rpcClassType ->
                        if (matchesType(messageType, rpcClassType)) {
                            response.add(
                                PluginProtos.CodeGeneratorResponse.File.newBuilder()
                                    .setName(actualFileName)
                                    .setInsertionPoint("interface_extends:$qualifiedMessageName")
                                    .setContent("${rpcClassType.qualifiedName},")
                                    .build()
                            )
                            response.add(
                                PluginProtos.CodeGeneratorResponse.File.newBuilder()
                                    .setName(actualFileName)
                                    .setInsertionPoint("message_implements:$qualifiedMessageName")
                                    .setContent("${rpcClassType.qualifiedName},")
                                    .build()
                            )
                        }
                    }
                }
            }
        }
        return response
    }

    private fun rpcRegex(rpcClassType: KClass<out Rpc>) = Regex(".*V\\d+${rpcClassType.simpleName}$")

    private fun matchesType(
        messageType: DescriptorProtos.DescriptorProto,
        rpcClassType: KClass<out Rpc>
    ) = messageType.name.matches(rpcRegex(rpcClassType))

    private fun shouldIncludeFile(protoFile: DescriptorProtos.FileDescriptorProto) =
        (ignorePackage == "" || !protoFile.`package`.startsWith(ignorePackage)) && !protoFile.options.javaMultipleFiles
}
