package pragma.maven.plugin.protofilter
import java.io.File
import java.io.IOException
import java.nio.file.Files
import java.nio.file.Paths
import java.nio.file.StandardOpenOption
import org.apache.maven.plugin.AbstractMojo
import org.apache.maven.plugin.MojoFailureException
import org.apache.maven.plugins.annotations.LifecyclePhase
import org.apache.maven.plugins.annotations.Mojo
import org.apache.maven.plugins.annotations.Parameter
import org.codehaus.plexus.util.FileUtils
import org.slf4j.LoggerFactory

// All Ints because thats how the reader reads characters.
private const val CHAR_OPEN_BRACKET = '{'
private const val CHAR_CLOSE_BRACKET = '}'
private const val CHAR_OPEN_ANGLE = '<'
private const val CHAR_CLOSE_ANGLE = '>'
private const val CHAR_FWD_SLASH = '/'
private const val CHAR_ASTERISK = '*'
private const val CHAR_NEW_LINE = '\n'
private const val CHAR_SPACE = ' '
private const val CHAR_TAB = '\t'
private const val CHAR_SEMICOLON = ';'
private val END_OF_TOKEN = setOf(
    CHAR_SPACE, CHAR_TAB, CHAR_SEMICOLON, CHAR_OPEN_BRACKET, CHAR_CLOSE_BRACKET, CHAR_FWD_SLASH,
    // These are to handle `map<type, tokenWeWant>`
    // Allow the open angle so the entire first part is part of the map token, then break on >.
    /*CHAR_OPEN_ANGLE,*/ CHAR_CLOSE_ANGLE
)

// Only needs to include chars not specified above.
private val IGNORED_CHARS = setOf('\r')

private fun isLineComment(charA: Char, charB: Char) = charA == CHAR_FWD_SLASH && charB == CHAR_FWD_SLASH
private fun isBlockCommentStart(charA: Char, charB: Char) = charA == CHAR_FWD_SLASH && charB == CHAR_ASTERISK
private fun isBlockCommentEnd(charA: Char, charB: Char) = charA == CHAR_ASTERISK && charB == CHAR_FWD_SLASH
private fun isEndOfToken(char: Char) = END_OF_TOKEN.contains(char)


private val PROTO_IMPORT_REGEX =
    """import\s+"(.+)"\s*;""".toRegex()

/**
 * This maven plugin exists to take a set of protobuf file inputs, and remove or modify their contents based on a set of
 * custom proto options (see proto-options module / pragmaOptions.proto), resulting in a filtered set of protos relevant to
 * the target usage.
 *
 * For example, game engine Pragma SDKs need messages with PragmaSessionType PLAYER and PARTNER, but
 * should not ship with OPERATOR or SERVICE Session messages. Similarly, the Pragma Portal additionally needs
 * messages for OPERATOR endpoints.
 *
 * This step is necessary, because the protobuf compiler (protoc) only allows customization that _adds_ to
 * generated files, or entirely custom plugins. So if we want to use the default generation, but filter out some
 * messages, we use this maven plugin to do so. Additionally, it allows us a clean slate going into our custom
 * protoc plugins.
 */
@Mojo(name = "proto-filter",
      defaultPhase = LifecyclePhase.GENERATE_SOURCES,
      threadSafe = true)
class ProtoFilterMojo : AbstractMojo() {
    /**
     * Root path of the proto files. All protos in this directory (recursively) will be included.
     * Recommend the same as the -I (--proto-path) option passed to protoc.
     *
     * Multiple paths can be used and all protos will be merged into one folder structure.
     * Duplicate relative paths to their respective inputDirectories will cause errors!
     */
    @Parameter(property = "inputDirectories", required = true)
    var inputDirectories: Set<String> = setOf()

    /**
     * Absolute path where the output files will end up.
     */
    @Parameter(property = "outputDirectory", required = true)
    var outputDirectory: String = ""

    /**
     * Clean the outputDirectory before building (Defaults to true).
     */
    @Parameter(property = "outputDirectory")
    var cleanOutputDirectory: Boolean = true

    /**
     * Messages with this name are always included in SDK generation. This should only be used for core types.
     */
    @Parameter(property = "alwaysIncludedMessages")
    var alwaysIncludedMessages = setOf<String>()

    /**
     * Messages with these PragmaSessionTypes (see pragmaOptions.proto) will be included in the output.
     * If a file has no types "tagged" with these any of these options, the entire file will be filtered out.
     */
    @Parameter(property = "includedPragmaSessionTypes")
    var includedPragmaSessionTypes = setOf<String>()

    /**
     * These are files that are known to contain nothing valuable for the filtered output. This will remove the file and any import statements
     * in other files. An example is a file that only contains options used in the filtering process.
     *
     * Note that this is an explicit exclusion because we do want to keep errors about missing imports that _are_ intended to exist.
     */
    @Parameter(property = "excludedFiles")
    var excludedFiles = setOf<String>()

    var logger = LoggerFactory.getLogger(this.javaClass)!!

    @Throws(MojoFailureException::class)
    override fun execute() {
        if (cleanOutputDirectory && File(outputDirectory).exists()) {
            FileUtils.cleanDirectory(outputDirectory)
        }
        val protoFiles = collectAllProtoFiles()
        val protoMetadata = processProtoMetadata(protoFiles)
        printResults(protoMetadata)
    }

    private fun collectAllProtoFiles(): MutableList<DirectoryAndFile> {
        val protoFiles = mutableListOf<DirectoryAndFile>()
        for (inputDirectory in inputDirectories) {
            assert(File(inputDirectory).exists()) { "InputDir: $inputDirectory not found!" }
            logger.debug("Copying files from $inputDirectory")
            val newProtos = File(inputDirectory).walk().filter { file -> file.isFile && file.path.endsWith(".proto") }.toList()
            if (newProtos.isEmpty()) {
                throw MojoFailureException("No .proto files found at inputDirectory: $inputDirectory")
            }
            protoFiles.addAll(newProtos.map { file -> DirectoryAndFile(inputDirectory, file) })
        }
        return protoFiles
    }


    /**
     * Iterates over the file, building relevant metadata information for use in determining what needs to be filtered.
     */
    private fun processProtoMetadata(fileData: MutableList<DirectoryAndFile>): List<ParsedProtoFileMetadata> {
        //
        // This function walks through each file one character at a time. It builds ProtoMetadata by tracking state
        // related to its current position in the file and what it has seen before.
        //
        // It has a separate buffer that stores the current line so far that is used to run regex against for certain events.
        // Those events can trigger state changes such tracking if we're inside of a special token, or a protobuf type,
        // during which it build up tokens or look regex for pragma session types.
        //
        // It completely ignores anything inside comments.
        //
        val parsedProtoFileMetadata = readFileData(fileData)

        //
        // Now that we have all types accounted for, we can walk the metadata list and ensure that every type that should be included, is included.
        // This is necessary because embedded types (those without pragma_session_type), will not be explicitly tagged as needed export, but
        // will be part of a top level message with a session type (hence, 'embedded'). By that logic we can propagate inclusion to all
        // embedded types that require it.
        //
        flagForSdkInclusion(parsedProtoFileMetadata)

        return parsedProtoFileMetadata
    }

    private fun readFileData(fileData: MutableList<DirectoryAndFile>): MutableList<ParsedProtoFileMetadata> {
        // If empty, we include all types.
        val includeAll = includedPragmaSessionTypes.isEmpty()
        val parsedFiles = mutableListOf<ParsedProtoFileMetadata>()
        for (fileAndDir in fileData) {
            val file = fileAndDir.file
            val inputDirectory = fileAndDir.inputDirectory
            val parsedFile = ParsedProtoFileMetadata(file, inputDirectory)
            parsedFiles.add(parsedFile)
            file.bufferedReader().use { reader ->
                // Position/char tracking.
                var filePos = -1
                var lastChar: Char // Initialized in loop.
                var char = 0.toChar()
                val line = StringBuilder(200)

                // Comment tracking.
                var isInLineComment = false
                var isInBlockComment = false

                // Type tracking.
                var currentType: ProtoMetadata? = null
                var scopeDepth = -1
                var isExtendType = false

                // Token tracking.
                var currentTokenType = TokenType.None
                val token = StringBuilder(50)

                val finishToken = finishToken@{
                    when (currentTokenType) {
                        TokenType.Package -> parsedFile.protoPackage = token.toString()
                        TokenType.TypeName,
                        TokenType.FieldTypeName -> {
                            val tokenStr = token.toString()
                            if (tokenStr == "option") {
                                // Ignore these since they're not actual types.
                                return@finishToken
                            }
                            if (tokenStr == "repeated" || tokenStr == "optional" || tokenStr.matches("""map<.+,""".toRegex())) {
                                // Need to continue token parsing after the repeated flag, e.g.
                                //      repeated TypeWeWantToInclude asdf = 1;
                                // or it's a map, e.g.
                                //      map<int64, TypeWeWantToInclude> asdf = 1;
                                // Clear the token, but leave us in the 'parsing token' state.
                                token.clear()
                                return@finishToken
                            }
                            if (currentType != null) {
                                currentType!!.setToken(currentTokenType, token.toString())
                            }
                        }
                        TokenType.None -> { /* nothing */
                        }
                    }
                    currentTokenType = TokenType.None
                    token.clear()
                }

                while (true) {
                    filePos++
                    lastChar = char
                    val value = reader.read()
                    if (value < 0) {
                        // End of file.
                        break
                    }
                    char = value.toChar()

                    if (char == CHAR_NEW_LINE) {
                        isInLineComment = false
                        finishToken()
                        // Find file imports.
                        val importMatch = PROTO_IMPORT_REGEX.find(line)
                        if (importMatch != null && importMatch.groupValues.count() > 1) {
                            // Track proto file imports so that we can remove them if all messages are filtered out.
                            val importMetadata = ProtoMetadata(parsedFile.protoPackage)
                            // -1 because we don't want to include the newline.
                            importMetadata.startIndex = filePos - line.length - 1
                            importMetadata.endIndex = filePos - 1
                            importMetadata.shouldBeIncluded = true
                            importMetadata.importFilename = importMatch.groupValues[1]
                            parsedFile.metadata.add(importMetadata)
                        }
                        // Check for session type.
                        if (currentType != null && !currentType.shouldBeIncluded) {
                            currentType.shouldBeIncluded = doesLineContainTag(line.toString(), isExtendType)
                        }
                        line.clear()
                        continue
                    }

                    //
                    // Comment handling.
                    //
                    if (isInLineComment) {
                        continue
                    }
                    if (isInBlockComment) {
                        if (isBlockCommentEnd(lastChar, char)) {
                            isInBlockComment = false
                        }
                        continue
                    }
                    if (isLineComment(lastChar, char)) {
                        line.deleteCharAt(line.lastIndex)
                        isInLineComment = true
                        continue
                    }
                    if (isBlockCommentStart(lastChar, char)) {
                        line.deleteCharAt(line.lastIndex)
                        isInBlockComment = true
                        continue
                    }

                    // Track the uncommented line in a buffer for regex matching.
                    if (!IGNORED_CHARS.contains(char)) {
                        line.append(char)
                    }

                    //
                    // Find tokens.
                    //
                    if (currentTokenType != TokenType.None) {
                        if (isEndOfToken(char)) {
                            if (token.isNotEmpty()) {
                                finishToken()
                            }
                            // Still may need to parse this char, e.g. `MessageName{`
                        } else {
                            token.append(char)
                            continue
                        }
                    }

                    //
                    // Package token.
                    //
                    if (currentType == null && line.matches("""^\s*package""".toRegex())) {
                        currentTokenType = TokenType.Package
                        continue
                    }

                    //
                    // Field type token.
                    //
                    // Matches these examples (where x is the beginning of the token we want).
                    // type x
                    // <anything>{ x
                    // <anything>{ type name = 5; x
                    if (currentType != null && line.matches("""(^|.*\{|.*;)\s*[a-zA-Z]""".toRegex())) {
                        currentTokenType = TokenType.FieldTypeName
                        token.append(char)
                        continue
                    }

                    //
                    // Look for beginning of a supported type.
                    //
                    // Note that this regex will only match once per line since it's a full-match line the next char will break it,
                    // which is good because we don't want to count a single-line message twice anyway.
                    //
                    if (currentType == null && line.matches("""^\s*(message|enum|extend)""".toRegex())) {
                        currentType = ProtoMetadata(parsedFile.protoPackage)
                        currentType.startIndex = filePos - line.length
                        currentType.shouldBeIncluded = includeAll
                        isExtendType = line.matches("""^\s*extend""".toRegex())
                        // Type name immediately follows the new type definition.
                        currentTokenType = TokenType.TypeName
                        // Set to -1 so we know to wait for first open bracket.
                        scopeDepth = -1
                        continue
                    }

                    //
                    // Track scope depth of a type.
                    //
                    if (currentType != null) {
                        when (char) {
                            CHAR_OPEN_BRACKET -> {
                                if (scopeDepth < 0) {
                                    // -1 indicates we are within a type but have not found the first open bracket yet.
                                    scopeDepth = 1
                                } else {
                                    scopeDepth++
                                }
                                continue
                            }
                            CHAR_CLOSE_BRACKET -> {
                                if (scopeDepth > 0) {
                                    scopeDepth--
                                    if (scopeDepth == 0) {
                                        // Reached the end of the type. Closing bracket is included.
                                        currentType.endIndex = filePos
                                        currentType.shouldBeIncluded = currentType.shouldBeIncluded ||
                                            (alwaysIncludedMessages.isNotEmpty() && alwaysIncludedMessages.contains(currentType.typeName))
                                        // Need to check for tag on this line, in case we close the type on the same line as the tag.
                                        if (!currentType.shouldBeIncluded) {
                                            currentType.shouldBeIncluded = doesLineContainTag(line.toString(), isExtendType)
                                        }
                                        parsedFile.metadata.add(currentType)
                                        currentType = null
                                        isExtendType = false
                                        scopeDepth = -1
                                    }
                                }
                                continue
                            }
                        }
                    }
                } // char loop
            } // file reader use
        } // for each file
        return parsedFiles
    }

    private fun doesLineContainTag(line: String, isExtendType: Boolean): Boolean {
        val anyIncludedSessionTypePattern = "(${includedPragmaSessionTypes.joinToString("|")})"
        // Matches:
        // PragmaSessionType pragma_session_type = <number>;
        val optionDefinitionRegex =
            """PragmaSessionType\s+pragma_session_type\s*=\s*\d+;""".toRegex()
        // Matches one of:
        // option (<any namespace>pragma_session_type) = <one of includedPragmaSessionTypes>;
        val tagRegex =
            """option\s+\((.*pragma_session_type)\)\s*=\s*$anyIncludedSessionTypePattern\s*;""".toRegex()
        // Include the extension itself.
        val isOptionDefinition = isExtendType && optionDefinitionRegex.find(line) != null
        val isTaggedMessage = tagRegex.find(line) != null
        return (isOptionDefinition || isTaggedMessage)
    }

    private fun flagForSdkInclusion(parsedFiles: MutableList<ParsedProtoFileMetadata>) {
        val fullNameToMetadata =
            parsedFiles.flatMap { file -> file.metadata }.associateBy { it.fullName() }

        for (parsedFile in parsedFiles) {
            if (excludedFiles.contains(parsedFile.file.name)) {
                parsedFile.shouldBeIncluded = false
            }
            for (metadata in parsedFile.metadata) {
                if (!parsedFile.shouldBeIncluded) {
                    metadata.shouldBeIncluded = false
                }

                propagateInclusion(metadata, fullNameToMetadata)
            }
        }

        // Any files without included types should not be included.
        val excludedFileNames = mutableSetOf<String>()
        for (parsedFile in parsedFiles) {
            if (parsedFile.metadata.none { it.shouldBeIncluded && it.typeName.isNotEmpty() }) {
                parsedFile.shouldBeIncluded = false
            }
            if (!parsedFile.shouldBeIncluded) {
                excludedFileNames.add(parsedFile.protocPath())
            }
        }

        // Flag any imports of excluded files as excluded so we don't get protoc errors.
        for (parsedFile in parsedFiles) {
            for (metadata in parsedFile.metadata) {
                if (metadata.importFilename.isNotEmpty() && excludedFileNames.contains(metadata.importFilename)) {
                    metadata.shouldBeIncluded = false
                }
            }
        }
    }

    private fun propagateInclusion(metadata: ProtoMetadata, fullNameToMetadata: Map<String, ProtoMetadata>) {
        if (metadata.typeName == "") {
            return
        }
        if (!metadata.shouldBeIncluded) {
            return
        }
        for (fieldTypeName in metadata.fieldTypeNames) {
            // Field type name will be relative to the package we're currently in, or one of its parents.
            // e.g.
            // package: a.b.c
            // typename: x.Type could be any one of:
            // a.b.c.x.Type
            // a.b.x.Type
            // a.x.Type
            // x.Type
            //
            // It's important we start looking from the closest (by package) so that we don't see a higher up message
            // with the same name that is already included, and stop looking, when we really want to _also_ include the closest one.
            var fieldMetadata = fullNameToMetadata[fieldTypeName]
            if (fieldMetadata == null) {
                val splitPackage = metadata.protoPackage.split('.').toMutableList()
                val count = splitPackage.count()
                for (i in 0..count) {
                    val qualifiedFieldTypeName = "${splitPackage.joinToString(".")}.$fieldTypeName"
                    fieldMetadata = fullNameToMetadata[qualifiedFieldTypeName]
                    if (fieldMetadata != null) {
                        break
                    }

                    splitPackage.removeLastOrNull()
                }
            }
            if (fieldMetadata != null && !fieldMetadata.shouldBeIncluded) {
                fieldMetadata.shouldBeIncluded = true
                propagateInclusion(fieldMetadata, fullNameToMetadata)
            }
        }
    }

    private fun printResults(protoMetadata: List<ParsedProtoFileMetadata>) {
        val copyResults = mutableListOf<CopyResult>()
        for (parsedProtoFileMetadata in protoMetadata) {
            copyResults.add(tryCopyFiltered(parsedProtoFileMetadata.inputDirectory, parsedProtoFileMetadata))
        }

        //
        // Write out nice results info.
        //
        val resultStr = StringBuilder()
        resultStr.append("\n  INCLUDED FILES:\n")
        copyResults.sortBy { it.included }
        for (copyResult in copyResults.filter { it.success && it.included }) {
            resultStr.append("    ").append(copyResult.path).append("\n")
        }
        resultStr.append("\n  EXCLUDED FILES:\n")
        for (copyResult in copyResults.filter { it.success && !it.included }) {
            resultStr.append("    ").append(copyResult.path).append("\n")
        }
        val erroredResults = copyResults.filter { !it.success }
        if (erroredResults.isNotEmpty()) {
            for (copyResult in erroredResults) {
                resultStr.append("\n  ERRORED FILES:\n")
                resultStr.append("    ").append(copyResult.path).append("\n")
            }
        }
        if (copyResults.any { !it.success }) {
            logger.error(resultStr.toString())
            throw MojoFailureException("There was one or more errors copy/filtering files.")
        }
        logger.debug(resultStr.toString())
    }

    private data class CopyResult(val path: String, var success: Boolean, var included: Boolean)
    /**
     * Copies the file to the same relative directory in outputDirectory, and filters based on input.
     * If all messages are filtered out, the file will not be copied.
     *
     * @param inputDirectory The root input dir of this file.
     * @param fileMetadata File to copy with metadata.
     * @return CopyResult.
     */
    private fun tryCopyFiltered(inputDirectory: String, fileMetadata: ParsedProtoFileMetadata): CopyResult {
        val sourceFile = fileMetadata.file
        val inputPath = Paths.get(inputDirectory)
        val outputPath = Paths.get(outputDirectory)
        val relativePath = inputPath.relativize(sourceFile.toPath())
        val destinationFile = outputPath.resolve(relativePath)
        val result = CopyResult(path = relativePath.toString(), success = false, included = false)
        if (!fileMetadata.shouldBeIncluded) {
            result.success = true
            return result
        }

        val content: String
        content = try {
            sourceFile.readText()
        } catch (ex: IOException) {
            logger.error(" > Failed to read/filter file $sourceFile. IOException: $ex")
            return result
        }

        val filteredContent = StringBuilder(content.length)
        fileMetadata.metadata.sortBy { it.startIndex }
        var metadataIndex = 0
        for ((i, char) in content.withIndex()) {
            val metadata = fileMetadata.metadata.getOrNull(metadataIndex)
            if (metadata != null) {
                if (i >= metadata.endIndex) {
                    metadataIndex++
                }
                if (i >= metadata.startIndex && !metadata.shouldBeIncluded) {
                    // Only exclude if it's specifically marked to exclude. Many passages in the file won't have metadata.
                    continue
                }
            }
            filteredContent.append(char)
        }
        try {
            Files.createDirectories(destinationFile.parent)
            Files.writeString(destinationFile, filteredContent, StandardOpenOption.CREATE_NEW)
        } catch (ex: IOException) {
            logger.error(" > Failed to write filtered file to $destinationFile. IOException:$ex")
            return result
        }
        result.success = true
        result.included = true
        return result
    }
}
