package net.polyv.vod.v1.upload.service.impl;

import com.aliyun.oss.ClientBuilderConfiguration;
import com.obs.services.ObsClient;
import com.obs.services.ObsConfiguration;
import com.obs.services.exception.ObsException;
import com.obs.services.model.Callback;
import com.obs.services.model.CompleteMultipartUploadResult;
import com.obs.services.model.ObjectMetadata;
import com.obs.services.model.ProgressListener;
import com.obs.services.model.ProgressStatus;
import com.obs.services.model.UploadFileRequest;

import lombok.extern.slf4j.Slf4j;
import net.polyv.common.v1.constant.Constant;
import net.polyv.common.v1.util.FileUtil;
import net.polyv.vod.v1.config.UploadConfig;
import net.polyv.vod.v1.entity.upload.vo.VodUploadConfigResponse;
import net.polyv.vod.v1.entity.upload.vo.VodUploadOBSTokenResponse;
import net.polyv.vod.v1.entity.upload.vo.VodUploadOSSTokenResponse;
import net.polyv.vod.v1.entity.upload.vo.VodUploadVideoConfigRequest;
import net.polyv.vod.v1.upload.callback.UploadCallBack;
import net.polyv.vod.v1.upload.enumeration.UploadErrorMsg;
import net.polyv.vod.v1.upload.service.VodOSSService;
import net.polyv.vod.v1.upload.service.VodUploadVideoService;

/**
 * 华为上传服务
 * @author: sadboy
 **/
@Slf4j
public class HuaWeiOBSService implements VodOSSService {
    
    private static final long NOTICE_UPLOAD_SIZE = 1024 * 1024;
    
    /**
     * 调用初始化视频接口返回的信息
     */
    private VodUploadConfigResponse vodUploadVideoConfigResponse;
    
    /**
     * 用户设置的上传参数，分片大小，进度文件位置，上传线程数
     */
    private UploadConfig uploadConfig;
    
    /**
     * 用户设置的视频信息，视频文件，所属分类等
     */
    private VodUploadVideoConfigRequest vodUploadVideoConfigRequest;
    
    private HuaWeiOBSService() {
    }
    
    public HuaWeiOBSService(VodUploadConfigResponse vodUploadVideoConfigResponse, UploadConfig uploadConfig,
            VodUploadVideoConfigRequest vodUploadVideoConfigRequest) {
        this.vodUploadVideoConfigResponse = vodUploadVideoConfigResponse;
        this.uploadConfig = uploadConfig;
        this.vodUploadVideoConfigRequest = vodUploadVideoConfigRequest;
    }
    
    @Override
    public boolean upload(UploadCallBack callBack, boolean printProcessLog) {
        VodUploadOBSTokenResponse uploadToken = vodUploadVideoConfigResponse.getHwObsInfo();
        
        //桶名称、线程数、分片大小
        String checkpoint = uploadConfig.getCheckpoint();
        int taskNum = uploadConfig.getThreadNum();
        int partitionSize = uploadConfig.getPartitionSize();
        
        final String vId = vodUploadVideoConfigResponse.getVid();
        String objectName = uploadToken.getDir() + vId + "." +
                FileUtil.getExtension(vodUploadVideoConfigRequest.getFile().getPath());
        String fileLocation = vodUploadVideoConfigRequest.getFile().getPath();
        String checkpointFile = checkpoint + "/" + vId + ".ucp";
        
        ObsClient obsClient = buildObsClient(uploadToken);
        
        ObjectMetadata meta = new ObjectMetadata();
        meta.setContentType("text/plain");
        
        UploadFileRequest uploadFileRequest = new UploadFileRequest(uploadToken.getBucketName(), objectName);
        
        
        // 设置待上传的本地文件，localfile为待上传的本地文件路径，需要指定到具体的文件名
        uploadFileRequest.setUploadFile(fileLocation);
        // 设置分段上传时的最大并发数
        uploadFileRequest.setTaskNum(taskNum);
        // 设置分段大小
        uploadFileRequest.setPartSize(partitionSize);
        // 开启断点续传模式
        uploadFileRequest.setEnableCheckpoint(true);
        Callback callback = new Callback();
        callback.setCallbackUrl(uploadToken.getCallbackUrl());//必选项，支持设置多个url，以英文分号（;）分隔，最多支持10个。
        callback.setCallbackBody(uploadToken.getCallbackBody());//必选项，发起回调请求的body体，其中变量key、etag是系统变量。
        callback.setCallbackBodyType(
                uploadToken.getCallbackBodyType());//非必选项，发起回调请求的Content-Type头域的值。如果不设置，默认为application/x-www-form
        // -urlencoded。
        uploadFileRequest.setCallback(callback);
        
        uploadFileRequest.setCheckpointFile(checkpointFile);
        uploadFileRequest.setProgressInterval(NOTICE_UPLOAD_SIZE);
        uploadFileRequest.setEnableCheckSum(true);
        // 文件的元数据。
        uploadFileRequest.setObjectMetadata(meta);
        uploadFileRequest.setProgressListener(new ProgressListener() {
            private boolean start = false;
            private long totalFileSize = vodUploadVideoConfigRequest.getFileSize();
            
            @Override
            public void progressChanged(ProgressStatus status) {
                final String videoPoolId = vId;
                if (!start) {
                    start = true;
                    callBack.start(videoPoolId);
                    if (!printProcessLog) {
                        return;
                    }
                    log.info("【{}】vid={}, Start to upload......", vodUploadVideoConfigRequest.getTitle(), videoPoolId);
                    log.info("【{}】File size is {} bytes", videoPoolId, this.totalFileSize);
                } else {
                    callBack.process(videoPoolId, status.getTransferredBytes(), status.getTotalBytes());
                }
            }
        });
        
        // 断点续传上传。
        return triggerUpload(vId, obsClient, uploadFileRequest, callBack);
    }
    
    /**
     * 触发上传接口
     */
    private boolean triggerUpload(String videoPoolId, ObsClient obsClient, UploadFileRequest uploadFileRequest,
            UploadCallBack eventCallBack) {
        try {
            CompleteMultipartUploadResult uploadResult = obsClient.uploadFile(uploadFileRequest);
            if (uploadResult.getStatusCode() == 200) {
                eventCallBack.success(videoPoolId);
                return true;
            } else {
                log.error("huawei upload return error，error status code is：{}，requestId：{}", uploadResult.getStatusCode(), uploadResult.getRequestId());
                eventCallBack.error(videoPoolId, UploadErrorMsg.ERROR_UPLOAD_EXCEPTION);
                return false;
            }
        } catch (ObsException e) {
            String errorCode = e.getErrorCode();
            if (!("InvalidAccessKeyId".equals(errorCode) && !"SecurityTokenExpired".equals(errorCode))) {
                try {
                    //再验证一遍token问题
                    obsClient.listObjects(uploadFileRequest.getBucketName());
                } catch (ObsException exception) {
                    errorCode = exception.getErrorCode();
                    if (!("InvalidAccessKeyId".equals(errorCode) && !"SecurityTokenExpired".equals(errorCode))) {
                        eventCallBack.error(videoPoolId, UploadErrorMsg.ERROR_UPLOAD_TOKEN_EXPIRE);
                        return false;
                    }
                }
            }
            log.info("token is expired. reupload the video. requestId={}", e.getErrorRequestId());
            obsClient = refreshObsClient(obsClient);
            return triggerUpload(videoPoolId, obsClient, uploadFileRequest, eventCallBack);
        } catch (Throwable throwable) {
            log.error(throwable.getMessage(), throwable);
            eventCallBack.error(videoPoolId, UploadErrorMsg.ERROR_UPLOAD_EXCEPTION);
        }
        return false;
    }
    
    /**
     * 构建client
     * @return
     */
    private ObsClient buildObsClient(VodUploadOBSTokenResponse uploadToken) {
        uploadToken = uploadToken == null ? getUploadToken() : uploadToken;
        String ak = uploadToken.getAk();
        String sk = uploadToken.getSk();
        String st = uploadToken.getSecuritytoken();
        String endPoint = uploadToken.getEndpoint();
        
        ClientBuilderConfiguration ossConfig = new ClientBuilderConfiguration();
        ossConfig.setSupportCname(true);
        
        ObsConfiguration obsConfiguration = new ObsConfiguration();
        obsConfiguration.setCname(true);
        obsConfiguration.setEndPoint("https://" + endPoint);
        
        // 创建ObsClient实例。
        return new ObsClient(ak, sk, st, Constant.PROTOCOL_HTTPS+endPoint);
    }
    
    /**
     * 刷新ObsClient的token
     * @param obsClient
     * @return
     */
    private ObsClient refreshObsClient(ObsClient obsClient) {
        VodUploadOBSTokenResponse uploadToken = getUploadToken();
        String ak = uploadToken.getAk();
        String sk = uploadToken.getSk();
        String st = uploadToken.getSecuritytoken();
        obsClient.refresh(ak, sk, st);
        return obsClient;
    }
    
    /**
     * 获取上传视频的token
     * @return
     */
    private VodUploadOBSTokenResponse getUploadToken() {
        VodUploadOSSTokenResponse uploadToken = new VodUploadVideoService().getUploadToken(3);
        return uploadToken.getHwObsInfo();
    }
    
}
