package net.polyv.vod.v1.upload.service.impl;

import com.alibaba.fastjson.JSON;
import com.aliyun.oss.ClientBuilderConfiguration;
import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.OSSException;
import com.aliyun.oss.event.ProgressEvent;
import com.aliyun.oss.event.ProgressEventType;
import com.aliyun.oss.event.ProgressListener;
import com.aliyun.oss.model.Callback;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.UploadFileRequest;

import lombok.extern.slf4j.Slf4j;
import net.polyv.common.v1.util.FileUtil;
import net.polyv.vod.v1.config.UploadConfig;
import net.polyv.vod.v1.entity.upload.vo.VodUploadConfigResponse;
import net.polyv.vod.v1.entity.upload.vo.VodUploadOSSTokenResponse;
import net.polyv.vod.v1.entity.upload.vo.VodUploadVideoConfigRequest;
import net.polyv.vod.v1.upload.callback.UploadCallBack;
import net.polyv.vod.v1.upload.enumeration.UploadErrorMsg;
import net.polyv.vod.v1.upload.provide.PolyvCredentialProvider;
import net.polyv.vod.v1.upload.service.VodOSSService;
import net.polyv.vod.v1.upload.service.VodUploadVideoService;

/**
 * 阿里云OSS上传服务
 * @author: sadboy
 **/
@Slf4j
public class AliOSSService implements VodOSSService {
    
    /**
     * 调用初始化视频接口返回的信息
     */
    private VodUploadConfigResponse vodUploadVideoConfigResponse;
    
    /**
     * 用户设置的上传参数，分片大小，进度文件位置，上传线程数
     */
    private UploadConfig uploadConfig;
    
    /**
     * 用户设置的视频信息，视频文件，所属分类等
     */
    private VodUploadVideoConfigRequest vodUploadVideoConfigRequest;
    
    private AliOSSService() {
    }
    
    public AliOSSService(VodUploadConfigResponse vodUploadVideoConfigResponse, UploadConfig uploadConfig,
            VodUploadVideoConfigRequest vodUploadVideoConfigRequest) {
        this.uploadConfig = uploadConfig;
        this.vodUploadVideoConfigResponse = vodUploadVideoConfigResponse;
        this.vodUploadVideoConfigRequest = vodUploadVideoConfigRequest;
    }
    
    @Override
    public boolean upload(UploadCallBack eventCallBack, boolean printProcessLog) {
        VodUploadOSSTokenResponse uploadToken = getUploadToken();
        
        String bucketName = uploadToken.getBucketName();
        int taskNum = uploadConfig.getThreadNum();
        
        final String vId = vodUploadVideoConfigResponse.getVid();
        String objectName =
                uploadToken.getDir() + vId + "." + FileUtil.getExtension(vodUploadVideoConfigRequest.getFile().getPath());
        String fileLocation = vodUploadVideoConfigRequest.getFile().getPath();
        String checkpointFile = uploadConfig.getCheckpoint() + "/" + vId + ".ucp";
        
        
        OSS ossClient = buildOssClient(uploadToken);
        
        ObjectMetadata meta = new ObjectMetadata();
        meta.setContentType("text/plain");
        
        UploadFileRequest uploadFileRequest = new UploadFileRequest(bucketName, objectName);
        
        uploadFileRequest.setUploadFile(fileLocation);
        // 指定上传并发线程数，默认为1。
        uploadFileRequest.setTaskNum(taskNum);
        // 指定上传的分片大小，范围为100KB~5GB，默认为1M。
        uploadFileRequest.setPartSize(uploadConfig.getPartitionSize());
        // 开启断点续传，默认关闭。
        uploadFileRequest.setEnableCheckpoint(true);
        // 记录本地分片上传结果的文件。开启断点续传功能时需要设置此参数，上传过程中的进度信息会保存在该文件中，如果某一分片上传失败，再次上传时会根据文件中记录的点继续上传。上传完成后，该文件会被删除。默认与待上传的本地文件同目录，为uploadFile.ucp。
        uploadFileRequest.setCheckpointFile(checkpointFile);
        // 文件的元数据。
        uploadFileRequest.setObjectMetadata(meta);
        // 设置上传成功回调，参数为Callback类型。
        
        Callback callback = JSON.parseObject(vodUploadVideoConfigResponse.getCallback(), Callback.class);
        uploadFileRequest.setCallback(callback);
        uploadFileRequest.setProgressListener(new ProgressListener() {
            private long bytesWritten = 0;
            private long totalBytes = -1;
            private long totalFileSize = vodUploadVideoConfigRequest.getFileSize();
            
            @Override
            public void progressChanged(ProgressEvent progressEvent) {
                long bytes = progressEvent.getBytes();
                ProgressEventType eventType = progressEvent.getEventType();
                String videoPoolId = vId;
                switch (eventType) {
                    case TRANSFER_STARTED_EVENT:
                        eventCallBack.start(videoPoolId);
                        if (!printProcessLog) {
                            break;
                        }
                        log.info("【{}】vid={}, Start to upload......", vodUploadVideoConfigRequest.getTitle(), videoPoolId);
                        log.info("【{}】File size is {} bytes", videoPoolId, this.totalFileSize);
                        break;
                    case REQUEST_CONTENT_LENGTH_EVENT:
                        this.totalBytes = bytes;
                        if (!printProcessLog) {
                            break;
                        }
                        log.info("【{}】{} bytes in total will be uploaded to Server", videoPoolId, this.totalBytes);
                        break;
                    case REQUEST_BYTE_TRANSFER_EVENT:
                        this.bytesWritten += bytes;
                        eventCallBack.process(videoPoolId, totalFileSize - totalBytes + this.bytesWritten,
                                this.totalFileSize);
                        if (!printProcessLog) {
                            break;
                        }
                        if (this.totalBytes != -1) {
                            int percent = (int) ((totalFileSize - totalBytes + this.bytesWritten) * 100.0 /
                                    this.totalFileSize);
                            log.info("【{}】{} bytes have been written at this time, upload progress: {}%({}/{})",
                                    videoPoolId, bytes, percent, totalFileSize - totalBytes + this.bytesWritten,
                                    this.totalFileSize);
                        } else {
                            log.info("【{}】{} bytes have been written at this time, upload ratio: unknown({}/...)",
                                    videoPoolId, bytes, totalFileSize - totalBytes + this.bytesWritten);
                        }
                        break;
                    case TRANSFER_COMPLETED_EVENT:
                        eventCallBack.complete(videoPoolId);
                        if (!printProcessLog) {
                            break;
                        }
                        log.info("【{}】Succeed to upload, {} bytes have been transferred in total", videoPoolId,
                                totalFileSize - totalBytes + this.bytesWritten);
                        break;
                    case TRANSFER_FAILED_EVENT:
                        eventCallBack.error(videoPoolId, UploadErrorMsg.ERROR_UPLOAD_PART);
                        if (!printProcessLog) {
                            break;
                        }
                        log.info("【{}】Failed to upload, {} bytes have been transferred", videoPoolId,
                                totalFileSize - totalBytes + this.bytesWritten);
                        break;
                    default:
                        break;
                }
            }
        });
        
        // 断点续传上传。
        return triggerUpload(vId, ossClient, uploadFileRequest, eventCallBack);
    }
    
    /**
     * 触发上传接口
     */
    private boolean triggerUpload(String videoPoolId, OSS ossClient, UploadFileRequest uploadFileRequest,
            UploadCallBack eventCallBack) {
        try {
            ossClient.uploadFile(uploadFileRequest);
            // 关闭OSSClient。
            ossClient.shutdown();
            eventCallBack.success(videoPoolId);
            return true;
        } catch (OSSException e) {
            //假如是token过期，需要更新token再重传
            if (("InvalidAccessKeyId".equals(e.getErrorCode()) || "SecurityTokenExpired".equals(e.getErrorCode()))) {
                log.info("token is expired. reupload the video. requestId={}", e.getRequestId());
                ossClient = buildOssClient(null);
                return triggerUpload(videoPoolId, ossClient, uploadFileRequest, eventCallBack);
            }
            log.error(e.getMessage(), e);
            eventCallBack.error(videoPoolId, UploadErrorMsg.ERROR_UPLOAD_TOKEN_EXPIRE);
        } catch (Throwable throwable) {
            log.error(throwable.getMessage(), throwable);
            eventCallBack.error(videoPoolId, UploadErrorMsg.ERROR_UPLOAD_EXCEPTION);
        }
        // 关闭OSSClient。
        ossClient.shutdown();
        return false;
    }
    
    /**
     * 构建client
     * @return
     */
    private OSS buildOssClient(VodUploadOSSTokenResponse uploadToken) {
        uploadToken = uploadToken == null?getUploadToken():uploadToken;
        String accessKeyId = uploadToken.getAccessId();
        String accessKeySecret = uploadToken.getAccessKey();
        String securityToken = uploadToken.getToken();
        String domain = uploadToken.getDomain();
        long validityTime = uploadToken.getValidityTime() * 1000;
    
        ClientBuilderConfiguration ossConfig = new ClientBuilderConfiguration();
        ossConfig.setSupportCname(true);
    
        // 创建OSSClient实例。
        return new OSSClient(domain, new PolyvCredentialProvider(accessKeyId, accessKeySecret, securityToken,
                System.currentTimeMillis() + validityTime), ossConfig);
    }
    
    /**
     * 获取上传视频的token
     * @return
     */
    private VodUploadOSSTokenResponse getUploadToken(){
        return new VodUploadVideoService().getUploadToken(3);
    }
    
}
