package net.polyv.vod.v1.upload.client;

import java.util.List;

import lombok.extern.slf4j.Slf4j;
import net.polyv.common.v1.constant.Constant;
import net.polyv.common.v1.entity.CommonReqeust;
import net.polyv.common.v1.exception.PloyvSdkException;
import net.polyv.common.v1.util.SDKValidateUtil;
import net.polyv.common.v1.validator.ViolationMsg;
import net.polyv.vod.v1.config.UploadConfig;
import net.polyv.vod.v1.entity.upload.VodUploadVideoPartsRequest;
import net.polyv.vod.v1.entity.upload.VodUploadVideoRequest;
import net.polyv.vod.v1.entity.upload.vo.VodUploadConfigResponse;
import net.polyv.vod.v1.entity.upload.vo.VodUploadVideoConfigRequest;
import net.polyv.vod.v1.upload.service.VodOSSService;
import net.polyv.vod.v1.upload.service.VodUploadVideoService;
import net.polyv.vod.v1.upload.service.impl.AliOSSService;
import net.polyv.vod.v1.upload.service.impl.HuaWeiOBSService;

/**
 * 保利威分片上传客户端
 * @author: sadboy
 **/
@Slf4j
public class PolyvUploadClient {
    /**
     * 接口调用重试次数
     */
    public static final int RETRY_TIME = 3;
    
    /**
     * 上传分片大小
     */
    private UploadConfig uploadConfig;
    
    /**
     * 使用默认参数构建上传客户端
     */
    public PolyvUploadClient() {
        uploadConfig = new UploadConfig();
    }
    
    /**
     * 通过参数构建上传客户端
     * @param partitionSize 上传分片大小，范围100KB~5GB，100*1024代表100KB
     * @param checkpoint 分片上传进度文件存放位置（需要创建文件夹和文件权限），默认：checkpoint_location
     * @param threadNum 上传线程数，根据服务器实际情况设置，默认为5个
     */
    public PolyvUploadClient(int partitionSize, String checkpoint, int threadNum) {
        uploadConfig = new UploadConfig(partitionSize, checkpoint, threadNum);
    }
    
    
    public VodOSSService buildOSSService(CommonReqeust commonRequest) {
        VodUploadVideoConfigRequest vodUploadVideoConfigRequest = null;
        if (commonRequest instanceof VodUploadVideoRequest) {
            VodUploadVideoRequest vodUploadVideoRequest = (VodUploadVideoRequest) commonRequest;
            validateBean(vodUploadVideoRequest);
            vodUploadVideoConfigRequest = new VodUploadVideoConfigRequest(vodUploadVideoRequest);
        } else if (commonRequest instanceof VodUploadVideoPartsRequest) {
            VodUploadVideoPartsRequest vodUploadVideoPartsRequest = (VodUploadVideoPartsRequest) commonRequest;
            validateBean(vodUploadVideoPartsRequest);
            vodUploadVideoConfigRequest = new VodUploadVideoConfigRequest(vodUploadVideoPartsRequest);
        } else {
            throw new PloyvSdkException(Constant.ERROR_CODE, "点播上传视频对象异常");
        }
        VodUploadConfigResponse vodUploadConfigResponse = new VodUploadVideoService().initUploadQueue(
                vodUploadVideoConfigRequest, RETRY_TIME);
        //华为云配置走华为云OBS上传
        if (vodUploadConfigResponse.getHwObsInfo() != null) {
            return new HuaWeiOBSService(vodUploadConfigResponse, uploadConfig, vodUploadVideoConfigRequest);
        } else {
            return new AliOSSService(vodUploadConfigResponse, uploadConfig, vodUploadVideoConfigRequest);
        }
    }
    
    
    /**
     * 验证参数必填等是否通过
     * @param e
     * @param <E>
     */
    private <E extends CommonReqeust> void validateBean(E e) {
        List<ViolationMsg> violationMsgList = SDKValidateUtil.validateBean(e);
        if (!violationMsgList.isEmpty()) {
            String errors = SDKValidateUtil.getViolationMsgStr(violationMsgList);
            errors = errors.substring(0, errors.length() - 3);
            errors = "输入参数 [" + e.getClass().getName() + "]对象校验失败 ,失败字段 [" + errors + "]";
            throw new PloyvSdkException(Constant.ERROR_CODE, errors);
        }
    }
    
}
