package net.polyv.live.v1.service.interact.impl;

import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;

import com.alibaba.fastjson.JSON;

import lombok.extern.slf4j.Slf4j;
import net.polyv.live.v1.constant.LiveURL;
import net.polyv.live.v1.service.LiveBaseService;
import net.polyv.live.v1.service.interact.ILiveLotteryActivityService;
import net.polyv.live.v1.util.LiveSignUtil;
import net.polyv.live.v2.entity.channel.interact.LiveCreateLotteryActivityRequest;
import net.polyv.live.v2.entity.channel.interact.LiveCreateLotteryActivityResponse;
import net.polyv.live.v2.entity.channel.interact.LiveDeleteLotteryActivityRequest;
import net.polyv.live.v2.entity.channel.interact.LiveLotteryActivityListRequest;
import net.polyv.live.v2.entity.channel.interact.LiveLotteryActivityListPageResponse;
import net.polyv.live.v2.entity.channel.interact.LiveLotteryActivityRequest;
import net.polyv.live.v2.entity.channel.interact.LiveLotteryActivityResponse;
import net.polyv.live.v2.entity.channel.interact.LiveUpdateLotteryActivityRequest;

/**
 * 抽奖活动
 */
@Slf4j
public class LiveLotteryActivityServiceImpl extends LiveBaseService implements ILiveLotteryActivityService {
    
    /**
     * 查询抽奖活动
     * API地址：https://help.polyv.net/#/live/api/v4/channel/lottery_activity/lottery_activity_get
     * @param liveLotteryActivityRequest 查询抽奖活动请求实体
     * @return 查询抽奖活动返回实体
     * @throws IOException 异常
     * @throws NoSuchAlgorithmException 异常
     */
    @Override
    public LiveLotteryActivityResponse getLotteryActivity(LiveLotteryActivityRequest liveLotteryActivityRequest)
            throws IOException, NoSuchAlgorithmException {
        String url = LiveURL.LIVE_GET_LOTTERY_ACTIVITY_URL;
        return super.getReturnOne(url, liveLotteryActivityRequest, LiveLotteryActivityResponse.class);
    }
    
    /**
     * 创建抽奖活动
     * API地址：https://help.polyv.net/#/live/api/v4/channel/lottery_activity/lottery_activity_create
     * @param liveCreateLotteryActivityRequest 创建抽奖活动请求实体
     * @return 创建抽奖活动返回实体
     * @throws IOException
     * @throws NoSuchAlgorithmException
     */
    @Override
    public LiveCreateLotteryActivityResponse createLotteryActivity(
            LiveCreateLotteryActivityRequest liveCreateLotteryActivityRequest)
            throws IOException, NoSuchAlgorithmException {
        String url = LiveURL.LIVE_CREATE_LOTTERY_ACTIVITY_URL;
        Map<String, String> signMap = LiveSignUtil.getSignMap(liveCreateLotteryActivityRequest);
        signMap.put("channelId", liveCreateLotteryActivityRequest.getChannelId());
        return super.postJsonBodyReturnOne(url, signMap, liveCreateLotteryActivityRequest,
                JSON.toJSONString(liveCreateLotteryActivityRequest), LiveCreateLotteryActivityResponse.class);
    }
    
    /**
     * 更新抽奖活动
     * API地址：https://help.polyv.net/#/live/api/v4/channel/lottery_activity/lottery_activity_update
     * @param liveUpdateLotteryActivityRequest 更新抽奖活动请求实体
     * @return 更新抽奖活动返回实体
     * @throws IOException
     * @throws NoSuchAlgorithmException
     */
    @Override
    public Boolean updateLotteryActivity(LiveUpdateLotteryActivityRequest liveUpdateLotteryActivityRequest)
            throws IOException, NoSuchAlgorithmException {
        String url = LiveURL.LIVE_UPDATE_LOTTERY_ACTIVITY_URL;
        Map<String, String> signMap = LiveSignUtil.getSignMap(liveUpdateLotteryActivityRequest);
        signMap.put("channelId", liveUpdateLotteryActivityRequest.getChannelId());
        super.postJsonBodyReturnOne(url, signMap, liveUpdateLotteryActivityRequest,
                JSON.toJSONString(liveUpdateLotteryActivityRequest), Boolean.class);
        return true;
    }
    
    /**
     * 删除抽奖活动
     * API地址：https://help.polyv.net/#/live/api/v4/channel/lottery_activity/lottery_activity_delete
     * @param liveDeleteLotteryActivityRequest 删除抽奖活动请求实体
     * @return 删除抽奖活动返回实体
     * @throws IOException 异常
     * @throws NoSuchAlgorithmException 异常
     */
    @Override
    public Boolean deleteLotteryActivity(LiveDeleteLotteryActivityRequest liveDeleteLotteryActivityRequest)
            throws IOException, NoSuchAlgorithmException {
        String url = LiveURL.LIVE_DELETE_LOTTERY_ACTIVITY_URL;
        Map<String, String> signMap = LiveSignUtil.getSignMap(liveDeleteLotteryActivityRequest);
        signMap.put("channelId", liveDeleteLotteryActivityRequest.getChannelId());
        super.postJsonBodyReturnOne(url, signMap, liveDeleteLotteryActivityRequest,
                JSON.toJSONString(liveDeleteLotteryActivityRequest), Boolean.class);
        return true;
    }
    
    /**
     * 查询抽奖活动列表
     * API地址：https://help.polyv.net/#/live/api/v4/channel/lottery_activity/lottery_activity_list
     * @param liveLotteryActivityListRequest 查询抽奖活动列表请求实体
     * @return 查询抽奖活动列表返回实体
     * @throws IOException 异常
     * @throws NoSuchAlgorithmException 异常
     */
    @Override
    public LiveLotteryActivityListPageResponse listLotteryActivity(
            LiveLotteryActivityListRequest liveLotteryActivityListRequest)
            throws IOException, NoSuchAlgorithmException {
        String url = LiveURL.LIVE_LIST_LOTTERY_ACTIVITY_URL;
        return super.getReturnOne(url, liveLotteryActivityListRequest, LiveLotteryActivityListPageResponse.class);
    }
    
}
