package net.plsar;

import net.plsar.environments.Environments;
import net.plsar.schemes.RenderingScheme;
import net.plsar.resources.*;
import net.plsar.security.SecurityAttributes;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.ServerSocket;
import java.util.*;
import java.util.concurrent.*;
import java.util.logging.Logger;

public class PLSAR {

    static Logger Log = Logger.getLogger(PLSAR.class.getName());

    Integer port;
    String PROPERTIES;
    String RENDERING_SCHEME;
    Integer STARTUP_EXECUTORS;

    ViewConfig viewConfig;
    SchemaConfig schemaConfig;
    PropertiesConfig propertiesConfig;
    Integer numberOfPartitions = 3;
    Integer numberOfRequestExecutors = 7;
    PersistenceConfig persistenceConfig;
    Class<?> securityAccessKlass;
    List<Class<?>> viewRenderers;

    SecurityAttributes securityAttributes;

    public PLSAR(int port){
        this.port = port;
        this.STARTUP_EXECUTORS = 19;
        this.PROPERTIES = "system.properties";
        this.RENDERING_SCHEME = RenderingScheme.CACHE_REQUESTS;
        this.viewConfig = new ViewConfig();
        this.viewRenderers = new ArrayList<>();
    }

    public void start(){
        try {

            ServerResources serverResources = new ServerResources();

            if(schemaConfig != null &&
                    schemaConfig.getEnvironment().equals(Environments.DEVELOPMENT)) {
                DatabaseEnvironmentManager databaseEnvironmentManager = new DatabaseEnvironmentManager();
                databaseEnvironmentManager.configure(schemaConfig, persistenceConfig);
            }

            StartupAnnotationInspector startupAnnotationInspector = new StartupAnnotationInspector(new ComponentsHolder());
            startupAnnotationInspector.inspect();
            ComponentsHolder componentsHolder = startupAnnotationInspector.getComponentsHolder();

            if(propertiesConfig == null){
                propertiesConfig = new PropertiesConfig();
                propertiesConfig.setPropertiesFile(PROPERTIES);
            }

            String propertiesFile = propertiesConfig.getPropertiesFile();
            RouteAttributesResolver routeAttributesResolver = new RouteAttributesResolver(propertiesFile);
            RouteAttributes routeAttributes = routeAttributesResolver.resolve();
            AnnotationComponent serverStartup = componentsHolder.getServerStartup();

            String resourcesDirectory = viewConfig.getResourcesPath();
            ConcurrentMap<String, byte[]> viewBytesMap = serverResources.getViewBytesMap(viewConfig);

            Log.info("Running startup routine, please wait...");
            if(serverStartup != null) {
                Method startupMethod = serverStartup.getKlass().getMethod("startup");
                startupMethod.invoke(serverResources.getInstance(serverStartup.getKlass()));
            }

            Log.info("Registering network request negotiators, please wait...\n");
            ServerSocket serverSocket = new ServerSocket(port);
            serverSocket.setPerformancePreferences(0, 1, 2);
            ExecutorService executors = Executors.newFixedThreadPool(numberOfPartitions);
            executors.execute(new PartitionedRunnable(RENDERING_SCHEME, numberOfRequestExecutors, resourcesDirectory, securityAttributes, routeAttributes, viewBytesMap, serverSocket, persistenceConfig, viewRenderers, securityAccessKlass));;

            Log.info("Ready!");

        }catch(IOException | NoSuchMethodException | InvocationTargetException | IllegalAccessException | InstantiationException | PlsarException ex){
            ex.printStackTrace();
        }
    }

    public void setPropertiesConfig(PropertiesConfig propertiesConfig){
        this.propertiesConfig = propertiesConfig;
    }

    public void setPageRenderingScheme(String RENDERING_SCHEME) {
        this.RENDERING_SCHEME = RENDERING_SCHEME;
    }

    public void setViewConfig(ViewConfig viewConfig) {
        this.viewConfig = viewConfig;
    }

    public void setSecurityAccess(Class<?> securityAccessKlass) {
        this.securityAccessKlass = securityAccessKlass;
    }

    public void setSchemaConfig(SchemaConfig schemaConfig) {
        this.schemaConfig = schemaConfig;
    }

    public void setNumberOfPartitions(int numberOfPartitions){
        this.numberOfPartitions = numberOfPartitions;
    }

    public void setNumberOfRequestExecutors(int numberOfRequestExecutors){
        this.numberOfRequestExecutors = numberOfRequestExecutors;
    }

    public PLSAR setStartupExecutors(Integer STARTUP_EXECUTORS){
        this.STARTUP_EXECUTORS = STARTUP_EXECUTORS;
        return this;
    }

    public PLSAR addViewRenderer(Class<?> viewRenderer){
        this.viewRenderers.add(viewRenderer);
        return this;
    }

    public PLSAR setPersistenceConfig(PersistenceConfig persistenceConfig) {
        this.persistenceConfig = persistenceConfig;
        return this;
    }

}